package org.apache.torque.generator.configuration.controller;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * Allows reading and setting the current loglevel using log4j.
 * @version $Id: Log4jLoggingAdapter.java 1446086 2013-02-14 09:46:57Z tfischer $
 *
 */
public class Log4jLoggingAdapter implements LoggingAdapter
{
    /** The logger. */
    private static Log log = LogFactory.getLog(Log4jLoggingAdapter.class);

    /** Map loglevel -> corresponding log4j level. */
    private static final Map<Loglevel, Level> LOG4J_LEVELS;

    static
    {
        Map<Loglevel, Level> log4jLevels = new HashMap<Loglevel, Level>();
        log4jLevels.put(Loglevel.TRACE, Level.TRACE);
        log4jLevels.put(Loglevel.DEBUG, Level.DEBUG);
        log4jLevels.put(Loglevel.INFO, Level.INFO);
        log4jLevels.put(Loglevel.WARN, Level.WARN);
        log4jLevels.put(Loglevel.ERROR, Level.ERROR);
        LOG4J_LEVELS = Collections.unmodifiableMap(log4jLevels);
    }

    /**
     * Returns the current loglevel by reading the loglevel of the root logger.
     *
     * @return the current loglevel, or INFO if the current loglevel cannot
     *         be determined.
     */
    public Loglevel getCurrentLoglevel()
    {
        Level level = Logger.getRootLogger().getLevel();
        for (Map.Entry<Loglevel, Level> loglevel : LOG4J_LEVELS.entrySet())
        {
            if (loglevel.getValue().equals(level))
            {
                return loglevel.getKey();
            }
        }
        return Loglevel.INFO;
    }

    /**
     * Sets the loglevel to the given loglevel
     * by changing the level of the log4j root logger.
     *
     *  @param loglevel the loglevel to set, not null.
     */
    public void setLoglevel(Loglevel loglevel)
    {
        if (loglevel == null)
        {
            return;
        }
        Level log4jLevel = LOG4J_LEVELS.get(loglevel);
        if (Logger.getRootLogger().getLevel() != LOG4J_LEVELS.get(loglevel))
        {
            log.info("apply() : Setting loglevel to " + loglevel);
            Logger.getRootLogger().setLevel(log4jLevel);
        }
    }

}
