package org.apache.torque.util;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.torque.Column;
import org.apache.torque.map.TableMap;

/**
 * A class containing values for database columns.
 *
 * @version $Id: ColumnValues.java 1448414 2013-02-20 21:06:35Z tfischer $
 */
public class ColumnValues implements Map<Column, JdbcTypedValue>
{
    /** The column values, keyed by the column names. */
    private Map<Column, JdbcTypedValue> columnValues;

    /**
     * The name of the database handle to use for connection opening if needed,
     * or null to use the default database handle for the table.
     */
    private String dbName;

    /**
     * Constructor with no contained column values.
     *
     * @throws NullPointerException if table is null.
     */
    public ColumnValues()
    {
        this.columnValues = new HashMap<Column, JdbcTypedValue>();
    }

    /**
     * Constructor with no contained column values.
     *
     * @param dbName the name of the database handle to use for connection
     *        opening if needed, or null to use the default database handle
     *        for the table.
     *
     * @throws NullPointerException if table is null.
     */
    public ColumnValues(TableMap table, String dbName)
    {
        this();
        this.dbName = dbName;
    }

    /**
     * Constructor.
     *
     * @param columnValues the column values, or null.
     *
     * @throws NullPointerException if table is null.
     */
    public ColumnValues(
            Map<Column, JdbcTypedValue> columnValues)
    {
        if (columnValues == null)
        {
            this.columnValues = new HashMap<Column, JdbcTypedValue>();
        }
        else
        {
            this.columnValues = columnValues;
        }
    }

    /**
     * Constructor.
     *
     * @param columnValues the column values, or null.
     * @param dbName the name of the database handle to use for connection
     *        opening if needed, or null to use the default database handle
     *        for the table.
     *
     * @throws NullPointerException if table is null.
     */
    public ColumnValues(
            Map<Column, JdbcTypedValue> columnValues,
            TableMap table,
            String dbName)
    {
        this(columnValues);
        this.dbName = dbName;
    }

    /**
     * Returns the name of the database handle to use for connection
     * opening.
     *
     * @return the database name, or null to use the default database handle
     *         for the table.
     */
    public String getDbName()
    {
        return dbName;
    }

    public int size()
    {
        return columnValues.size();
    }

    public boolean isEmpty()
    {
        return columnValues.isEmpty();
    }

    public boolean containsKey(Object key)
    {
        return columnValues.containsKey(key);
    }

    public boolean containsValue(Object value)
    {
        return columnValues.containsValue(value);
    }

    public JdbcTypedValue get(Object key)
    {
        return columnValues.get(key);
    }

    public JdbcTypedValue put(Column key, JdbcTypedValue value)
    {
        return columnValues.put(key, value);
    }

    public JdbcTypedValue remove(Object key)
    {
        return columnValues.remove(key);
    }

    public void putAll(Map<? extends Column, ? extends JdbcTypedValue> t)
    {
        columnValues.putAll(t);
    }

    public void clear()
    {
        columnValues.clear();
    }

    public Set<Column> keySet()
    {
        return columnValues.keySet();
    }

    public Collection<JdbcTypedValue> values()
    {
        return columnValues.values();
    }

    public Set<java.util.Map.Entry<Column, JdbcTypedValue>> entrySet()
    {
        return columnValues.entrySet();
    }

    @Override
    public String toString()
    {
        return "ColumnValues [dbName=" + dbName
            + ", columnValues=" + columnValues + "]";
    }

}
