/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;
import java.io.DataInput;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.hints.ChecksumMismatchException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.utils.FBUtilities;
import org.json.simple.JSONValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDescriptor {
    private static final Logger logger = LoggerFactory.getLogger(HintsDescriptor.class);
    static final int VERSION_30 = 1;
    static final int CURRENT_VERSION = 1;
    static final String COMPRESSION = "compression";
    static final Pattern pattern = Pattern.compile("^[a-fA-F0-9]{8}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{12}\\-(\\d+)\\-(\\d+)\\.hints$");
    final UUID hostId;
    final int version;
    final long timestamp;
    final ImmutableMap<String, Object> parameters;
    final ParameterizedClass compressionConfig;

    HintsDescriptor(UUID hostId, int version, long timestamp, ImmutableMap<String, Object> parameters) {
        this.hostId = hostId;
        this.version = version;
        this.timestamp = timestamp;
        this.parameters = parameters;
        this.compressionConfig = HintsDescriptor.createCompressionConfig(parameters);
    }

    HintsDescriptor(UUID hostId, long timestamp, ImmutableMap<String, Object> parameters) {
        this(hostId, 1, timestamp, parameters);
    }

    HintsDescriptor(UUID hostId, long timestamp) {
        this(hostId, 1, timestamp, (ImmutableMap<String, Object>)ImmutableMap.of());
    }

    static ParameterizedClass createCompressionConfig(Map<String, Object> params) {
        if (params.containsKey(COMPRESSION)) {
            Map compressorConfig = (Map)params.get(COMPRESSION);
            return new ParameterizedClass((String)compressorConfig.get("class_name"), (Map)compressorConfig.get("parameters"));
        }
        return null;
    }

    String fileName() {
        return String.format("%s-%s-%s.hints", this.hostId, this.timestamp, this.version);
    }

    String checksumFileName() {
        return String.format("%s-%s-%s.crc32", this.hostId, this.timestamp, this.version);
    }

    int messagingVersion() {
        return HintsDescriptor.messagingVersion(this.version);
    }

    static int messagingVersion(int hintsVersion) {
        switch (hintsVersion) {
            case 1: {
                return MessagingService.FORCE_3_0_PROTOCOL_VERSION ? 10 : 11;
            }
        }
        throw new AssertionError();
    }

    static boolean isHintFileName(Path path) {
        return pattern.matcher(path.getFileName().toString()).matches();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static Optional<HintsDescriptor> readFromFileQuietly(Path path) {
        try (RandomAccessFile raf = new RandomAccessFile(path.toFile(), "r");){
            Optional<HintsDescriptor> optional = Optional.of(HintsDescriptor.deserialize(raf));
            return optional;
        }
        catch (ChecksumMismatchException e) {
            throw new FSReadError((Throwable)e, path.toFile());
        }
        catch (IOException e) {
            HintsDescriptor.handleDescriptorIOE(e, path);
            return Optional.empty();
        }
    }

    @VisibleForTesting
    static void handleDescriptorIOE(IOException e, Path path) {
        try {
            if (Files.size(path) > 0L) {
                String newFileName = path.getFileName().toString().replace(".hints", ".corrupt.hints");
                Path target = path.getParent().resolve(newFileName);
                logger.error("Failed to deserialize hints descriptor {} - saving file as {}", new Object[]{path.toString(), target, e});
                Files.move(path, target, new CopyOption[0]);
            } else {
                logger.warn("Found empty hints file {} on startup, removing", (Object)path.toString());
                Files.delete(path);
            }
        }
        catch (IOException ex) {
            logger.error("Error handling corrupt hints file {}", (Object)path.toString(), (Object)ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static HintsDescriptor readFromFile(Path path) {
        try (RandomAccessFile raf = new RandomAccessFile(path.toFile(), "r");){
            HintsDescriptor hintsDescriptor = HintsDescriptor.deserialize(raf);
            return hintsDescriptor;
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, path.toFile());
        }
    }

    public boolean isCompressed() {
        return this.compressionConfig != null;
    }

    public ICompressor createCompressor() {
        return this.isCompressed() ? CompressionParams.createCompressor(this.compressionConfig) : null;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("hostId", (Object)this.hostId).add("version", this.version).add("timestamp", this.timestamp).add("parameters", this.parameters).toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof HintsDescriptor)) {
            return false;
        }
        HintsDescriptor hd = (HintsDescriptor)o;
        return Objects.equal((Object)this.hostId, (Object)hd.hostId) && Objects.equal((Object)this.version, (Object)hd.version) && Objects.equal((Object)this.timestamp, (Object)hd.timestamp) && Objects.equal(this.parameters, hd.parameters);
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.hostId, this.version, this.timestamp, this.parameters});
    }

    void serialize(DataOutputPlus out) throws IOException {
        CRC32 crc = new CRC32();
        out.writeInt(this.version);
        FBUtilities.updateChecksumInt(crc, this.version);
        out.writeLong(this.timestamp);
        HintsDescriptor.updateChecksumLong(crc, this.timestamp);
        out.writeLong(this.hostId.getMostSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getMostSignificantBits());
        out.writeLong(this.hostId.getLeastSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getLeastSignificantBits());
        byte[] paramsBytes = JSONValue.toJSONString(this.parameters).getBytes(StandardCharsets.UTF_8);
        out.writeInt(paramsBytes.length);
        FBUtilities.updateChecksumInt(crc, paramsBytes.length);
        out.writeInt((int)crc.getValue());
        out.write(paramsBytes);
        crc.update(paramsBytes, 0, paramsBytes.length);
        out.writeInt((int)crc.getValue());
    }

    int serializedSize() {
        int size = TypeSizes.sizeof(this.version);
        size += TypeSizes.sizeof(this.timestamp);
        size += TypeSizes.sizeof(this.hostId.getMostSignificantBits());
        size += TypeSizes.sizeof(this.hostId.getLeastSignificantBits());
        byte[] paramsBytes = JSONValue.toJSONString(this.parameters).getBytes(StandardCharsets.UTF_8);
        size += TypeSizes.sizeof(paramsBytes.length);
        size += 4;
        size += paramsBytes.length;
        return size += 4;
    }

    static HintsDescriptor deserialize(DataInput in) throws IOException {
        CRC32 crc = new CRC32();
        int version = in.readInt();
        FBUtilities.updateChecksumInt(crc, version);
        long timestamp = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, timestamp);
        long msb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, msb);
        long lsb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, lsb);
        UUID hostId = new UUID(msb, lsb);
        int paramsLength = in.readInt();
        FBUtilities.updateChecksumInt(crc, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        byte[] paramsBytes = new byte[paramsLength];
        in.readFully(paramsBytes, 0, paramsLength);
        crc.update(paramsBytes, 0, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        return new HintsDescriptor(hostId, version, timestamp, HintsDescriptor.decodeJSONBytes(paramsBytes));
    }

    private static ImmutableMap<String, Object> decodeJSONBytes(byte[] bytes) {
        return ImmutableMap.copyOf((Map)((Map)JSONValue.parse((String)new String(bytes, StandardCharsets.UTF_8))));
    }

    private static void updateChecksumLong(CRC32 crc, long value) {
        FBUtilities.updateChecksumInt(crc, (int)(value & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(value >>> 32));
    }

    private static void validateCRC(int expected, int actual) throws IOException {
        if (expected != actual) {
            throw new ChecksumMismatchException("Hints Descriptor CRC Mismatch");
        }
    }
}

