## Domain Registry Interface, Login Security Extension Mapping for EPP
##
## Copyright (c) 2019 Patrick Mevzek <netdri@dotandco.com>. All rights reserved.
##
## This file is part of Net::DRI
##
## Net::DRI is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## See the LICENSE file that comes with this distribution for more details.
#########################################################################################

package Net::DRI::Protocol::EPP::Extensions::LoginSecurity;

use strict;
use warnings;
use feature 'state';

use Net::DRI::Exception;
use Net::DRI::Util;

####################################################################################################

my $trigger_password = '[LOGIN-SECURITY]';

sub register_commands
{
 my ($class, $version)=@_;
 state $rsession = { 'session' => { 'login' => [ \&login, \&parse ]} };

 return $rsession;
}

sub setup
{
 my ($class, $po, $version)=@_;
 state $ns = { 'loginSec' => 'urn:ietf:params:xml:ns:epp:loginSec-1.0' };
 $po->ns($ns);

 # Patch core login
 # (this is bad and should not be needed if we would be able to register_commands at specific time of run)
 my $core_login = $po->{commands_by_class}{'Net::DRI::Protocol::EPP::Core::Session'}{session}{login}[0];

 my $new_login = sub {
    my ($po, $login, $password, $rdata) = @_;

    my $ns = $po->ns();
    if (defined $ns && exists $ns->{loginSec} && grep { $_ eq $ns->{loginSec} } @{$po->default_parameters()->{server}->{extensions_selected}})
    {
     if ($password eq $trigger_password)
     {
      Net::DRI::Exception::usererr_invalid_parameters(sprintf('password can not be "%s"', $trigger_password));
     }
     if (length($password) > 16)
     {
      $rdata->{login_security} = { password => $password };
      $password = $trigger_password; # §3.2
     }
     if (Net::DRI::Util::has_key($rdata, 'client_newpassword') and length($rdata->{client_newpassword}) > 16)
     {
      $rdata->{login_security}->{client_newpassword} = $rdata->{client_newpassword};
      $rdata->{client_newpassword} = $trigger_password;
     }
    }
    return $core_login->($po, $login, $password, $rdata);
 };

 $po->{commands_by_class}{'Net::DRI::Protocol::EPP::Core::Session'}{session}{login} = [ $new_login, undef ];
 return;
}

sub implements { return 'https://tools.ietf.org/html/draft-ietf-regext-login-security-03'; }


sub login
{
 my ($po, $login, $password, $rdata)=@_;

 return unless Net::DRI::Util::has_key($rdata, 'login_security');

 my ($long_password, $long_new_password)=@{$rdata->{login_security}}{qw/password client_newpassword/};
 delete $rdata->{login_security};

 my $agent = {
    app => sprintf('Net::DRI/%s (%s)', $Net::DRI::VERSION, $po->nameversion()),
    tech => sprintf('perl/%vd XML::LibXML/%s', $^V, $XML::LibXML::VERSION),
    os => $^O,
 };

 if (Net::DRI::Util::has_key($rdata, 'client_agent') && ref $rdata->{'client_agent'} eq 'HASH')
 {
  foreach my $t (qw/app tech os/)
  {
   $agent->{$t} = $rdata->{client_agent}->{$t} if exists $rdata->{client_agent}->{$t};
  }
 }
 my @agent = map { [$_, $agent->{$_}] } grep { defined $agent->{$_} } qw/app tech os/;

 my @nodes;
 push @nodes, ['userAgent', @agent] if @agent;

 if (defined $long_password)
 {
  Net::DRI::Exception::usererr_invalid_parameters('password') unless Net::DRI::Util::xml_is_token($long_password, 6);
  push @nodes, ['pw', $long_password];
 }

 if (defined $long_new_password)
 {
  Net::DRI::Exception::usererr_invalid_parameters('client_newpassword') unless Net::DRI::Util::xml_is_token($long_new_password, 6);
  push @nodes, ['newPW', $long_new_password];
 }

 $po->message()->command_extension('loginSec', ['loginSec', @nodes]);

 return;
}

sub parse
{
 my ($po, $otype, $oaction, $oname, $rinfo)=@_;
 my $mes=$po->message();

 # This needs to be parsed for both success and error cases
 my $data = $mes->get_extension('loginSec','loginSecData');
 return unless defined $data;

 my %events = (warning => [], error => []);
 foreach my $el (Net::DRI::Util::xml_list_children($data))
 {
  my ($name, $node)=@$el;

  next unless $name eq 'event';

  my %event = (type => $node->getAttribute('type'),  # mandatory
               lang => 'en');  # optional, default value
  foreach my $attr (qw/name exDate value duration lang/)
  {
   next unless $node->hasAttribute($attr);
   $event{$attr eq 'name' ? 'subtype' : $attr} = $node->getAttribute($attr);
  }

  my $content = $node->textContent();
  $content =~ s/^\s+//;
  $content =~ s/\s+$//;
  $event{description} = $content if $content ne '';

  push @{$events{$node->getAttribute('level')}}, \%event;
 }

 $rinfo->{session}->{login_security} = \%events;

 return;
}

####################################################################################################
1;


__END__

=pod

=head1 NAME

Net::DRI::Protocol::EPP::Extensions::LoginSecurity - EPP Login Security Extension mapping (draft-ietf-regext-login-security-03) for Net::DRI

=head1 DESCRIPTION

Please see the README file for details.

=head1 SUPPORT

For now, support questions should be sent to:

E<lt>netdri@dotandco.comE<gt>

Please also see the SUPPORT file in the distribution.

=head1 SEE ALSO

E<lt>http://www.dotandco.com/services/software/Net-DRI/E<gt>

=head1 AUTHOR

Patrick Mevzek, E<lt>netdri@dotandco.comE<gt>

=head1 COPYRIGHT

Copyright (c) 2019 Patrick Mevzek <netdri@dotandco.com>.
All rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

See the LICENSE file that comes with this distribution for more details.

=cut

