/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "_xitk.h"
#include "font.h"
#include "skin.h"
#include "backend.h"

/** labelbutton **************************************************************/

#include "labelbutton.h"

typedef struct {
  xitk_widget_t           w;

  int                     bType;

  xitk_hv_t               img_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t       skin, temp_image;

  xitk_int_callback_t     callback;

  struct {
    int                   align, dy, offset, visible, stay, shortcut_pos;
  }                       lb;

  xitk_font_textent_t     l, s;
  int                     origin;

  xitk_short_string_t     label, shortcut_label, font, shortcut_font;
  uint32_t                color[XITK_IMG_STATE_SELECTED + 1];
} _lbutton_private_t;

size_t xitk_short_string_set (xitk_short_string_t *s, const char *v) {
  size_t n;

  if (!v)
    v = "";
  if (!strcmp (s->s, v))
    return (size_t)-1;

  if (s->s != s->buf) {
    free (s->s);
    s->s = s->buf;
  }
  /* TJ. trying to avoid a Coverity Scan false positive with v == "PlInputText":
   * >>> Overrunning buffer pointed to by "inp.skin_element_name" of 12 bytes
   * by passing it to a function which accesses it at byte offset 62. */
  if ((n = strlen (v) + 1) > sizeof (s->buf)) {
    char *d = malloc (n);
    if (d) {
      s->s = d;
    } else {
      n = sizeof (s->buf) - 1;
      s->buf[sizeof (s->buf) - 1] = 0;
    }
  }
  memcpy (s->s, v, n);
  return s->len = n - 1;
}

static int _labelbutton_paint (_lbutton_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
  widget_event_t _event;

  if (!event) {
    _event.x = XITK_HV_H (wp->w.pos);
    _event.y = XITK_HV_V (wp->w.pos);
    _event.width = XITK_HV_H (wp->w.size);
    _event.height = XITK_HV_V (wp->w.size);
    event = &_event;
    wp->w.shown_state = wp->w.state;
  }

#ifdef XITK_PAINT_DEBUG
  printf ("xitk.labelbutton.paint (%d, %d, %d, %d).\n", event->x, event->y, event->width, event->height);
#endif
  if ((wp->w.state & XITK_WIDGET_STATE_VISIBLE) && wp->skin.width) {
    xitk_img_state_t state = xitk_image_find_state (wp->skin.num_states - 1, wp->w.state);

    wp->temp_image.image = xitk_image_temp_get (wp->w.wl->xitk);
    /* background image */
    xitk_part_image_copy (wp->w.wl, &wp->skin, &wp->temp_image,
      XITK_HV_H (wp->img_state_list[1 + state]), XITK_HV_V (wp->img_state_list[1 + state]),
      XITK_HV_H (wp->img_state_list[0]), XITK_HV_V (wp->img_state_list[0]), 0, 0);
    /* text */
    do {
      xitk_font_t *fs;
      int label_right, xoff, roff, mode;
      int ysize = XITK_HV_V (wp->img_state_list[0]), shortcut_pos = wp->lb.shortcut_pos;
      unsigned int fg = 0;

      if (!wp->lb.visible)
        break;
      /* shortcut is only permited if alignement is set to left */
      mode = (wp->label.s[0] ? 2 : 0)
           + ((wp->lb.align == ALIGN_LEFT) && wp->shortcut_label.s[0] && (shortcut_pos >= 0) ? 1 : 0);
      if (!mode)
        break;
      /* Try to load font */
      fs = xitk_font_load_font (wp->w.wl->xitk, wp->font.s[0] ? wp->font.s : xitk_get_cfg_string (wp->w.wl->xitk, XITK_SYSTEM_FONT));
      if (!fs)
        XITK_DIE ("%s()@%d: xitk_font_load_font() failed. Exiting\n", __FUNCTION__, __LINE__);
      if (wp->l.width < 0)
        xitk_font_text_extent (fs, wp->label.s[0] ? wp->label.s : "Button", wp->label.s[0] ? wp->label.len : 6, &wp->l);
      /*  Some colors configurations */
      fg = wp->color[state > XITK_IMG_STATE_SELECTED ? XITK_IMG_STATE_SELECTED : state];
      if (fg & 0x80000000) {
        fg = xitk_get_cfg_num (wp->w.wl->xitk, (fg == XITK_NOSKIN_TEXT_INV)
           ? ((wp->w.state & XITK_WIDGET_STATE_ENABLE) ? XITK_WHITE_COLOR : XITK_DISABLED_WHITE_COLOR)
           : ((wp->w.state & XITK_WIDGET_STATE_ENABLE) ? XITK_BLACK_COLOR : XITK_DISABLED_BLACK_COLOR));
      } else {
        if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE))
          fg = xitk_disabled_color (fg);
        fg = xitk_color_db_get (wp->w.wl->xitk, fg);
      }
      /* text position */
      xoff = wp->origin = 0;
      {
        static const uint8_t mode_offs[XITK_IMG_STATE_LAST] = {
          [XITK_IMG_STATE_SELECTED] = 1,
          [XITK_IMG_STATE_SEL_FOCUS] = 1
        };
        roff = mode_offs[state];
      }
      if (!wp->lb.stay) {
        wp->origin = roff;
        xoff = (int)roff * -4;
      }
      roff = roff * 4 + 1;
      if (wp->bType == TAB_BUTTON) {
        static const uint8_t mode_offs[XITK_IMG_STATE_LAST] = {
          [XITK_IMG_STATE_NORMAL] = 3,
          [XITK_IMG_STATE_DISABLED_NORMAL] = 3,
          [XITK_IMG_STATE_DISABLED_SELECTED] = 3
        };
        /* The tab lift effect. */
        ysize -= 3;
        wp->origin = mode_offs[state];
        wp->lb.dy = 0;
      }
      wp->origin += ((ysize + wp->l.ascent - wp->l.descent) >> 1) + wp->lb.dy;
      /*  Put text in the right place */
      label_right = 0;
      if (mode & 2) {
        int xx = wp->lb.align == ALIGN_LEFT ? roff
               : wp->lb.align == ALIGN_RIGHT  ? (XITK_HV_H (wp->img_state_list[0]) - wp->l.width - roff)
               : ((XITK_HV_H (wp->img_state_list[0]) - wp->l.width - xoff) >> 1); /** << default = center */
        xitk_image_draw_string (wp->temp_image.image, fs, xx + wp->lb.offset, wp->origin, wp->label.s, wp->label.len, fg);
        label_right = xx + wp->lb.offset + wp->l.width;
      }
      /* shortcut label */
      if (mode & 1) {
        if (wp->shortcut_font.s[0]) {
          xitk_font_t *short_font = xitk_font_load_font (wp->w.wl->xitk, wp->shortcut_font.s);

          if (short_font) {
            xitk_font_unload_font (fs);
            fs = short_font;
          }
        }
        if (!shortcut_pos) {
          if (wp->s.width < 0)
            xitk_font_text_extent (fs, wp->shortcut_label.s, wp->shortcut_label.len, &wp->s);
          shortcut_pos = XITK_HV_H (wp->img_state_list[0]) - 5 - wp->s.width;
        }
        /* dont paint over the right edge when selected. */
        roff >>= 1;
        /* clear away overlapping main text, if any. */
        if (label_right > roff + shortcut_pos)
          xitk_part_image_copy (wp->w.wl, &wp->skin, &wp->temp_image,
            XITK_HV_H (wp->img_state_list[1 + state]) + XITK_HV_H (wp->img_state_list[0]) - wp->s.width - 10,
            XITK_HV_V (wp->img_state_list[1 + state]),
            wp->s.width + 10, XITK_HV_V (wp->img_state_list[0]),
            XITK_HV_H (wp->img_state_list[0]) - wp->s.width - 10, 0);
        xitk_image_draw_string (wp->temp_image.image, fs,
          roff + shortcut_pos, wp->origin, wp->shortcut_label.s, wp->shortcut_label.len, fg);
      }
      xitk_font_unload_font (fs);
    } while (0);
    
    xitk_part_image_draw (wp->w.wl, &wp->skin, &wp->temp_image,
      event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
      event->width, event->height,
      event->x, event->y);
  }
  return 1;
}

int xitk_labelbutton_change_label (xitk_widget_t *w, const char *newlabel) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)) {
    wp->l.width = -1;
    xitk_short_string_set (&wp->label, newlabel);
    if (wp->w.wl->flags & XITK_WL_EXPOSED)
      return _labelbutton_paint (wp, NULL);
  }
  return 0;
}

const char *xitk_labelbutton_get_label (xitk_widget_t *w) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)) ? wp->label.s : NULL;
}

int xitk_labelbutton_change_shortcut_label (xitk_widget_t *w, const char *newlabel, int pos, const char *newfont) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (wp
    && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)
    && (wp->w.type & (WIDGET_GROUP_MENU | WIDGET_GROUP_BROWSER))) {
    wp->s.width = -1;
    xitk_short_string_set (&wp->shortcut_label, newlabel);
    if (newfont)
      xitk_short_string_set (&wp->shortcut_font, !strcmp (newfont, wp->font.s) ? "" : newfont);
    if (wp->shortcut_label.s[0])
      wp->lb.shortcut_pos = (pos >= 0) ? pos : -1;
    if (wp->w.wl->flags & XITK_WL_EXPOSED)
      return _labelbutton_paint (wp, NULL);
  }
  return 0;
}

static void _labelbutton_new_skin (_lbutton_private_t *wp, xitk_skin_config_t *skonfig) {
  XITK_HV_INIT;

  if (wp->w.skin_element_name[0]) {
    const xitk_skin_element_info_t *info;

    info = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
    if (info) {
      wp->skin = info->pixmap_img;
      xitk_part_image_states (&wp->skin, wp->img_state_list, 3);
      wp->color[XITK_IMG_STATE_NORMAL] = info->label_color;
      wp->color[XITK_IMG_STATE_FOCUS] = info->label_color_focus;
      wp->color[XITK_IMG_STATE_SELECTED] = info->label_color_click;
      xitk_short_string_set (&wp->font, info->label_fontname);
      wp->lb.visible = info->label_printable;
      wp->lb.stay    = info->label_staticity;
      wp->lb.align   = info->label_alignment;
      XITK_HV_H (wp->w.pos) = info->x;
      XITK_HV_V (wp->w.pos) = info->y;
      wp->w.size.w = wp->img_state_list[0].w;
      wp->lb.dy = (info->label_y > 0) && (info->label_y < wp->skin.height)
                ? info->label_y - (wp->skin.height >> 1) : 0;
      xitk_widget_state_from_info (&wp->w, info);
      xitk_image_temp_size (wp->w.wl->xitk, XITK_HV_H (wp->img_state_list[0]), XITK_HV_V (wp->img_state_list[0]));
      wp->temp_image.x = 0;
      wp->temp_image.y = 0;
      wp->temp_image.width = XITK_HV_H (wp->img_state_list[0]);
      wp->temp_image.height = XITK_HV_V (wp->img_state_list[0]);
    } else {
      wp->skin.x        = 0;
      wp->skin.y        = 0;
      wp->skin.width    = 0;
      wp->skin.height   = 0;
      wp->skin.image    = NULL;
      wp->color[XITK_IMG_STATE_NORMAL]   = 0;
      wp->color[XITK_IMG_STATE_FOCUS]    = 0;
      wp->color[XITK_IMG_STATE_SELECTED] = 0;
      xitk_short_string_set (&wp->font, "");
      wp->lb.visible = 0;
      wp->lb.stay    = 0;
      wp->lb.align   = ALIGN_LEFT;
      wp->w.pos.w   = 0;
      wp->w.size.w  = 0;
      wp->lb.dy = 0;
      wp->w.state  &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
      wp->temp_image.x      = 0;
      wp->temp_image.y      = 0;
      wp->temp_image.width  = 0;
      wp->temp_image.height = 0;
    }
    wp->l.width = wp->s.width = -1;
  }
}

static int labelbutton_event (xitk_widget_t *w, const widget_event_t *event) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (!event || !wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_LABELBUTTON)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _labelbutton_paint (wp, event);
      break;
    case WIDGET_EVENT_KEY:
      {
        const char k[] = {
          ' ', 0,
          XITK_CTRL_KEY_PREFIX, XITK_KEY_RETURN,
          XITK_CTRL_KEY_PREFIX, XITK_KEY_RIGHT,
        };
        if (event->modifier & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5))
          break;
        if (!event->string)
          break;
        if (  !memcmp (event->string, k + 0, 2)
          ||  !memcmp (event->string, k + 2, 2)
          || (!memcmp (event->string, k + 4, 2) && (wp->w.type & WIDGET_GROUP_MENU)))
          goto _input;
      }
      break;
    case WIDGET_EVENT_CLICK:
      if (event->button != 1)
        break;
    _input:
      if (wp->w.state & XITK_WIDGET_STATE_FOCUS) {
        uint32_t fire = event->pressed ? 0 : ~0u;
        wp->w.state &= ~XITK_WIDGET_STATE_CLICK;
        wp->w.state |= ~fire & XITK_WIDGET_STATE_CLICK;
        fire = (wp->w.state ^ fire) & XITK_WIDGET_STATE_IMMEDIATE;
        if (fire && (wp->w.state & XITK_WIDGET_STATE_TOGGLE))
          wp->w.state ^= XITK_WIDGET_STATE_ON;
        _labelbutton_paint (wp, NULL);
        if (fire && wp->callback)
          wp->callback (&wp->w, wp->w.userdata, !!(wp->w.state & XITK_WIDGET_STATE_ON), event->modifier);
        return 1;
      }
      break;
    case WIDGET_EVENT_INSIDE:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        XITK_HV_INIT;
        return xitk_image_inside (wp->skin.image,
          event->x + wp->skin.x - XITK_HV_H (wp->w.pos),
          event->y + wp->skin.y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
      }
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      _labelbutton_new_skin (wp, event->skonfig);
      break;
    case WIDGET_EVENT_DESTROY:
      if (!wp->w.skin_element_name[0])
        xitk_image_free_image (&wp->skin.image);
      xitk_short_string_deinit (&wp->label);
      xitk_short_string_deinit (&wp->shortcut_label);
      xitk_short_string_deinit (&wp->font);
      xitk_short_string_deinit (&wp->shortcut_font);
      break;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == FOREGROUND_SKIN) ? wp->skin.image : NULL;
        return 1;
      }
      break;
    default: ;
  }
  return 0;
}

void xitk_labelbutton_set_alignment (xitk_widget_t *w, int align) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON))
    wp->lb.align = align;
}

int xitk_labelbutton_get_alignment (xitk_widget_t *w) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)) ? wp->lb.align : -1;
}

void xitk_labelbutton_set_label_offset (xitk_widget_t *w, int offset) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON))
    wp->lb.offset = offset;
}

#if 0
const char *xitk_labelbutton_get_shortcut_label (xitk_widget_t *w) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  if (wp
    && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)
    && (wp->w.type & (WIDGET_GROUP_MENU | WIDGET_GROUP_BROWSER)))
    return wp->shortcut_label.s;
  return NULL;
}

const char *xitk_labelbutton_get_fontname (xitk_widget_t *w) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)) ? wp->font.s : NULL;
}

int xitk_labelbutton_get_label_offset (xitk_widget_t *w) {
  _lbutton_private_t *wp;

  xitk_container (wp, w, w);
  return (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABELBUTTON)) ? wp->lb.offset : 0;
}
#endif

/*
 * Create the labeled button.
 * NOTE: menu calls this with b->skin_element_name == NULL, info->pixmap_name == NULL,
 * and info->pixmap_img to be used but not to be freed like a regular skin.
 */
xitk_widget_t *xitk_info_labelbutton_create (const xitk_labelbutton_widget_t *b, const xitk_skin_element_info_t *info) {
  _lbutton_private_t *wp;
  xitk_skin_element_info_t dinf;
  XITK_HV_INIT;

  wp = (_lbutton_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  if (!info) {
    memset (&dinf, 0, sizeof (dinf));
    dinf.label_alignment = b->align;
    info = &dinf;
  }

  wp->bType             = b->button_type;

  wp->callback          = b->callback;

  xitk_short_string_init (&wp->label);
  xitk_short_string_set (&wp->label, b->label);

  xitk_short_string_init (&wp->shortcut_label);

  xitk_short_string_init (&wp->shortcut_font);

  xitk_short_string_init (&wp->font);
  xitk_short_string_set (&wp->font, info->label_fontname);

  wp->lb.shortcut_pos = -1;
  wp->lb.visible      = info->label_printable;
  wp->lb.stay         = info->label_staticity;

  wp->skin              = info->pixmap_img;
  xitk_part_image_states (&wp->skin, wp->img_state_list, 3);
  if (((wp->skin.width <= 0) || (wp->skin.height <= 0)) && wp->skin.image) {
    wp->skin.width = wp->skin.image->width;
    wp->skin.height = wp->skin.image->height;
  }
  xitk_image_temp_size (wp->w.wl->xitk, XITK_HV_H (wp->img_state_list[0]), XITK_HV_V (wp->img_state_list[0]));
  wp->temp_image.x = 0;
  wp->temp_image.y = 0;
  wp->temp_image.width = XITK_HV_H (wp->img_state_list[0]);
  wp->temp_image.height = XITK_HV_V (wp->img_state_list[0]);

  wp->color[XITK_IMG_STATE_NORMAL] = info->label_color;
  wp->color[XITK_IMG_STATE_FOCUS] = info->label_color_focus;
  wp->color[XITK_IMG_STATE_SELECTED] = info->label_color_click;

  wp->lb.offset = 0;
  wp->lb.align  = info->label_alignment;
  wp->lb.dy     = (info->label_y > 0) && (info->label_y < wp->skin.height)
                ? info->label_y - (wp->skin.height >> 1) : 0;

  /* modified xitk_widget_state_from_info (&wp->w, info).
   * for some reason, labelbutton wants to be always visible.
   * why not after skin change? */
  wp->w.state |= XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE;
  if (!info->enability)
    wp->w.state &= ~XITK_WIDGET_STATE_ENABLE;
  wp->w.saved_state = wp->w.state;

  wp->w.state          |= (wp->bType == RADIO_BUTTON) || (wp->bType== TAB_BUTTON)
                        ? XITK_WIDGET_STATE_TOGGLE : 0;
  XITK_HV_H (wp->w.pos) = info->x;
  XITK_HV_V (wp->w.pos) = info->y;
  wp->w.size.w = wp->img_state_list[0].w;
  wp->w.type            = WIDGET_TYPE_LABELBUTTON | WIDGET_FOCUSABLE | WIDGET_TABABLE
                        | WIDGET_CLICKABLE | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event           = labelbutton_event;

  wp->l.width = wp->s.width = -1;

  return _xitk_new_widget_apply (&b->nw, &wp->w);
}

xitk_widget_t *xitk_labelbutton_create (const xitk_labelbutton_widget_t *b, xitk_skin_config_t *skonfig) {
  return b ? xitk_info_labelbutton_create (b, xitk_skin_get_info (skonfig, b->nw.skin_element_name)) : NULL;
}

xitk_widget_t *xitk_noskin_labelbutton_create (const xitk_labelbutton_widget_t *b,
  int x, int y, int width, int height,
  uint32_t ncolor, uint32_t fcolor, uint32_t ccolor, const char *fname) {
  _lbutton_private_t *wp;
  XITK_HV_INIT;

  wp = (_lbutton_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->bType             = b->button_type;

  wp->callback          = b->callback;

  xitk_short_string_init (&wp->label);
  xitk_short_string_set (&wp->label, b->label);

  xitk_short_string_init (&wp->shortcut_label);

  xitk_short_string_init (&wp->shortcut_font);

  xitk_short_string_init (&wp->font);
  xitk_short_string_set (&wp->font, fname);

  wp->lb.shortcut_pos = -1;
  wp->lb.visible      = 1;
  wp->lb.stay         = 0;

  if (b->button_type == TAB_BUTTON) {
    if (xitk_shared_image (wp->w.wl, "xitk_tabbutton", width * 4, height, &wp->skin.image) == 1) {
      wp->skin.image->last_state = XITK_IMG_STATE_SEL_FOCUS;
      wp->skin.width = width * 4;
      wp->skin.height = height;
      wp->skin.num_states = 4;
      xitk_image_draw_tab (&wp->skin);
    }
  } else if (b->nw.skin_element_name && !strcmp (b->nw.skin_element_name, "XITK_NOSKIN_FLAT")) {
    if (xitk_shared_image (wp->w.wl, "xitk_labelbutton_f", width * 4, height, &wp->skin.image) == 1) {
      wp->skin.image->last_state = XITK_IMG_STATE_SEL_FOCUS;
      xitk_image_draw_bevel_style (wp->skin.image, XITK_DRAW_FLATTER);
    }
  } else {
    char name[24] = "xitk_labelbutton_b_";
    name[19] = '0' + ((b->style >> 16) & 7);
    name[20] = 0;
    if (xitk_shared_image (wp->w.wl, name, width * 4, height, &wp->skin.image) == 1) {
      wp->skin.image->last_state = XITK_IMG_STATE_SEL_FOCUS;
      xitk_image_draw_bevel_style (wp->skin.image, XITK_DRAW_BEVEL | b->style);
    }
  }
  wp->skin.x     = 0;
  wp->skin.y     = 0;
  wp->skin.width = 0;
  wp->skin.height = 0;
  if (wp->skin.image) {
    wp->skin.width = wp->skin.image->width;
    wp->skin.height = wp->skin.image->height;
  }
  xitk_part_image_states (&wp->skin, wp->img_state_list, 4);

  xitk_image_temp_size (wp->w.wl->xitk, XITK_HV_H (wp->img_state_list[0]), XITK_HV_V (wp->img_state_list[0]));
  wp->temp_image.x = 0;
  wp->temp_image.y = 0;
  wp->temp_image.width = XITK_HV_H (wp->img_state_list[0]);
  wp->temp_image.height = XITK_HV_V (wp->img_state_list[0]);

  wp->w.skin_element_name[0] = 0;
  wp->color[XITK_IMG_STATE_NORMAL] = ncolor;
  wp->color[XITK_IMG_STATE_FOCUS] = fcolor;
  wp->color[XITK_IMG_STATE_SELECTED] = ccolor;

  wp->lb.offset = 0;
  wp->lb.align  = b->align;
  wp->lb.dy     = 0;
  wp->l.width   =
  wp->s.width   = -1;

  wp->w.state          &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.state          |= (wp->bType == RADIO_BUTTON) || (wp->bType == TAB_BUTTON)
                        ? XITK_WIDGET_STATE_TOGGLE : 0;
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  wp->w.size.w = wp->img_state_list[0].w;
  wp->w.type            = WIDGET_TYPE_LABELBUTTON | WIDGET_FOCUSABLE | WIDGET_TABABLE
                        | WIDGET_CLICKABLE | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.event           = labelbutton_event;

  return _xitk_new_widget_apply (&b->nw, &wp->w);
}

/** button *******************************************************************/

#include "button.h"

typedef struct {
  xitk_widget_t       w;

  xitk_hv_t           img_state_list[XITK_IMG_STATE_LIST_SIZE];
  xitk_part_image_t   skin;

  /* callback function (active_widget, user_data) */
  xitk_simple_callback_t  callback;
  xitk_int_callback_t state_callback;
} _button_private_t;

static void _button_paint (_button_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
#ifdef XITK_PAINT_DEBUG
  printf ("xitk.button.paint (%d, %d, %d, %d).\n", event->x, event->y, event->width, event->height);
#endif
  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    xitk_img_state_t state = xitk_image_find_state (wp->skin.num_states - 1, wp->w.state);

    xitk_part_image_draw (wp->w.wl, &wp->skin, NULL,
      (int)XITK_HV_H (wp->img_state_list[1 + state]) + event->x - XITK_HV_H (wp->w.pos),
      (int)XITK_HV_V (wp->img_state_list[1 + state]) + event->y - XITK_HV_V (wp->w.pos),
      event->width, event->height,
      event->x, event->y);
  }
}

static void _button_read_skin (_button_private_t *wp, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *s = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
  XITK_HV_INIT;

  xitk_widget_state_from_info (&wp->w, s);
  if (s) {
    wp->skin = s->pixmap_img;
    xitk_part_image_states (&wp->skin, wp->img_state_list, 3);
    XITK_HV_H (wp->w.pos) = s->x;
    XITK_HV_V (wp->w.pos) = s->y;
  } else {
    wp->skin.x      = 0;
    wp->skin.y      = 0;
    wp->skin.width  = 0;
    wp->skin.height = 0;
    wp->skin.image  = NULL;
    wp->w.pos.w     = 0;
  }
}

static int button_event (xitk_widget_t *w, const widget_event_t *event) {
  _button_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp || ! event)
    return 0;
  if (((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_BUTTON)
    && ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_CHECKBOX))
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _button_paint (wp, event);
      break;
    case WIDGET_EVENT_KEY:
      {
        const char k[] = {
          ' ', 0,
          XITK_CTRL_KEY_PREFIX, XITK_KEY_RETURN,
        };
        if (event->modifier & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5))
          break;
        if (!event->string)
          break;
        if ( !memcmp (event->string, k + 0, 2)
          || !memcmp (event->string, k + 2, 2))
          goto _input;
      }
      break;
    case WIDGET_EVENT_CLICK:
      if (event->button != 1)
        break;
    _input:
      if (wp->w.state & XITK_WIDGET_STATE_FOCUS) {
        XITK_HV_INIT;
        widget_event_t ev2;
        uint32_t fire = event->pressed ? 0 : ~0u;
        wp->w.state &= ~XITK_WIDGET_STATE_CLICK;
        wp->w.state |= ~fire & XITK_WIDGET_STATE_CLICK;
        fire = (wp->w.state ^ fire) & XITK_WIDGET_STATE_IMMEDIATE;
        if (fire && (wp->w.state & XITK_WIDGET_STATE_TOGGLE))
          wp->w.state ^= XITK_WIDGET_STATE_ON;
        ev2.x = XITK_HV_H (wp->w.pos);
        ev2.y = XITK_HV_V (wp->w.pos);
        ev2.width = XITK_HV_H (wp->w.size);
        ev2.height = XITK_HV_V (wp->w.size);
        _button_paint (wp, &ev2);
        wp->w.shown_state = wp->w.state;
        if (fire) {
          if (wp->state_callback)
            wp->state_callback (&wp->w, wp->w.userdata, !!(wp->w.state & XITK_WIDGET_STATE_ON), event->modifier);
          else if (wp->callback)
            wp->callback (&wp->w, wp->w.userdata);
        }
        return 1;
      }
      break;
    case WIDGET_EVENT_INSIDE:
      if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
        XITK_HV_INIT;
        return xitk_image_inside (wp->skin.image,
          event->x + wp->skin.x - XITK_HV_H (wp->w.pos),
          event->y + wp->skin.y - XITK_HV_V (wp->w.pos)) ? 1 : 2;
      }
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      if (wp->w.skin_element_name[0]) {
        _button_read_skin (wp, event->skonfig);
        wp->w.size.w = wp->img_state_list[0].w;
      }
      break;
    case WIDGET_EVENT_DESTROY:
      if (!wp->w.skin_element_name[0])
        xitk_image_free_image (&wp->skin.image);
      break;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == FOREGROUND_SKIN) ? wp->skin.image : NULL;
        return 1;
      }
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static xitk_widget_t *_xitk_button_create (_button_private_t *wp, const xitk_button_widget_t *b) {
  wp->callback          = b->callback;
  wp->state_callback    = b->state_callback;

  wp->w.size.w          = wp->img_state_list[0].w;
  wp->w.type            = (wp->state_callback ? WIDGET_TYPE_CHECKBOX : WIDGET_TYPE_BUTTON)
                        | WIDGET_CLICKABLE | WIDGET_FOCUSABLE | WIDGET_TABABLE
                        | WIDGET_KEYABLE | WIDGET_PARTIAL_PAINTABLE;
  wp->w.state          |= wp->state_callback ? XITK_WIDGET_STATE_TOGGLE : 0;
  wp->w.event           = button_event;

  return _xitk_new_widget_apply (&b->nw, &wp->w);
}

/*
 *
 */
xitk_widget_t *xitk_button_create (const xitk_button_widget_t *b, xitk_skin_config_t *skonfig) {
  _button_private_t *wp;

  wp = (_button_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _button_read_skin (wp, skonfig);

  return _xitk_button_create (wp, b);
}

/*
 *
 */
xitk_widget_t *xitk_noskin_button_create (const xitk_button_widget_t *b, int x, int y, int width, int height) {
  XITK_HV_INIT;
  _button_private_t *wp;
  xitk_button_widget_t _b;
  xitk_image_t *i;
  int states;

  wp = (_button_private_t *)xitk_widget_new (&b->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  _b = *b;
  _b.nw.skin_element_name = NULL;

  if (_b.symbol == XITK_SYMBOL_USER) {
    i = xitk_image_new (wp->w.wl->xitk, NULL, 0, width * 3, height);
    xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
    states = 3;
  } else {
    char name[16] = "xitk_button_00";

    name[12] = '0' + _b.symbol;
    name[13] = '0' + ((_b.style >> 16) & 7);
    states = 6;
    if (xitk_shared_image (wp->w.wl, name, width * 6, height, &i) == 1) {
      i->last_state = XITK_IMG_STATE_DISABLED_SELECTED;
      switch (_b.symbol) {
        case XITK_SYMBOL_LEFT:
        case XITK_SYMBOL_RIGHT:
        case XITK_SYMBOL_UP:
        case XITK_SYMBOL_DOWN:
        case XITK_SYMBOL_PLUS:
        case XITK_SYMBOL_MINUS:
          xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
          xitk_image_draw_symbol (i, _b.symbol);
          break;
        case XITK_SYMBOL_CHECK:
          xitk_image_draw_checkbox_check (i);
          break;
        /* case XITK_SYMBOL_NONE: */
        default:
          xitk_image_draw_bevel_style (i, XITK_DRAW_BEVEL | _b.style);
      }
    }
  }

  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;

  wp->w.skin_element_name[0] = 0;
  wp->skin.image = i;
  wp->skin.x        = 0;
  wp->skin.y        = 0;
  wp->skin.width    = i->width;
  wp->skin.height   = i->height;
  xitk_part_image_states (&wp->skin, wp->img_state_list, states);
  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);

  return _xitk_button_create (wp, &_b);
}
