#!/usr/bin/env bash
# shellcheck disable=SC1090
# shellcheck disable=SC1091
# shellcheck disable=SC2164

: "${GPG_VERSION:=stable}"
: "${BUILD_SHARED_LIBS:=off}"
: "${USE_STATIC_DEPENDENCIES:=}"
: "${OS:=}"
: "${DIST:=}"
: "${DIST_VERSION:=}"
: "${DIST_VERSION_ID:=}"

: "${MINIMUM_CMAKE_VERSION:=3.20.0}"
: "${MINIMUM_RUBY_VERSION:=2.5.0}"

: "${RECOMMENDED_BOTAN_VERSION:=2.18.2}"
: "${RECOMMENDED_JSONC_VERSION:=0.12.1}"
: "${RECOMMENDED_CMAKE_VERSION:=3.20.5}"
: "${RECOMMENDED_PYTHON_VERSION:=3.9.2}"
: "${RECOMMENDED_RUBY_VERSION:=2.5.8}"
# Bundler version to use if Ruby version is less then
# FALLBACK_BUNDLER_RUBY_VERSION
: "${FALLBACK_BUNDLER_VERSION:=2.3.26}"
: "${FALLBACK_BUNDLER_RUBY_VERSION:=2.6.0}"

: "${RECOMMENDED_BOTAN_VERSION_MSYS:=${RECOMMENDED_BOTAN_VERSION}-1}"

: "${CMAKE_VERSION:=${RECOMMENDED_CMAKE_VERSION}}"
# if [[ "${OS}" = msys ]]; then
#   : "${BOTAN_VERSION:=${RECOMMENDED_BOTAN_VERSION_MSYS}}"
# else
#   : "${BOTAN_VERSION:=${RECOMMENDED_BOTAN_VERSION}}"
# fi
: "${BOTAN_VERSION:=${RECOMMENDED_BOTAN_VERSION}}"
: "${JSONC_VERSION:=${RECOMMENDED_JSONC_VERSION}}"
: "${PYTHON_VERSION:=${RECOMMENDED_PYTHON_VERSION}}"
: "${RUBY_VERSION:=${RECOMMENDED_RUBY_VERSION}}"



if [[ "${GPG_VERSION}" = 2.3.* || "${GPG_VERSION}" = beta ]]; then
  : "${MINIMUM_AUTOMAKE_VERSION:=1.16.3}"
else
  : "${MINIMUM_AUTOMAKE_VERSION:=1.16.1}"
fi
: "${RECOMMENDED_AUTOMAKE_VERSION:=1.16.4}"

: "${AUTOMAKE_VERSION:=${RECOMMENDED_AUTOMAKE_VERSION}}"

: "${VERBOSE:=1}"


if [[ "${OS}" = "freebsd" ]] || \
   [[ "${DIST}" = "ubuntu" ]] || \
   [[ "${DIST}" = "centos" ]] || \
   [[ "${DIST}" = "fedora" ]]
then
  SUDO="${SUDO:-sudo}"
else
  SUDO="${SUDO:-run}"
fi

# Simply run its arguments.
run() {
  "$@"
}

. ci/lib/cacheable_install_functions.inc.sh

macos_install() {
  brew update-reset
  # homebrew fails because `openssl` is a symlink while it tries to remove a directory.
  rm /usr/local/Cellar/openssl || true
  # homebrew fails to update python 3.9.1 to 3.9.1.1 due to unlinking failure
  rm /usr/local/bin/2to3 || true
  # homebrew fails to update python from 3.9 to 3.10 due to another unlinking failure
  rm /usr/local/bin/idle3 || true
  rm /usr/local/bin/pydoc3 || true
  rm /usr/local/bin/python3 || true
  rm /usr/local/bin/python3-config || true
  # homebrew fails to update python from 3.11.0 to 3.11.1
  rm /usr/local/bin/2to3-3.11 || true
  rm /usr/local/bin/idle3.11 || true
  rm /usr/local/bin/pydoc3.11 || true
  rm /usr/local/bin/python3.11 || true
  rm /usr/local/bin/python3.11-config || true
  # homebrew fails to update openssl@1.1 1.1.1l to 1.1.1l_1 due to linking failure of nghttp2.h
  brew unlink nghttp2 || true
  brew update
  brew bundle
  ensure_automake
}

freebsd_install() {
  local packages=(
    git
    readline
    bash
    gnupg
    devel/pkgconf
    wget
    cmake
    gmake
    autoconf
    automake
    libtool
    gettext-tools
    python
    lang/ruby27
  )

  # Note: we assume sudo is already installed
  "${SUDO}" pkg install -y "${packages[@]}"

  cd /usr/ports/devel/ruby-gems
  "${SUDO}" make -DBATCH RUBY_VER=2.7 install
  cd

  mkdir -p ~/.gnupg
  echo "disable-ipv6" >> ~/.gnupg/dirmngr.conf
  dirmngr </dev/null
  dirmngr --daemon
  ensure_automake
}

openbsd_install() {
  echo ""
}

netbsd_install() {
  echo ""
}

linux_prepare_ribose_yum_repo() {
  "${SUDO}" rpm --import https://github.com/riboseinc/yum/raw/master/ribose-packages.pub
  "${SUDO}" rpm --import https://github.com/riboseinc/yum/raw/master/ribose-packages-next.pub
  "${SUDO}" curl -L https://github.com/riboseinc/yum/raw/master/ribose.repo \
    -o /etc/yum.repos.d/ribose.repo
}

# Prepare the system by updating and optionally installing repos
yum_prepare_repos() {
  yum_install "${util_depedencies_yum[@]}"
  linux_prepare_ribose_yum_repo
  "${SUDO}" "${YUM}" -y update
  if [[ $# -gt 0 ]]; then
    yum_install "$@"
  fi
}

linux_install_fedora() {
  yum_prepare_repos
  yum_install_build_dependencies \
    cmake
  yum_install_dynamic_build_dependencies_if_needed

  ensure_automake
  ensure_cmake
  ensure_ruby
  rubygem_install_build_dependencies
}

linux_install_centos() {
  case "${DIST_VERSION}" in
    centos-7)
      linux_install_centos7
      ;;
    centos-8)
      linux_install_centos8
      ;;
    centos-9)
      linux_install_centos9
      ;;
    *)
      >&2 echo "Error: unsupported CentOS version \"${DIST_VERSION_ID}\" (supported: 7, 8, 9).  Aborting."
      exit 1
  esac
}

declare util_depedencies_yum=(
  sudo # NOTE: Needed to avoid "sudo: command not found"
  wget
  git
)

declare basic_build_dependencies_yum=(
  # cmake3 # XXX: Fedora 22+ only has cmake
  clang
  gcc
  gcc-c++
  make
  autoconf
  automake
  libtool
  bzip2
  gzip
)

declare build_dependencies_yum=(
  bison
  byacc
  bzip2-devel
  gettext-devel
  ncurses-devel
  python3
  ribose-automake116
  ruby-devel
  zlib-devel
)

declare dynamic_build_dependencies_yum=(
  botan2
  botan2-devel
  json-c12-devel
  # python2-devel # TODO: needed?
)


apt_install() {
  local apt_command=(apt-get -y -q install "$@")
  if command -v sudo >/dev/null; then
    sudo "${apt_command[@]}"
  else
    "${apt_command[@]}"
  fi
}

yum_install() {
  local yum_command=("${YUM}" -y -q install "$@")
  if command -v sudo >/dev/null; then
    sudo "${yum_command[@]}"
  else
    "${yum_command[@]}"
  fi
}

prepare_build_tool_env() {
  case "${DIST}" in
    centos)
      post_build_tool_install_set_env
      ;;
  esac

  prepare_rbenv_env
}

yum_install_build_dependencies() {
  yum_install \
    "${basic_build_dependencies_yum[@]}" \
    "${build_dependencies_yum[@]}" \
    "$@"
}

linux_install_centos7() {
  yum_prepare_repos epel-release centos-release-scl
  yum_install_build_dependencies \
    cmake3 \
    rh-ruby25 rh-ruby25-ruby-devel \
    llvm-toolset-7.0

  yum_install_dynamic_build_dependencies_if_needed

  ensure_automake
  ensure_cmake
  ensure_ruby
  rubygem_install_build_dependencies
}

linux_install_centos8() {
  "${SUDO}" "${YUM}" -y -q install 'dnf-command(config-manager)'
  "${SUDO}" "${YUM}" config-manager --set-enabled powertools
  yum_prepare_repos epel-release
  yum_install_build_dependencies \
    cmake \
    texinfo

  yum_install_dynamic_build_dependencies_if_needed

  ensure_automake
  ensure_cmake
  ensure_ruby
  rubygem_install_build_dependencies
}

linux_install_centos9() {
  yum_prepare_repos epel-release
  yum_install_build_dependencies \
    cmake \
    texinfo

  yum_install_dynamic_build_dependencies_if_needed

  ensure_automake
  ensure_cmake
  ensure_ruby
  rubygem_install_build_dependencies
}

is_use_static_dependencies() {
  [[ -n "${USE_STATIC_DEPENDENCIES}" ]] && \
    [[ no    != "${USE_STATIC_DEPENDENCIES}" ]] && \
    [[ off   != "${USE_STATIC_DEPENDENCIES}" ]] && \
    [[ false != "${USE_STATIC_DEPENDENCIES}" ]] && \
    [[ 0     != "${USE_STATIC_DEPENDENCIES}" ]]
}

yum_install_dynamic_build_dependencies_if_needed() {
  if ! is_use_static_dependencies; then
    yum_install_dynamic_build_dependencies
  fi
}

install_static_noncacheable_build_dependencies_if_needed() {
  if is_use_static_dependencies; then
    install_static_noncacheable_build_dependencies "$@"
  fi
}

install_static_cacheable_build_dependencies_if_needed() {
  if is_use_static_dependencies || [[ "$#" -gt 0 ]]; then
    USE_STATIC_DEPENDENCIES=true
    install_static_cacheable_build_dependencies "$@"
  fi
}

install_static_cacheable_build_dependencies() {
  prepare_build_tool_env

  mkdir -p "$LOCAL_BUILDS"

  local default=(jsonc gpg)
  if [ "${CRYPTO_BACKEND:-}" != "openssl" ]; then
    default=(botan "${default[@]}")
  fi
  local items=("${@:-${default[@]}}")
  for item in "${items[@]}"; do
    install_"$item"
  done
}

install_static_noncacheable_build_dependencies() {
  mkdir -p "$LOCAL_BUILDS"

  local default=(bundler asciidoctor)
  local items=("${@:-${default[@]}}")
  for item in "${items[@]}"; do
    install_"$item"
  done
}

rubygem_install_build_dependencies() {
  install_bundler
  install_asciidoctor
}

yum_install_dynamic_build_dependencies() {
  yum_install \
    "${dynamic_build_dependencies_yum[@]}"

  # Work around pkg-config giving out wrong include path for json-c:
  ensure_symlink_to_target /usr/include/json-c12 /usr/include/json-c
}

# Make sure cmake is at least 3.14+ as required by rnp
# Also make sure ctest is available.
# If not, build cmake from source.
ensure_cmake() {
  ensure_symlink_to_target /usr/bin/cmake3 /usr/bin/cmake
  ensure_symlink_to_target /usr/bin/cpack3 /usr/bin/cpack

  local cmake_version
  cmake_version=$({
    command -v cmake  >/dev/null && command cmake --version || \
    command -v cmake3 >/dev/null && command cmake3 --version
    } | head -n1 | cut -f3 -d' '
  )

  local need_to_build_cmake=

  # Make sure ctest is also in PATH.  If not, build cmake from source.
  # TODO: Check CentOS7 tests in GHA.
  if ! command -v ctest >/dev/null; then
    >&2 echo "ctest not found."
    need_to_build_cmake=1
  elif ! is_version_at_least cmake "${MINIMUM_CMAKE_VERSION}" echo "${cmake_version}"; then
    >&2 echo "cmake version lower than ${MINIMUM_CMAKE_VERSION}."
    need_to_build_cmake=1
  fi

  if [[ "${need_to_build_cmake}" != 1 ]]; then
    CMAKE="$(command -v cmake)"
    >&2 echo "cmake rebuild is NOT needed."
    return
  fi

  >&2 echo "cmake rebuild is needed."

  pushd "$(mktemp -d)" || return 1

  install_prebuilt_cmake Linux-x86_64

  command -v cmake

  popd

  # Abort if ctest still not found.
  if ! command -v ctest >/dev/null; then
    >&2 echo "Error: ctest not found.  Aborting."
    exit 1
  fi
}

# E.g. for i386
# NOTE: Make sure cmake's build prerequisites are installed.
build_and_install_cmake() {
  local cmake_build=${LOCAL_BUILDS}/cmake
  mkdir -p "${cmake_build}"
  pushd "${cmake_build}"
  wget https://github.com/Kitware/CMake/releases/download/v"${CMAKE_VERSION}"/cmake-"${CMAKE_VERSION}".tar.gz -O cmake.tar.gz
  tar xzf cmake.tar.gz --strip 1

  PREFIX="${PREFIX:-/usr}"
  mkdir -p "${PREFIX}"
  ./configure --prefix="${PREFIX}" && ${MAKE} -j"${MAKE_PARALLEL}" && "${SUDO}" make install
  popd
  CMAKE="${PREFIX}"/bin/cmake
}

# 'arch' corresponds to the last segment of GitHub release URL
install_prebuilt_cmake() {
  local arch="${1:?Missing architecture}"
  local cmake_build=${LOCAL_BUILDS}/cmake
  mkdir -p "${cmake_build}"
  pushd "${cmake_build}"
  curl -L -o \
    cmake.sh \
    https://github.com/Kitware/CMake/releases/download/v"${CMAKE_VERSION}"/cmake-"${CMAKE_VERSION}"-"${arch}".sh

  PREFIX="${PREFIX:-/usr}"
  mkdir -p "${PREFIX}"
  "${SUDO}" sh cmake.sh --skip-license --prefix="${PREFIX}"
  popd
  CMAKE="${PREFIX}"/bin/cmake
}

build_and_install_python() {
  python_build=${LOCAL_BUILDS}/python
  mkdir -p "${python_build}"
  pushd "${python_build}"
  curl -L -o python.tar.xz https://www.python.org/ftp/python/"${PYTHON_VERSION}"/Python-"${PYTHON_VERSION}".tar.xz
  tar -xf python.tar.xz --strip 1
  ./configure --enable-optimizations --prefix=/usr && ${MAKE} -j"${MAKE_PARALLEL}" && "${SUDO}" make install
  ${SUDO} ln -sf /usr/bin/python3 /usr/bin/python
  popd
}

# Make sure automake is at least 1.16.3+ as required by GnuPG 2.3.
# If not, build automake from source.
ensure_automake() {

  local automake_version
  automake_version=$({
    command -v automake >/dev/null && command automake --version
    } | head -n1 | cut -f4 -d' '
  )

  local need_to_build_automake=

  if ! is_version_at_least automake "${MINIMUM_AUTOMAKE_VERSION}" echo "${automake_version}"; then
    >&2 echo "automake version lower than ${MINIMUM_AUTOMAKE_VERSION}."
    need_to_build_automake=1
  fi

  if [[ "${need_to_build_automake}" != 1 ]]; then
    >&2 echo "automake rebuild is NOT needed."
    return
  fi

  >&2 echo "automake rebuild is needed."

  pushd "$(mktemp -d)" || return 1

  build_and_install_automake

  # Disable automake116 from Ribose's repository as that may be too old.
  case "${DIST}" in
    centos)
      if [[ -r /opt/ribose/ribose-automake116/disable ]]; then
        >&2 echo "ribose-automake116 will be disabled."
        . /opt/ribose/ribose-automake116/disable
      fi

      if rpm --quiet -q ribose-automake116; then
        >&2 echo "ribose-automake116 is installed.  Removing."
        # "${SUDO}" "${YUM}" remove -y ribose-automake116
        "${SUDO}" rpm -e ribose-automake116
      fi
      ;;
  esac

  command -v automake

  popd
}

build_and_install_automake() {
  # automake
  automake_build=${LOCAL_BUILDS}/automake
  mkdir -p "${automake_build}"
  pushd "${automake_build}"
  curl -L -o automake.tar.xz "https://ftp.gnu.org/gnu/automake/automake-${AUTOMAKE_VERSION}.tar.xz"
  tar -xf automake.tar.xz --strip 1
  ./configure --enable-optimizations --prefix=/usr && ${MAKE} -j"${MAKE_PARALLEL}" && ${SUDO} make install
  popd
}

# json-c is installed with install_jsonc
# asciidoctor is installed with install_asciidoctor
linux_install_ubuntu() {
  "${SUDO}" apt-get update
  apt_install \
    "${util_dependencies_ubuntu[@]}" \
    "${basic_build_dependencies_ubuntu[@]}" \
    "${build_dependencies_ubuntu[@]}" \
    "$@"

  ubuntu_install_dynamic_build_dependencies_if_needed
  ensure_automake
}

ubuntu_install_dynamic_build_dependencies_if_needed() {
  if ! is_use_static_dependencies; then
    ubuntu_install_dynamic_build_dependencies
  fi
}

ubuntu_install_dynamic_build_dependencies() {
  apt_install \
    "${dynamic_build_dependencies_ubuntu[@]}"
}

declare util_dependencies_ubuntu=()

declare util_dependencies_deb=(
  sudo
  wget
  git
  software-properties-common
  # botan # Debian 9 does not have botan in default repos?
)

declare basic_build_dependencies_ubuntu=(
  build-essential
  cmake
)

declare basic_build_dependencies_deb=(
  autoconf
  automake
  build-essential
  curl
  libtool
)

declare build_dependencies_ubuntu=(
  gettext
  libbz2-dev
  libncurses-dev
  python3
  python3-venv
  ruby-dev
  zlib1g-dev
)

declare dynamic_build_dependencies_ubuntu=(
  botan
  libbotan-2-dev
)

declare build_dependencies_deb=(
  # botan # Debian 9 does not have botan in default repos?
  gettext
  libbz2-dev
  libncurses5-dev
  libssl-dev
  python3
  python3-venv
  ruby-dev
  zlib1g-dev
)

declare ruby_build_dependencies_ubuntu=(
  bison
  curl
  libbz2-dev
  libssl-dev
  ruby-bundler
  rubygems
  zlib1g-dev
)

declare ruby_build_dependencies_deb=(
  bison
  curl
  libbz2-dev
  libssl-dev
  ruby-bundler
  rubygems
  zlib1g-dev
)

linux_install_debian() {
  "${SUDO}" apt-get update
  apt_install \
    "${util_dependencies_deb[@]}" \
    "${basic_build_dependencies_deb[@]}" \
    "${build_dependencies_deb[@]}" \
    "$@"

  if [ "${CC-gcc}" = "clang" ]; then
# Add apt.llvm.org repository and install clang
# We may use https://packages.debian.org/stretch/clang-3.8 as well but this package gets installed to
# /usr/lib/clang... and requires update-alternatives which would be very ugly considering CC/CXX environment
# settings coming from yaml already
    wget -O - https://apt.llvm.org/llvm-snapshot.gpg.key|sudo apt-key add -
    ${SUDO} apt-add-repository "deb http://apt.llvm.org/stretch/ llvm-toolchain-stretch main"
    ${SUDO} apt-get install -y clang
  fi

  ensure_automake
  build_and_install_cmake
}

linux_install() {
  if type "linux_install_${DIST}" | grep -qwi 'function'; then
    "linux_install_${DIST}"
  fi
}

msys_install() {
  local packages=(
    tar
    git
    automake
    autoconf
    libtool
    automake-wrapper
    gnupg2
    make
    pkg-config
    p7zip
    mingw64/mingw-w64-x86_64-cmake
    mingw64/mingw-w64-x86_64-python3
  )

  if [ "${CC}" = "gcc" ]; then
    packages+=(mingw64/mingw-w64-x86_64-gcc
               mingw64/mingw-w64-x86_64-libbotan
               mingw64/mingw-w64-x86_64-json-c
    )
  else
    packages+=(clang64/mingw-w64-clang-x86_64-clang
               clang64/mingw-w64-clang-x86_64-openmp
               clang64/mingw-w64-clang-x86_64-libc++
               clang64/mingw-w64-clang-x86_64-libbotan
               clang64/mingw-w64-clang-x86_64-libssp
               clang64/mingw-w64-clang-x86_64-json-c
               clang64/mingw-w64-clang-x86_64-libsystre
    )
  fi

  pacman --noconfirm -S --needed "${packages[@]}"

  # msys includes ruby 2.6.1 while we need lower version
  #wget http://repo.msys2.org/mingw/x86_64/mingw-w64-x86_64-ruby-2.5.3-1-any.pkg.tar.xz -O /tmp/ruby-2.5.3.pkg.tar.xz
  #pacman --noconfirm --needed -U /tmp/ruby-2.5.3.pkg.tar.xz
  #rm /tmp/ruby-2.5.3.pkg.tar.xz
}

# Mainly for all python scripts with shebangs pointing to
# 'python', which is
# unavailable in CentOS 8 by default.
#
# This creates an environment where straight 'python' is available.
prepare_python_virtualenv() {
  python3 -m venv ~/.venv
}

# Run its arguments inside a python-virtualenv-enabled sub-shell.
run_in_python_venv() {
  if [[ ! -e ~/.venv ]] || [[ ! -f ~/.venv/bin/activate ]]; then
    prepare_python_virtualenv
  fi

  (
    # Avoid issues like '_OLD_VIRTUAL_PATH: unbound variable'
    set +u
    . ~/.venv/bin/activate
    set -u
    "$@"
  )
}

# ruby-rnp
install_bundler() {
  if is_version_at_least ruby "${FALLBACK_BUNDLER_RUBY_VERSION}" command ruby -e 'puts RUBY_VERSION'; then
    gem_install bundler bundle
  else
    gem_install "bundler:${FALLBACK_BUNDLER_VERSION}" bundle
  fi
}

install_asciidoctor() {
  gem_install asciidoctor
}

declare ruby_build_dependencies_yum=(
  git-core
  zlib
  zlib-devel
  gcc-c++
  patch
  readline
  readline-devel
  libyaml-devel
  libffi-devel
  openssl-devel
  make
  bzip2
  autoconf
  automake
  libtool
  bison
  curl
  sqlite-devel
  which # for rbenv-doctor
)

ensure_ruby() {
  if is_version_at_least ruby "${MINIMUM_RUBY_VERSION}" command ruby -e 'puts RUBY_VERSION'; then
    return
  fi

  # XXX: Fedora20 seems to have problems installing ruby build dependencies in
  # yum?
  # "${YUM}" repolist all
  # "${SUDO}" rpm -qa | sort

  if [[ "${DIST_VERSION}" = fedora-20 ]]; then
    ruby_build_dependencies_yum+=(--enablerepo=updates-testing)
  fi

  case "${DIST}" in
    centos|fedora)
      yum_install "${ruby_build_dependencies_yum[@]}"
      setup_rbenv
      rbenv install -v "${RUBY_VERSION}"
      rbenv global "${RUBY_VERSION}"
      rbenv rehash
      sudo chown -R "$(whoami)" "$(rbenv prefix)"
      ;;
    debian)
      apt_install "${ruby_build_dependencies_deb[@]}"
      ;;
    ubuntu)
      apt_install "${ruby_build_dependencies_ubuntu[@]}"
      ;;
    *)
      # TODO: handle ubuntu?
      >&2 echo "Error: Need to install ruby ${MINIMUM_RUBY_VERSION}+"
      exit 1
  esac
}


# shellcheck disable=SC2016
setup_rbenv() {
  pushd "$(mktemp -d)" || return 1
  local rbenv_rc=$HOME/setup_rbenv.sh
  git clone https://github.com/sstephenson/rbenv.git ~/.rbenv
  echo 'export PATH="$HOME/.rbenv/bin:$PATH"' >> "${rbenv_rc}"
  echo 'eval "$($HOME/.rbenv/bin/rbenv init -)"' >> "${rbenv_rc}"

  git clone https://github.com/sstephenson/ruby-build.git ~/.rbenv/plugins/ruby-build
  echo 'export PATH="$HOME/.rbenv/plugins/ruby-build/bin:$PATH"' >> "${rbenv_rc}"
  echo ". \"${rbenv_rc}\"" >> ~/.bash_profile
  prepare_rbenv_env

  # Verify rbenv is set up correctly
  curl -fsSL https://github.com/rbenv/rbenv-installer/raw/master/bin/rbenv-doctor | bash
  popd || return 1
}

prepare_rbenv_env() {
  case "${DIST}" in
    centos|fedora)
      local rbenv_rc=$HOME/setup_rbenv.sh
      [[ ! -r "${rbenv_rc}" ]] || . "${rbenv_rc}"
      ;;
  esac

  if command -v rbenv >/dev/null; then
    rbenv rehash
  fi
}

is_version_at_least() {
  local bin_name="${1:?Missing bin name}"; shift
  local version_constraint="${1:?Missing version constraint}"; shift
  local need_to_build=0

  if ! command -v "${bin_name}"; then
    >&2 echo "Warning: ${bin_name} not installed."
    need_to_build=1
  fi

  local installed_version installed_version_major installed_version_minor #version_patch
  installed_version="$("$@")"

  # shellcheck disable=SC2181
  # shellcheck disable=SC2295
  if [[ $? -ne 0 ]]; then
    need_to_build=1
  else
    installed_version_major="${installed_version%%.*}"
    installed_version_minor="${installed_version#*.}"
    installed_version_minor="${installed_version_minor%%.*}"
    installed_version_minor="${installed_version_minor:-0}"
    installed_version_patch="${installed_version#${installed_version_major}.}"
    installed_version_patch="${installed_version_patch#${installed_version_minor}}"
    installed_version_patch="${installed_version_patch#.}"
    installed_version_patch="${installed_version_patch%%.*}"
    installed_version_patch="${installed_version_patch:-0}"

    local need_version_major
    need_version_major="${version_constraint%%.*}"
    need_version_minor="${version_constraint#*.}"
    need_version_minor="${need_version_minor%%.*}"
    need_version_minor="${need_version_minor:-0}"
    need_version_patch="${version_constraint##*.}"
    need_version_patch="${version_constraint#${need_version_major}.}"
    need_version_patch="${need_version_patch#${need_version_minor}}"
    need_version_patch="${need_version_patch#.}"
    need_version_patch="${need_version_patch%%.*}"
    need_version_patch="${need_version_patch:-0}"

    >&2 echo "
    -> installed_version_major=${installed_version_major}
    -> installed_version_minor=${installed_version_minor}
    -> installed_version_patch=${installed_version_patch}
    -> need_version_major=${need_version_major}
    -> need_version_minor=${need_version_minor}
    -> need_version_patch=${need_version_patch}"

    # Naive semver comparison
    if [[ "${installed_version_major}" -lt "${need_version_major}" ]] || \
       [[ "${installed_version_major}" = "${need_version_major}" && "${installed_version_minor}" -lt "${need_version_minor}" ]] || \
       [[ "${installed_version_major}.${installed_version_minor}" = "${need_version_major}.${need_version_minor}" && "${installed_version_patch}" -lt "${need_version_patch}" ]]; then
      need_to_build=1
    fi
  fi

  if [[ 1 = "${need_to_build}" ]]; then
    >&2 echo "Warning: Need to build ${bin_name} since version constraint ${version_constraint} not met."
  else
    >&2 echo "No need to build ${bin_name} since version constraint ${version_constraint} is met."
  fi

  return "${need_to_build}"
}

# Install specified gem.
# Use rbenv when available.  Otherwise use system 'gem', and use 'sudo'
# depending on OS.
# Set SUDO_GEM to 'sudo' to force use of sudo.
# Set SUDO_GEM to 'run' to disable sudo.
gem_install() {
  local gem_name="${1:?Missing gem name}"
  local bin_name="${2:-${gem_name}}"
  if ! command -v "${bin_name}" >/dev/null; then
    if command -v rbenv >/dev/null; then
      gem install "${gem_name}"
      rbenv rehash
    else
      "${SUDO_GEM:-${SUDO:-run}}" gem install "${gem_name}"
    fi
  fi
}

# build+install
build_and_install() {

  export cmakeopts=(
    -DBUILD_SHARED_LIBS="${BUILD_SHARED_LIBS}"
    -DBUILD_TESTING=no
    -DCMAKE_INSTALL_PREFIX="${1:-/tmp}"
  )

  if [[ $# -gt 0 ]]; then
    shift
  fi

  build_rnp "$@"
  make_install VERBOSE="${VERBOSE}"
}

build_rnp() {
  "${CMAKE:-cmake}" "${cmakeopts[@]}" "${1:-.}"
}

make_install() {
  make -j"${MAKE_PARALLEL}" install "$@"
}

is_true_cmake_bool() {
  local arg="${1:?Missing parameter}"
  case "${arg}" in
    yes|on|true|y)
      true
      ;;
    no|off|false|n)
      false
      ;;
    *)
      >&2 echo "Warning: unrecognized boolean expression ($arg).  Continuing and interpreting as 'false' anyway."
      false
  esac
}

# check for install issues
check_build() {
  if is_true_cmake_bool "${BUILD_SHARED_LIBS}"; then
    export pkgflags=
    export gccflags=
    if ! find /usr/lib64/ -maxdepth 1 -type f -name 'librnp*.so' | grep -q . || \
         find /usr/lib64/ -maxdepth 1 -type f -name 'librnp*.a'  | grep -q .; then
      >&2 echo "librnp installed libraries incorrect"
    fi
  else
    export pkgflags=--static
    export gccflags=-lstdc++
    if  find /usr/lib64/ -maxdepth 1 -type f -name 'librnp*.so' | grep -q . || \
      ! find /usr/lib64/ -maxdepth 1 -type f -name 'librnp*.a'  | grep -q .; then
      >&2 echo "librnp installed libraries incorrect"
    fi
  fi
}

# build an example using pkg-config
build_example_pkgconfig() {
  local rnpsrc="$PWD"
  pushd "$(mktemp -d)" || return 1

  # shellcheck disable=SC2046,SC2086
  gcc "${rnpsrc}/src/examples/generate.c" -ogenerate $(pkg-config --cflags --libs $pkgflags librnp) $gccflags
  ./generate
  readelf -d generate
  if is_true_cmake_bool "${BUILD_SHARED_LIBS}"; then
    readelf -d generate | grep -q 'librnp\>'
  else
    readelf -d generate | grep -qv 'librnp\>'
  fi

  # remove the pkgconfig for the next test
  >&2 echo "Checking if librnp- is found in pkg-config list:"
  pkg-config --list-all
  pkg-config --list-all | grep -q '^librnp\>'

  # XXX: debug
  find /usr/lib64 -type f -name 'librnp*'
  find /usr/lib -type f -name 'librnp*'

  find /usr/lib64/pkgconfig -regextype sed -regex '.*librnp\>.*' -exec rm {} +

  # XXX: debug
  find /usr/lib64 -type f -name 'librnp*'
  find /usr/lib -type f -name 'librnp*'

  # should not be found
  >&2 echo "Checking if librnp- is NOT found in pkg-config list:"
  pkg-config --list-all
  pkg-config --list-all | grep -qv '^librnp\>'

  # build an example using cmake targets
  mkdir rnp-project
  pushd rnp-project || return 1

  cat <<"EOF" > mytest.cpp
  #include <rnp/rnp.h>

  int main(int argc, char *argv[]) {
      printf("RNP version: %s\n", rnp_version_string());
      return 0;
  }
EOF

  cat <<"EOF" > CMakeLists.txt
  set(CMAKE_MODULE_PATH "${PROJECT_SOURCE_DIR}")
  find_package(BZip2 REQUIRED)
  find_package(ZLIB REQUIRED)
  find_package(JSON-C 0.11 REQUIRED)
  find_package(Botan2 2.14.0 REQUIRED)
  find_package(rnp REQUIRED)

  cmake_minimum_required(VERSION 3.12)
  add_executable(mytest mytest.cpp)
  target_link_libraries(mytest rnp::librnp)
EOF

  cp "${rnpsrc}"/cmake/Modules/* .
  cmake .
  make VERBOSE="${VERBOSE}"
  ./mytest
  popd
  popd
}
