% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_restricted.R
\name{bayesfactor_restricted}
\alias{bayesfactor_restricted}
\alias{bf_restricted}
\alias{bayesfactor_restricted.stanreg}
\alias{bayesfactor_restricted.brmsfit}
\alias{bayesfactor_restricted.blavaan}
\alias{bayesfactor_restricted.emmGrid}
\alias{bayesfactor_restricted.data.frame}
\alias{as.logical.bayesfactor_restricted}
\title{Bayes Factors (BF) for Order Restricted Models}
\usage{
bayesfactor_restricted(posterior, ...)

bf_restricted(posterior, ...)

\method{bayesfactor_restricted}{stanreg}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  ...
)

\method{bayesfactor_restricted}{brmsfit}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  ...
)

\method{bayesfactor_restricted}{blavaan}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  ...
)

\method{bayesfactor_restricted}{emmGrid}(
  posterior,
  hypothesis,
  prior = NULL,
  verbose = TRUE,
  ...
)

\method{bayesfactor_restricted}{data.frame}(
  posterior,
  hypothesis,
  prior = NULL,
  rvar_col = NULL,
  ...
)

\method{as.logical}{bayesfactor_restricted}(x, which = c("posterior", "prior"), ...)
}
\arguments{
\item{posterior}{A \code{stanreg} / \code{brmsfit} object, \code{emmGrid} or a data frame - representing
a posterior distribution(s) from (see Details).}

\item{...}{Currently not used.}

\item{hypothesis}{A character vector specifying the restrictions as logical conditions (see examples below).}

\item{prior}{An object representing a prior distribution (see Details).}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects, random effects or both be
returned? Only applies to mixed models. May be abbreviated.}

\item{component}{Should results for all parameters, parameters for the
conditional model or the zero-inflated part of the model be returned? May
be abbreviated. Only applies to \pkg{brms}-models.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{x}{An object of class \code{bayesfactor_restricted}}

\item{which}{Should the logical matrix be of the posterior or prior distribution(s)?}
}
\value{
A data frame containing the (log) Bayes factor representing evidence
\emph{against} the un-restricted model (Use \code{as.numeric()} to extract the
non-log Bayes factors; see examples). (A \code{bool_results} attribute contains
the results for each sample, indicating if they are included or not in the
hypothesized restriction.)
}
\description{
This method computes Bayes factors for comparing a model with an order restrictions on its parameters
with the fully unrestricted model. \emph{Note that this method should only be used for confirmatory analyses}.
\cr \cr
The \verb{bf_*} function is an alias of the main function.
\cr \cr
\strong{For more info, in particular on specifying correct priors for factors with more than 2 levels,
see \href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.}
}
\details{
This method is used to compute Bayes factors for order-restricted models vs un-restricted
models by setting an order restriction on the prior and posterior distributions
(\cite{Morey & Wagenmakers, 2013}).
\cr\cr
(Though it is possible to use \code{bayesfactor_restricted()} to test interval restrictions,
it is more suitable for testing order restrictions; see examples).
}
\section{Setting the correct \code{prior}}{

For the computation of Bayes factors, the model priors must be proper priors
(at the very least they should be \emph{not flat}, and it is preferable that
they be \emph{informative}); As the priors for the alternative get wider, the
likelihood of the null value(s) increases, to the extreme that for completely
flat priors the null is infinitely more favorable than the alternative (this
is called \emph{the Jeffreys-Lindley-Bartlett paradox}). Thus, you should
only ever try (or want) to compute a Bayes factor when you have an informed
prior.
\cr\cr
(Note that by default, \code{brms::brm()} uses flat priors for fixed-effects;
See example below.)
\cr\cr
It is important to provide the correct \code{prior} for meaningful results,
to match the \code{posterior}-type input:
\itemize{
\item \strong{A numeric vector} - \code{prior} should also be a \emph{numeric vector}, representing the prior-estimate.
\item \strong{A data frame} - \code{prior} should also be a \emph{data frame}, representing the prior-estimates, in matching column order.
\itemize{
\item If \code{rvar_col} is specified, \code{prior} should be \emph{the name of an \code{rvar} column} that represents the prior-estimates.
}
\item \strong{Supported Bayesian model (\code{stanreg}, \code{brmsfit}, etc.)}
\itemize{
\item \code{prior} should be \emph{a model an equivalent model with MCMC samples from the priors \strong{only}}. See \code{\link[=unupdate]{unupdate()}}.
\item If \code{prior} is set to \code{NULL}, \code{\link[=unupdate]{unupdate()}} is called internally (not supported for \code{brmsfit_multiple} model).
}
\item \strong{Output from a \code{{marginaleffects}} function} - \code{prior} should also be \emph{an equivalent output} from a \code{{marginaleffects}} function based on a prior-model
(See \code{\link[=unupdate]{unupdate()}}).
\item \strong{Output from an \code{{emmeans}} function}
\itemize{
\item \code{prior} should also be \emph{an equivalent output} from an \code{{emmeans}} function based on a prior-model (See \code{\link[=unupdate]{unupdate()}}).
\item \code{prior} can also be \emph{the original (posterior) model}, in which case the function
will try to "unupdate" the estimates (not supported if the estimates have undergone
any transformations -- \code{"log"}, \code{"response"}, etc. -- or any \code{regrid}ing).
}
}
}

\section{Interpreting Bayes Factors}{

A Bayes factor greater than 1 can be interpreted as evidence against the
null, at which one convention is that a Bayes factor greater than 3 can be
considered as "substantial" evidence against the null (and vice versa, a
Bayes factor smaller than 1/3 indicates substantial evidence in favor of the
null-model) (\cite{Wetzels et al. 2011}).
}

\examples{
set.seed(444)
library(bayestestR)
prior <- data.frame(
  A = rnorm(500),
  B = rnorm(500),
  C = rnorm(500)
)

posterior <- data.frame(
  A = rnorm(500, .4, 0.7),
  B = rnorm(500, -.2, 0.4),
  C = rnorm(500, 0, 0.5)
)

hyps <- c(
  "A > B & B > C",
  "A > B & A > C",
  "C > A"
)


(b <- bayesfactor_restricted(posterior, hypothesis = hyps, prior = prior))

bool <- as.logical(b, which = "posterior")
head(bool)

\dontshow{if (require("see") && require("patchwork")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

see::plots(
  plot(estimate_density(posterior)),
  # distribution **conditional** on the restrictions
  plot(estimate_density(posterior[bool[, hyps[1]], ])) + ggplot2::ggtitle(hyps[1]),
  plot(estimate_density(posterior[bool[, hyps[2]], ])) + ggplot2::ggtitle(hyps[2]),
  plot(estimate_density(posterior[bool[, hyps[3]], ])) + ggplot2::ggtitle(hyps[3]),
  guides = "collect"
)
\dontshow{\}) # examplesIf}
\dontshow{if (require("rstanarm")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# rstanarm models
# ---------------
data("mtcars")

fit_stan <- rstanarm::stan_glm(mpg ~ wt + cyl + am,
  data = mtcars, refresh = 0
)
hyps <- c(
  "am > 0 & cyl < 0",
  "cyl < 0",
  "wt - cyl > 0"
)

bayesfactor_restricted(fit_stan, hypothesis = hyps)
}
\dontshow{\}) # examplesIf}
\dontshow{if (require("rstanarm") && require("emmeans")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# emmGrid objects
# ---------------
# replicating http://bayesfactor.blogspot.com/2015/01/multiple-comparisons-with-bayesfactor-2.html
data("disgust")
contrasts(disgust$condition) <- contr.equalprior_pairs # see vignette
fit_model <- rstanarm::stan_glm(score ~ condition, data = disgust, family = gaussian())

em_condition <- emmeans::emmeans(fit_model, ~condition, data = disgust)
hyps <- c("lemon < control & control < sulfur")

bayesfactor_restricted(em_condition, prior = fit_model, hypothesis = hyps)
# > # Bayes Factor (Order-Restriction)
# >
# >                          Hypothesis P(Prior) P(Posterior)   BF
# >  lemon < control & control < sulfur     0.17         0.75 4.49
# > ---
# > Bayes factors for the restricted model vs. the un-restricted model.
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Morey, R. D., & Wagenmakers, E. J. (2014). Simple relation between Bayesian order-restricted and
point-null hypothesis tests. Statistics & Probability Letters, 92, 121-124.
\item Morey, R. D., & Rouder, J. N. (2011). Bayes factor approaches for testing interval null hypotheses.
Psychological methods, 16(4), 406.
\item Morey, R. D. (Jan, 2015). Multiple Comparisons with BayesFactor, Part 2 – order restrictions.
Retrieved from https://richarddmorey.org/category/order-restrictions/.
}
}
