/* ========================================================================
 * Copyright (c) 2005-2022 The OPC Foundation, Inc. All rights reserved.
 *
 * OPC Foundation MIT License 1.00
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * The complete license agreement can be found here:
 * http://opcfoundation.org/License/MIT/1.00/
 * ======================================================================*/

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Runtime.Serialization;

namespace Opc.Ua
{
    #region ServerObjectState Class
    #if (!OPCUA_EXCLUDE_ServerObjectState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerObjectState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public ServerObjectState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ServerType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (UrisVersion != null)
            {
                UrisVersion.Initialize(context, UrisVersion_InitializationString);
            }

            if (EstimatedReturnTime != null)
            {
                EstimatedReturnTime.Initialize(context, EstimatedReturnTime_InitializationString);
            }

            if (LocalTime != null)
            {
                LocalTime.Initialize(context, LocalTime_InitializationString);
            }

            if (Namespaces != null)
            {
                Namespaces.Initialize(context, Namespaces_InitializationString);
            }

            if (GetMonitoredItems != null)
            {
                GetMonitoredItems.Initialize(context, GetMonitoredItems_InitializationString);
            }

            if (ResendData != null)
            {
                ResendData.Initialize(context, ResendData_InitializationString);
            }

            if (SetSubscriptionDurable != null)
            {
                SetSubscriptionDurable.Initialize(context, SetSubscriptionDurable_InitializationString);
            }

            if (RequestServerStateChange != null)
            {
                RequestServerStateChange.Initialize(context, RequestServerStateChange_InitializationString);
            }
        }

        #region Initialization String
        private const string UrisVersion_InitializationString =
           "//////////8VcIkKAgAAAAAACwAAAFVyaXNWZXJzaW9uAQCbOgAuAESbOgAAAQAGUv////8BAQAAAAAA" +
           "QI9A/////wAAAAA=";

        private const string EstimatedReturnTime_InitializationString =
           "//////////8VcIkKAgAAAAAAEwAAAEVzdGltYXRlZFJldHVyblRpbWUBAFIyAC4ARFIyAAAADf////8B" +
           "AQAAAAAAQI9A/////wAAAAA=";

        private const string LocalTime_InitializationString =
           "//////////8VcIkKAgAAAAAACQAAAExvY2FsVGltZQEAzEQALgBEzEQAAAEA0CL/////AQEAAAAAAECP" +
           "QP////8AAAAA";

        private const string Namespaces_InitializationString =
           "//////////8EYIAKAQAAAAAACgAAAE5hbWVzcGFjZXMBAActAC8BAH0tBy0AAP////8AAAAA";

        private const string GetMonitoredItems_InitializationString =
           "//////////8EYYIKBAAAAAAAEQAAAEdldE1vbml0b3JlZEl0ZW1zAQDhLAAvAQDhLOEsAAABAf////8C" +
           "AAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEA4iwALgBE4iwAAJYBAAAAAQAqAQEdAAAADgAA" +
           "AFN1YnNjcmlwdGlvbklkAAf/////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAABdgqQoCAAAA" +
           "AAAPAAAAT3V0cHV0QXJndW1lbnRzAQDjLAAuAETjLAAAlgIAAAABACoBASAAAAANAAAAU2VydmVySGFu" +
           "ZGxlcwAHAQAAAAEAAAAAAAAAAAEAKgEBIAAAAA0AAABDbGllbnRIYW5kbGVzAAcBAAAAAQAAAAAAAAAA" +
           "AQAoAQEAAAABAAAAAgAAAAEB/////wAAAAA=";

        private const string ResendData_InitializationString =
           "//////////8EYYIKBAAAAAAACgAAAFJlc2VuZERhdGEBAEcyAC8BAEcyRzIAAAEB/////wEAAAAXYKkK" +
           "AgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBIMgAuAERIMgAAlgEAAAABACoBAR0AAAAOAAAAU3Vic2Ny" +
           "aXB0aW9uSWQAB/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAA";

        private const string SetSubscriptionDurable_InitializationString =
           "//////////8EYYIKBAAAAAAAFgAAAFNldFN1YnNjcmlwdGlvbkR1cmFibGUBAMoxAC8BAMoxyjEAAAEB" +
           "/////wIAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQDLMQAuAETLMQAAlgIAAAABACoBAR0A" +
           "AAAOAAAAU3Vic2NyaXB0aW9uSWQAB/////8AAAAAAAEAKgEBHgAAAA8AAABMaWZldGltZUluSG91cnMA" +
           "B/////8AAAAAAAEAKAEBAAAAAQAAAAIAAAABAf////8AAAAAF2CpCgIAAAAAAA8AAABPdXRwdXRBcmd1" +
           "bWVudHMBAMwxAC4ARMwxAACWAQAAAAEAKgEBJQAAABYAAABSZXZpc2VkTGlmZXRpbWVJbkhvdXJzAAf/" +
           "////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAAA==";

        private const string RequestServerStateChange_InitializationString =
           "//////////8EYYIKBAAAAAAAGAAAAFJlcXVlc3RTZXJ2ZXJTdGF0ZUNoYW5nZQEAUzIALwEAUzJTMgAA" +
           "AQH/////AQAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAFQyAC4ARFQyAACWBQAAAAEAKgEB" +
           "FgAAAAUAAABTdGF0ZQEAVAP/////AAAAAAABACoBASIAAAATAAAARXN0aW1hdGVkUmV0dXJuVGltZQAN" +
           "/////wAAAAAAAQAqAQEiAAAAEwAAAFNlY29uZHNUaWxsU2h1dGRvd24AB/////8AAAAAAAEAKgEBFQAA" +
           "AAYAAABSZWFzb24AFf////8AAAAAAAEAKgEBFgAAAAcAAABSZXN0YXJ0AAH/////AAAAAAABACgBAQAA" +
           "AAEAAAAFAAAAAQH/////AAAAAA==";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAEgAAAFNlcnZlclR5cGVJbnN0YW5jZQEA1AcBANQH1AcAAP////8RAAAA" +
           "F3CJCgIAAAAAAAsAAABTZXJ2ZXJBcnJheQEA1QcALgBE1QcAAAAMAQAAAAEAAAAAAAAAAQEAAAAAAECP" +
           "QP////8AAAAAF3CJCgIAAAAAAA4AAABOYW1lc3BhY2VBcnJheQEA1gcALgBE1gcAAAAMAQAAAAEAAAAA" +
           "AAAAAQEAAAAAAECPQP////8AAAAAFXCJCgIAAAAAAAsAAABVcmlzVmVyc2lvbgEAmzoALgBEmzoAAAEA" +
           "BlL/////AQEAAAAAAECPQP////8AAAAAFXCJCgIAAAAAAAwAAABTZXJ2ZXJTdGF0dXMBANcHAC8BAFoI" +
           "1wcAAAEAXgP/////AQEAAAAAAECPQP////8GAAAAFWCJCgIAAAAAAAkAAABTdGFydFRpbWUBAAIMAC8A" +
           "PwIMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAEN1cnJlbnRUaW1lAQADDAAvAD8DDAAA" +
           "AQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAUAAABTdGF0ZQEABAwALwA/BAwAAAEAVAP/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAJAAAAQnVpbGRJbmZvAQAFDAAvAQDrCwUMAAABAFIB/////wEB/////wYA" +
           "AAAVcIkKAgAAAAAACgAAAFByb2R1Y3RVcmkBAAYMAC8APwYMAAAADP////8BAQAAAAAAQI9A/////wAA" +
           "AAAVcIkKAgAAAAAAEAAAAE1hbnVmYWN0dXJlck5hbWUBAAcMAC8APwcMAAAADP////8BAQAAAAAAQI9A" +
           "/////wAAAAAVcIkKAgAAAAAACwAAAFByb2R1Y3ROYW1lAQAIDAAvAD8IDAAAAAz/////AQEAAAAAAECP" +
           "QP////8AAAAAFXCJCgIAAAAAAA8AAABTb2Z0d2FyZVZlcnNpb24BAAkMAC8APwkMAAAADP////8BAQAA" +
           "AAAAQI9A/////wAAAAAVcIkKAgAAAAAACwAAAEJ1aWxkTnVtYmVyAQAKDAAvAD8KDAAAAAz/////AQEA" +
           "AAAAAECPQP////8AAAAAFXCJCgIAAAAAAAkAAABCdWlsZERhdGUBAAsMAC8APwsMAAABACYB/////wEB" +
           "AAAAAABAj0D/////AAAAABVgiQoCAAAAAAATAAAAU2Vjb25kc1RpbGxTaHV0ZG93bgEADAwALwA/DAwA" +
           "AAAH/////wEB/////wAAAAAVYIkKAgAAAAAADgAAAFNodXRkb3duUmVhc29uAQANDAAvAD8NDAAAABX/" +
           "////AQH/////AAAAABVwiQoCAAAAAAAMAAAAU2VydmljZUxldmVsAQDYBwAuAETYBwAAAAP/////AQEA" +
           "AAAAAECPQP////8AAAAAFXCJCgIAAAAAAAgAAABBdWRpdGluZwEAtgoALgBEtgoAAAAB/////wEBAAAA" +
           "AABAj0D/////AAAAABVwiQoCAAAAAAATAAAARXN0aW1hdGVkUmV0dXJuVGltZQEAUjIALgBEUjIAAAAN" +
           "/////wEBAAAAAABAj0D/////AAAAABVwiQoCAAAAAAAJAAAATG9jYWxUaW1lAQDMRAAuAETMRAAAAQDQ" +
           "Iv////8BAQAAAAAAQI9A/////wAAAAAEYIAKAQAAAAAAEgAAAFNlcnZlckNhcGFiaWxpdGllcwEA2QcA" +
           "LwEA3QfZBwAA/////wkAAAAXYIkKAgAAAAAAEgAAAFNlcnZlclByb2ZpbGVBcnJheQEADgwALgBEDgwA" +
           "AAAMAQAAAAEAAAAAAAAAAQH/////AAAAABdgiQoCAAAAAAANAAAATG9jYWxlSWRBcnJheQEADwwALgBE" +
           "DwwAAAEAJwEBAAAAAQAAAAAAAAABAf////8AAAAAFWCJCgIAAAAAABYAAABNaW5TdXBwb3J0ZWRTYW1w" +
           "bGVSYXRlAQAQDAAuAEQQDAAAAQAiAf////8BAf////8AAAAAFWCJCgIAAAAAABsAAABNYXhCcm93c2VD" +
           "b250aW51YXRpb25Qb2ludHMBABEMAC4ARBEMAAAABf////8BAf////8AAAAAFWCJCgIAAAAAABoAAABN" +
           "YXhRdWVyeUNvbnRpbnVhdGlvblBvaW50cwEAEgwALgBEEgwAAAAF/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAHAAAAE1heEhpc3RvcnlDb250aW51YXRpb25Qb2ludHMBABMMAC4ARBMMAAAABf////8BAf////8A" +
           "AAAAF2CJCgIAAAAAABQAAABTb2Z0d2FyZUNlcnRpZmljYXRlcwEAFAwALgBEFAwAAAEAWAEBAAAAAQAA" +
           "AAAAAAABAf////8AAAAABGCACgEAAAAAAA4AAABNb2RlbGxpbmdSdWxlcwEAFQwALwA9FQwAAP////8A" +
           "AAAABGCACgEAAAAAABIAAABBZ2dyZWdhdGVGdW5jdGlvbnMBABYMAC8APRYMAAD/////AAAAAARggAoB" +
           "AAAAAAARAAAAU2VydmVyRGlhZ25vc3RpY3MBANoHAC8BAOQH2gcAAP////8EAAAAFWCJCgIAAAAAABgA" +
           "AABTZXJ2ZXJEaWFnbm9zdGljc1N1bW1hcnkBABcMAC8BAGYIFwwAAAEAWwP/////AQH/////DAAAABVg" +
           "iQoCAAAAAAAPAAAAU2VydmVyVmlld0NvdW50AQAYDAAvAD8YDAAAAAf/////AQH/////AAAAABVgiQoC" +
           "AAAAAAATAAAAQ3VycmVudFNlc3Npb25Db3VudAEAGQwALwA/GQwAAAAH/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAAFQAAAEN1bXVsYXRlZFNlc3Npb25Db3VudAEAGgwALwA/GgwAAAAH/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAAHAAAAFNlY3VyaXR5UmVqZWN0ZWRTZXNzaW9uQ291bnQBABsMAC8APxsMAAAAB/////8B" +
           "Af////8AAAAAFWCJCgIAAAAAABQAAABSZWplY3RlZFNlc3Npb25Db3VudAEAHAwALwA/HAwAAAAH////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAEwAAAFNlc3Npb25UaW1lb3V0Q291bnQBAB0MAC8APx0MAAAAB///" +
           "//8BAf////8AAAAAFWCJCgIAAAAAABEAAABTZXNzaW9uQWJvcnRDb3VudAEAHgwALwA/HgwAAAAH////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAFwAAAFB1Ymxpc2hpbmdJbnRlcnZhbENvdW50AQAgDAAvAD8gDAAA" +
           "AAf/////AQH/////AAAAABVgiQoCAAAAAAAYAAAAQ3VycmVudFN1YnNjcmlwdGlvbkNvdW50AQAhDAAv" +
           "AD8hDAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAaAAAAQ3VtdWxhdGVkU3Vic2NyaXB0aW9uQ291" +
           "bnQBACIMAC8APyIMAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAB0AAABTZWN1cml0eVJlamVjdGVk" +
           "UmVxdWVzdHNDb3VudAEAIwwALwA/IwwAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFQAAAFJlamVj" +
           "dGVkUmVxdWVzdHNDb3VudAEAJAwALwA/JAwAAAAH/////wEB/////wAAAAAXYIkKAgAAAAAAHAAAAFN1" +
           "YnNjcmlwdGlvbkRpYWdub3N0aWNzQXJyYXkBACYMAC8BAHsIJgwAAAEAagMBAAAAAQAAAAAAAAABAf//" +
           "//8AAAAABGCACgEAAAAAABoAAABTZXNzaW9uc0RpYWdub3N0aWNzU3VtbWFyeQEAJwwALwEA6gcnDAAA" +
           "/////wIAAAAXYIkKAgAAAAAAFwAAAFNlc3Npb25EaWFnbm9zdGljc0FycmF5AQAoDAAvAQCUCCgMAAAB" +
           "AGEDAQAAAAEAAAAAAAAAAQH/////AAAAABdgiQoCAAAAAAAfAAAAU2Vzc2lvblNlY3VyaXR5RGlhZ25v" +
           "c3RpY3NBcnJheQEAKQwALwEAwwgpDAAAAQBkAwEAAAABAAAAAAAAAAEB/////wAAAAAVYIkKAgAAAAAA" +
           "CwAAAEVuYWJsZWRGbGFnAQAqDAAuAEQqDAAAAAH/////AwP/////AAAAAARggAoBAAAAAAAQAAAAVmVu" +
           "ZG9yU2VydmVySW5mbwEA2wcALwEA8QfbBwAA/////wAAAAAEYIAKAQAAAAAAEAAAAFNlcnZlclJlZHVu" +
           "ZGFuY3kBANwHAC8BAPIH3AcAAP////8BAAAAFWCJCgIAAAAAABEAAABSZWR1bmRhbmN5U3VwcG9ydAEA" +
           "KwwALgBEKwwAAAEAUwP/////AQH/////AAAAAARggAoBAAAAAAAKAAAATmFtZXNwYWNlcwEABy0ALwEA" +
           "fS0HLQAA/////wAAAAAEYYIKBAAAAAAAEQAAAEdldE1vbml0b3JlZEl0ZW1zAQDhLAAvAQDhLOEsAAAB" +
           "Af////8CAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEA4iwALgBE4iwAAJYBAAAAAQAqAQEd" +
           "AAAADgAAAFN1YnNjcmlwdGlvbklkAAf/////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAABdg" +
           "qQoCAAAAAAAPAAAAT3V0cHV0QXJndW1lbnRzAQDjLAAuAETjLAAAlgIAAAABACoBASAAAAANAAAAU2Vy" +
           "dmVySGFuZGxlcwAHAQAAAAEAAAAAAAAAAAEAKgEBIAAAAA0AAABDbGllbnRIYW5kbGVzAAcBAAAAAQAA" +
           "AAAAAAAAAQAoAQEAAAABAAAAAgAAAAEB/////wAAAAAEYYIKBAAAAAAACgAAAFJlc2VuZERhdGEBAEcy" +
           "AC8BAEcyRzIAAAEB/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBIMgAuAERIMgAA" +
           "lgEAAAABACoBAR0AAAAOAAAAU3Vic2NyaXB0aW9uSWQAB/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAAB" +
           "Af////8AAAAABGGCCgQAAAAAABYAAABTZXRTdWJzY3JpcHRpb25EdXJhYmxlAQDKMQAvAQDKMcoxAAAB" +
           "Af////8CAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEAyzEALgBEyzEAAJYCAAAAAQAqAQEd" +
           "AAAADgAAAFN1YnNjcmlwdGlvbklkAAf/////AAAAAAABACoBAR4AAAAPAAAATGlmZXRpbWVJbkhvdXJz" +
           "AAf/////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAABdgqQoCAAAAAAAPAAAAT3V0cHV0QXJn" +
           "dW1lbnRzAQDMMQAuAETMMQAAlgEAAAABACoBASUAAAAWAAAAUmV2aXNlZExpZmV0aW1lSW5Ib3VycwAH" +
           "/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAAAAAAGAAAAFJlcXVlc3RTZXJ2" +
           "ZXJTdGF0ZUNoYW5nZQEAUzIALwEAUzJTMgAAAQH/////AQAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1" +
           "bWVudHMBAFQyAC4ARFQyAACWBQAAAAEAKgEBFgAAAAUAAABTdGF0ZQEAVAP/////AAAAAAABACoBASIA" +
           "AAATAAAARXN0aW1hdGVkUmV0dXJuVGltZQAN/////wAAAAAAAQAqAQEiAAAAEwAAAFNlY29uZHNUaWxs" +
           "U2h1dGRvd24AB/////8AAAAAAAEAKgEBFQAAAAYAAABSZWFzb24AFf////8AAAAAAAEAKgEBFgAAAAcA" +
           "AABSZXN0YXJ0AAH/////AAAAAAABACgBAQAAAAEAAAAFAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string[]> ServerArray
        {
            get
            {
                return m_serverArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverArray = value;
            }
        }

        /// <remarks />
        public PropertyState<string[]> NamespaceArray
        {
            get
            {
                return m_namespaceArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespaceArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespaceArray = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> UrisVersion
        {
            get
            {
                return m_urisVersion;
            }

            set
            {
                if (!Object.ReferenceEquals(m_urisVersion, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_urisVersion = value;
            }
        }

        /// <remarks />
        public ServerStatusState ServerStatus
        {
            get
            {
                return m_serverStatus;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverStatus, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverStatus = value;
            }
        }

        /// <remarks />
        public PropertyState<byte> ServiceLevel
        {
            get
            {
                return m_serviceLevel;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serviceLevel, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serviceLevel = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> Auditing
        {
            get
            {
                return m_auditing;
            }

            set
            {
                if (!Object.ReferenceEquals(m_auditing, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_auditing = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> EstimatedReturnTime
        {
            get
            {
                return m_estimatedReturnTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_estimatedReturnTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_estimatedReturnTime = value;
            }
        }

        /// <remarks />
        public PropertyState<TimeZoneDataType> LocalTime
        {
            get
            {
                return m_localTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_localTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_localTime = value;
            }
        }

        /// <remarks />
        public ServerCapabilitiesState ServerCapabilities
        {
            get
            {
                return m_serverCapabilities;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverCapabilities, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverCapabilities = value;
            }
        }

        /// <remarks />
        public ServerDiagnosticsState ServerDiagnostics
        {
            get
            {
                return m_serverDiagnostics;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverDiagnostics, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverDiagnostics = value;
            }
        }

        /// <remarks />
        public VendorServerInfoState VendorServerInfo
        {
            get
            {
                return m_vendorServerInfo;
            }

            set
            {
                if (!Object.ReferenceEquals(m_vendorServerInfo, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_vendorServerInfo = value;
            }
        }

        /// <remarks />
        public ServerRedundancyState ServerRedundancy
        {
            get
            {
                return m_serverRedundancy;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverRedundancy, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverRedundancy = value;
            }
        }

        /// <remarks />
        public NamespacesState Namespaces
        {
            get
            {
                return m_namespaces;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespaces, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespaces = value;
            }
        }

        /// <remarks />
        public GetMonitoredItemsMethodState GetMonitoredItems
        {
            get
            {
                return m_getMonitoredItemsMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_getMonitoredItemsMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_getMonitoredItemsMethod = value;
            }
        }

        /// <remarks />
        public ResendDataMethodState ResendData
        {
            get
            {
                return m_resendDataMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_resendDataMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_resendDataMethod = value;
            }
        }

        /// <remarks />
        public SetSubscriptionDurableMethodState SetSubscriptionDurable
        {
            get
            {
                return m_setSubscriptionDurableMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_setSubscriptionDurableMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_setSubscriptionDurableMethod = value;
            }
        }

        /// <remarks />
        public RequestServerStateChangeMethodState RequestServerStateChange
        {
            get
            {
                return m_requestServerStateChangeMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_requestServerStateChangeMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_requestServerStateChangeMethod = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverArray != null)
            {
                children.Add(m_serverArray);
            }

            if (m_namespaceArray != null)
            {
                children.Add(m_namespaceArray);
            }

            if (m_urisVersion != null)
            {
                children.Add(m_urisVersion);
            }

            if (m_serverStatus != null)
            {
                children.Add(m_serverStatus);
            }

            if (m_serviceLevel != null)
            {
                children.Add(m_serviceLevel);
            }

            if (m_auditing != null)
            {
                children.Add(m_auditing);
            }

            if (m_estimatedReturnTime != null)
            {
                children.Add(m_estimatedReturnTime);
            }

            if (m_localTime != null)
            {
                children.Add(m_localTime);
            }

            if (m_serverCapabilities != null)
            {
                children.Add(m_serverCapabilities);
            }

            if (m_serverDiagnostics != null)
            {
                children.Add(m_serverDiagnostics);
            }

            if (m_vendorServerInfo != null)
            {
                children.Add(m_vendorServerInfo);
            }

            if (m_serverRedundancy != null)
            {
                children.Add(m_serverRedundancy);
            }

            if (m_namespaces != null)
            {
                children.Add(m_namespaces);
            }

            if (m_getMonitoredItemsMethod != null)
            {
                children.Add(m_getMonitoredItemsMethod);
            }

            if (m_resendDataMethod != null)
            {
                children.Add(m_resendDataMethod);
            }

            if (m_setSubscriptionDurableMethod != null)
            {
                children.Add(m_setSubscriptionDurableMethod);
            }

            if (m_requestServerStateChangeMethod != null)
            {
                children.Add(m_requestServerStateChangeMethod);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerArray:
                {
                    if (createOrReplace)
                    {
                        if (ServerArray == null)
                        {
                            if (replacement == null)
                            {
                                ServerArray = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                ServerArray = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = ServerArray;
                    break;
                }

                case Opc.Ua.BrowseNames.NamespaceArray:
                {
                    if (createOrReplace)
                    {
                        if (NamespaceArray == null)
                        {
                            if (replacement == null)
                            {
                                NamespaceArray = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                NamespaceArray = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = NamespaceArray;
                    break;
                }

                case Opc.Ua.BrowseNames.UrisVersion:
                {
                    if (createOrReplace)
                    {
                        if (UrisVersion == null)
                        {
                            if (replacement == null)
                            {
                                UrisVersion = new PropertyState<uint>(this);
                            }
                            else
                            {
                                UrisVersion = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = UrisVersion;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerStatus:
                {
                    if (createOrReplace)
                    {
                        if (ServerStatus == null)
                        {
                            if (replacement == null)
                            {
                                ServerStatus = new ServerStatusState(this);
                            }
                            else
                            {
                                ServerStatus = (ServerStatusState)replacement;
                            }
                        }
                    }

                    instance = ServerStatus;
                    break;
                }

                case Opc.Ua.BrowseNames.ServiceLevel:
                {
                    if (createOrReplace)
                    {
                        if (ServiceLevel == null)
                        {
                            if (replacement == null)
                            {
                                ServiceLevel = new PropertyState<byte>(this);
                            }
                            else
                            {
                                ServiceLevel = (PropertyState<byte>)replacement;
                            }
                        }
                    }

                    instance = ServiceLevel;
                    break;
                }

                case Opc.Ua.BrowseNames.Auditing:
                {
                    if (createOrReplace)
                    {
                        if (Auditing == null)
                        {
                            if (replacement == null)
                            {
                                Auditing = new PropertyState<bool>(this);
                            }
                            else
                            {
                                Auditing = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = Auditing;
                    break;
                }

                case Opc.Ua.BrowseNames.EstimatedReturnTime:
                {
                    if (createOrReplace)
                    {
                        if (EstimatedReturnTime == null)
                        {
                            if (replacement == null)
                            {
                                EstimatedReturnTime = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                EstimatedReturnTime = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = EstimatedReturnTime;
                    break;
                }

                case Opc.Ua.BrowseNames.LocalTime:
                {
                    if (createOrReplace)
                    {
                        if (LocalTime == null)
                        {
                            if (replacement == null)
                            {
                                LocalTime = new PropertyState<TimeZoneDataType>(this);
                            }
                            else
                            {
                                LocalTime = (PropertyState<TimeZoneDataType>)replacement;
                            }
                        }
                    }

                    instance = LocalTime;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerCapabilities:
                {
                    if (createOrReplace)
                    {
                        if (ServerCapabilities == null)
                        {
                            if (replacement == null)
                            {
                                ServerCapabilities = new ServerCapabilitiesState(this);
                            }
                            else
                            {
                                ServerCapabilities = (ServerCapabilitiesState)replacement;
                            }
                        }
                    }

                    instance = ServerCapabilities;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerDiagnostics:
                {
                    if (createOrReplace)
                    {
                        if (ServerDiagnostics == null)
                        {
                            if (replacement == null)
                            {
                                ServerDiagnostics = new ServerDiagnosticsState(this);
                            }
                            else
                            {
                                ServerDiagnostics = (ServerDiagnosticsState)replacement;
                            }
                        }
                    }

                    instance = ServerDiagnostics;
                    break;
                }

                case Opc.Ua.BrowseNames.VendorServerInfo:
                {
                    if (createOrReplace)
                    {
                        if (VendorServerInfo == null)
                        {
                            if (replacement == null)
                            {
                                VendorServerInfo = new VendorServerInfoState(this);
                            }
                            else
                            {
                                VendorServerInfo = (VendorServerInfoState)replacement;
                            }
                        }
                    }

                    instance = VendorServerInfo;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerRedundancy:
                {
                    if (createOrReplace)
                    {
                        if (ServerRedundancy == null)
                        {
                            if (replacement == null)
                            {
                                ServerRedundancy = new ServerRedundancyState(this);
                            }
                            else
                            {
                                ServerRedundancy = (ServerRedundancyState)replacement;
                            }
                        }
                    }

                    instance = ServerRedundancy;
                    break;
                }

                case Opc.Ua.BrowseNames.Namespaces:
                {
                    if (createOrReplace)
                    {
                        if (Namespaces == null)
                        {
                            if (replacement == null)
                            {
                                Namespaces = new NamespacesState(this);
                            }
                            else
                            {
                                Namespaces = (NamespacesState)replacement;
                            }
                        }
                    }

                    instance = Namespaces;
                    break;
                }

                case Opc.Ua.BrowseNames.GetMonitoredItems:
                {
                    if (createOrReplace)
                    {
                        if (GetMonitoredItems == null)
                        {
                            if (replacement == null)
                            {
                                GetMonitoredItems = new GetMonitoredItemsMethodState(this);
                            }
                            else
                            {
                                GetMonitoredItems = (GetMonitoredItemsMethodState)replacement;
                            }
                        }
                    }

                    instance = GetMonitoredItems;
                    break;
                }

                case Opc.Ua.BrowseNames.ResendData:
                {
                    if (createOrReplace)
                    {
                        if (ResendData == null)
                        {
                            if (replacement == null)
                            {
                                ResendData = new ResendDataMethodState(this);
                            }
                            else
                            {
                                ResendData = (ResendDataMethodState)replacement;
                            }
                        }
                    }

                    instance = ResendData;
                    break;
                }

                case Opc.Ua.BrowseNames.SetSubscriptionDurable:
                {
                    if (createOrReplace)
                    {
                        if (SetSubscriptionDurable == null)
                        {
                            if (replacement == null)
                            {
                                SetSubscriptionDurable = new SetSubscriptionDurableMethodState(this);
                            }
                            else
                            {
                                SetSubscriptionDurable = (SetSubscriptionDurableMethodState)replacement;
                            }
                        }
                    }

                    instance = SetSubscriptionDurable;
                    break;
                }

                case Opc.Ua.BrowseNames.RequestServerStateChange:
                {
                    if (createOrReplace)
                    {
                        if (RequestServerStateChange == null)
                        {
                            if (replacement == null)
                            {
                                RequestServerStateChange = new RequestServerStateChangeMethodState(this);
                            }
                            else
                            {
                                RequestServerStateChange = (RequestServerStateChangeMethodState)replacement;
                            }
                        }
                    }

                    instance = RequestServerStateChange;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string[]> m_serverArray;
        private PropertyState<string[]> m_namespaceArray;
        private PropertyState<uint> m_urisVersion;
        private ServerStatusState m_serverStatus;
        private PropertyState<byte> m_serviceLevel;
        private PropertyState<bool> m_auditing;
        private PropertyState<DateTime> m_estimatedReturnTime;
        private PropertyState<TimeZoneDataType> m_localTime;
        private ServerCapabilitiesState m_serverCapabilities;
        private ServerDiagnosticsState m_serverDiagnostics;
        private VendorServerInfoState m_vendorServerInfo;
        private ServerRedundancyState m_serverRedundancy;
        private NamespacesState m_namespaces;
        private GetMonitoredItemsMethodState m_getMonitoredItemsMethod;
        private ResendDataMethodState m_resendDataMethod;
        private SetSubscriptionDurableMethodState m_setSubscriptionDurableMethod;
        private RequestServerStateChangeMethodState m_requestServerStateChangeMethod;
        #endregion
    }
    #endif
    #endregion

    #region ServerCapabilitiesState Class
    #if (!OPCUA_EXCLUDE_ServerCapabilitiesState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerCapabilitiesState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public ServerCapabilitiesState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ServerCapabilitiesType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (MaxArrayLength != null)
            {
                MaxArrayLength.Initialize(context, MaxArrayLength_InitializationString);
            }

            if (MaxStringLength != null)
            {
                MaxStringLength.Initialize(context, MaxStringLength_InitializationString);
            }

            if (MaxByteStringLength != null)
            {
                MaxByteStringLength.Initialize(context, MaxByteStringLength_InitializationString);
            }

            if (OperationLimits != null)
            {
                OperationLimits.Initialize(context, OperationLimits_InitializationString);
            }

            if (RoleSet != null)
            {
                RoleSet.Initialize(context, RoleSet_InitializationString);
            }

            if (MaxSessions != null)
            {
                MaxSessions.Initialize(context, MaxSessions_InitializationString);
            }

            if (MaxSubscriptions != null)
            {
                MaxSubscriptions.Initialize(context, MaxSubscriptions_InitializationString);
            }

            if (MaxMonitoredItems != null)
            {
                MaxMonitoredItems.Initialize(context, MaxMonitoredItems_InitializationString);
            }

            if (MaxSubscriptionsPerSession != null)
            {
                MaxSubscriptionsPerSession.Initialize(context, MaxSubscriptionsPerSession_InitializationString);
            }

            if (MaxMonitoredItemsPerSubscription != null)
            {
                MaxMonitoredItemsPerSubscription.Initialize(context, MaxMonitoredItemsPerSubscription_InitializationString);
            }

            if (MaxSelectClauseParameters != null)
            {
                MaxSelectClauseParameters.Initialize(context, MaxSelectClauseParameters_InitializationString);
            }

            if (MaxWhereClauseParameters != null)
            {
                MaxWhereClauseParameters.Initialize(context, MaxWhereClauseParameters_InitializationString);
            }

            if (MaxMonitoredItemsQueueSize != null)
            {
                MaxMonitoredItemsQueueSize.Initialize(context, MaxMonitoredItemsQueueSize_InitializationString);
            }

            if (ConformanceUnits != null)
            {
                ConformanceUnits.Initialize(context, ConformanceUnits_InitializationString);
            }
        }

        #region Initialization String
        private const string MaxArrayLength_InitializationString =
           "//////////8VYIkKAgAAAAAADgAAAE1heEFycmF5TGVuZ3RoAQAdLQAuAEQdLQAAAAf/////AQH/////" +
           "AAAAAA==";

        private const string MaxStringLength_InitializationString =
           "//////////8VYIkKAgAAAAAADwAAAE1heFN0cmluZ0xlbmd0aAEAHi0ALgBEHi0AAAAH/////wEB////" +
           "/wAAAAA=";

        private const string MaxByteStringLength_InitializationString =
           "//////////8VYIkKAgAAAAAAEwAAAE1heEJ5dGVTdHJpbmdMZW5ndGgBAG4yAC4ARG4yAAAAB/////8B" +
           "Af////8AAAAA";

        private const string OperationLimits_InitializationString =
           "//////////8EYIAKAQAAAAAADwAAAE9wZXJhdGlvbkxpbWl0cwEAHy0ALwEALC0fLQAA/////wAAAAA=";

        private const string RoleSet_InitializationString =
           "//////////8EYIAKAQAAAAAABwAAAFJvbGVTZXQBAKc/AC8BAPc8pz8AAP////8CAAAABGGCCgQAAAAA" +
           "AAcAAABBZGRSb2xlAQCoPwAvAQB9Pqg/AAABAf////8CAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3Vt" +
           "ZW50cwEAqT8ALgBEqT8AAJYCAAAAAQAqAQEXAAAACAAAAFJvbGVOYW1lAAz/////AAAAAAABACoBARsA" +
           "AAAMAAAATmFtZXNwYWNlVXJpAAz/////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAABdgqQoC" +
           "AAAAAAAPAAAAT3V0cHV0QXJndW1lbnRzAQCqPwAuAESqPwAAlgEAAAABACoBARkAAAAKAAAAUm9sZU5v" +
           "ZGVJZAAR/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAAAAAACgAAAFJlbW92" +
           "ZVJvbGUBAKs/AC8BAIA+qz8AAAEB/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQCs" +
           "PwAuAESsPwAAlgEAAAABACoBARkAAAAKAAAAUm9sZU5vZGVJZAAR/////wAAAAAAAQAoAQEAAAABAAAA" +
           "AQAAAAEB/////wAAAAA=";

        private const string MaxSessions_InitializationString =
           "//////////8VYIkKAgAAAAAACwAAAE1heFNlc3Npb25zAQAYXgAuAEQYXgAAAAf/////AQH/////AAAA" +
           "AA==";

        private const string MaxSubscriptions_InitializationString =
           "//////////8VYIkKAgAAAAAAEAAAAE1heFN1YnNjcmlwdGlvbnMBABleAC4ARBleAAAAB/////8BAf//" +
           "//8AAAAA";

        private const string MaxMonitoredItems_InitializationString =
           "//////////8VYIkKAgAAAAAAEQAAAE1heE1vbml0b3JlZEl0ZW1zAQAaXgAuAEQaXgAAAAf/////AQH/" +
           "////AAAAAA==";

        private const string MaxSubscriptionsPerSession_InitializationString =
           "//////////8VYIkKAgAAAAAAGgAAAE1heFN1YnNjcmlwdGlvbnNQZXJTZXNzaW9uAQAbXgAuAEQbXgAA" +
           "AAf/////AQH/////AAAAAA==";

        private const string MaxMonitoredItemsPerSubscription_InitializationString =
           "//////////8VYIkKAgAAAAAAIAAAAE1heE1vbml0b3JlZEl0ZW1zUGVyU3Vic2NyaXB0aW9uAQAnXgAu" +
           "AEQnXgAAAAf/////AQH/////AAAAAA==";

        private const string MaxSelectClauseParameters_InitializationString =
           "//////////8VYIkKAgAAAAAAGQAAAE1heFNlbGVjdENsYXVzZVBhcmFtZXRlcnMBABxeAC4ARBxeAAAA" +
           "B/////8BAf////8AAAAA";

        private const string MaxWhereClauseParameters_InitializationString =
           "//////////8VYIkKAgAAAAAAGAAAAE1heFdoZXJlQ2xhdXNlUGFyYW1ldGVycwEAHV4ALgBEHV4AAAAH" +
           "/////wEB/////wAAAAA=";

        private const string MaxMonitoredItemsQueueSize_InitializationString =
           "//////////8VYIkKAgAAAAAAGgAAAE1heE1vbml0b3JlZEl0ZW1zUXVldWVTaXplAQAafAAuAEQafAAA" +
           "AAf/////AQH/////AAAAAA==";

        private const string ConformanceUnits_InitializationString =
           "//////////8XYIkKAgAAAAAAEAAAAENvbmZvcm1hbmNlVW5pdHMBAB5eAC4ARB5eAAAAFAEAAAABAAAA" +
           "AAAAAAEB/////wAAAAA=";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHgAAAFNlcnZlckNhcGFiaWxpdGllc1R5cGVJbnN0YW5jZQEA3QcBAN0H" +
           "3QcAAP////8YAAAAF2CJCgIAAAAAABIAAABTZXJ2ZXJQcm9maWxlQXJyYXkBAN4HAC4ARN4HAAAADAEA" +
           "AAABAAAAAAAAAAEB/////wAAAAAXYIkKAgAAAAAADQAAAExvY2FsZUlkQXJyYXkBAOAHAC4AROAHAAAB" +
           "ACcBAQAAAAEAAAAAAAAAAQH/////AAAAABVgiQoCAAAAAAAWAAAATWluU3VwcG9ydGVkU2FtcGxlUmF0" +
           "ZQEA4QcALgBE4QcAAAEAIgH/////AQH/////AAAAABVgiQoCAAAAAAAbAAAATWF4QnJvd3NlQ29udGlu" +
           "dWF0aW9uUG9pbnRzAQCsCgAuAESsCgAAAAX/////AQH/////AAAAABVgiQoCAAAAAAAaAAAATWF4UXVl" +
           "cnlDb250aW51YXRpb25Qb2ludHMBAK0KAC4ARK0KAAAABf////8BAf////8AAAAAFWCJCgIAAAAAABwA" +
           "AABNYXhIaXN0b3J5Q29udGludWF0aW9uUG9pbnRzAQCuCgAuAESuCgAAAAX/////AQH/////AAAAABdg" +
           "iQoCAAAAAAAUAAAAU29mdHdhcmVDZXJ0aWZpY2F0ZXMBAOkLAC4AROkLAAABAFgBAQAAAAEAAAAAAAAA" +
           "AQH/////AAAAABVgiQoCAAAAAAAOAAAATWF4QXJyYXlMZW5ndGgBAB0tAC4ARB0tAAAAB/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAA8AAABNYXhTdHJpbmdMZW5ndGgBAB4tAC4ARB4tAAAAB/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAABMAAABNYXhCeXRlU3RyaW5nTGVuZ3RoAQBuMgAuAERuMgAAAAf/////AQH/////" +
           "AAAAAARggAoBAAAAAAAPAAAAT3BlcmF0aW9uTGltaXRzAQAfLQAvAQAsLR8tAAD/////AAAAAARggAoB" +
           "AAAAAAAOAAAATW9kZWxsaW5nUnVsZXMBAOMHAC8APeMHAAD/////AAAAAARggAoBAAAAAAASAAAAQWdn" +
           "cmVnYXRlRnVuY3Rpb25zAQDCCgAvAD3CCgAA/////wAAAAAVYMkKAgAAABwAAABWZW5kb3JDYXBhYmls" +
           "aXR5X1BsYWNlaG9sZGVyAAASAAAAPFZlbmRvckNhcGFiaWxpdHk+AQAqLQAvAQBZCCotAAAAGP////8B" +
           "Af////8AAAAABGCACgEAAAAAAAcAAABSb2xlU2V0AQCnPwAvAQD3PKc/AAD/////AgAAAARhggoEAAAA" +
           "AAAHAAAAQWRkUm9sZQEAqD8ALwEAfT6oPwAAAQH/////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1" +
           "bWVudHMBAKk/AC4ARKk/AACWAgAAAAEAKgEBFwAAAAgAAABSb2xlTmFtZQAM/////wAAAAAAAQAqAQEb" +
           "AAAADAAAAE5hbWVzcGFjZVVyaQAM/////wAAAAAAAQAoAQEAAAABAAAAAgAAAAEB/////wAAAAAXYKkK" +
           "AgAAAAAADwAAAE91dHB1dEFyZ3VtZW50cwEAqj8ALgBEqj8AAJYBAAAAAQAqAQEZAAAACgAAAFJvbGVO" +
           "b2RlSWQAEf////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAABGGCCgQAAAAAAAoAAABSZW1v" +
           "dmVSb2xlAQCrPwAvAQCAPqs/AAABAf////8BAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEA" +
           "rD8ALgBErD8AAJYBAAAAAQAqAQEZAAAACgAAAFJvbGVOb2RlSWQAEf////8AAAAAAAEAKAEBAAAAAQAA" +
           "AAEAAAABAf////8AAAAAFWCJCgIAAAAAAAsAAABNYXhTZXNzaW9ucwEAGF4ALgBEGF4AAAAH/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAAEAAAAE1heFN1YnNjcmlwdGlvbnMBABleAC4ARBleAAAAB/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABEAAABNYXhNb25pdG9yZWRJdGVtcwEAGl4ALgBEGl4AAAAH/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAGgAAAE1heFN1YnNjcmlwdGlvbnNQZXJTZXNzaW9uAQAbXgAuAEQbXgAAAAf/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAgAAAATWF4TW9uaXRvcmVkSXRlbXNQZXJTdWJzY3JpcHRpb24B" +
           "ACdeAC4ARCdeAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABkAAABNYXhTZWxlY3RDbGF1c2VQYXJh" +
           "bWV0ZXJzAQAcXgAuAEQcXgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAYAAAATWF4V2hlcmVDbGF1" +
           "c2VQYXJhbWV0ZXJzAQAdXgAuAEQdXgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAaAAAATWF4TW9u" +
           "aXRvcmVkSXRlbXNRdWV1ZVNpemUBABp8AC4ARBp8AAAAB/////8BAf////8AAAAAF2CJCgIAAAAAABAA" +
           "AABDb25mb3JtYW5jZVVuaXRzAQAeXgAuAEQeXgAAABQBAAAAAQAAAAAAAAABAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string[]> ServerProfileArray
        {
            get
            {
                return m_serverProfileArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverProfileArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverProfileArray = value;
            }
        }

        /// <remarks />
        public PropertyState<string[]> LocaleIdArray
        {
            get
            {
                return m_localeIdArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_localeIdArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_localeIdArray = value;
            }
        }

        /// <remarks />
        public PropertyState<double> MinSupportedSampleRate
        {
            get
            {
                return m_minSupportedSampleRate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_minSupportedSampleRate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_minSupportedSampleRate = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> MaxBrowseContinuationPoints
        {
            get
            {
                return m_maxBrowseContinuationPoints;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxBrowseContinuationPoints, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxBrowseContinuationPoints = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> MaxQueryContinuationPoints
        {
            get
            {
                return m_maxQueryContinuationPoints;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxQueryContinuationPoints, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxQueryContinuationPoints = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> MaxHistoryContinuationPoints
        {
            get
            {
                return m_maxHistoryContinuationPoints;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxHistoryContinuationPoints, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxHistoryContinuationPoints = value;
            }
        }

        /// <remarks />
        public PropertyState<SignedSoftwareCertificate[]> SoftwareCertificates
        {
            get
            {
                return m_softwareCertificates;
            }

            set
            {
                if (!Object.ReferenceEquals(m_softwareCertificates, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_softwareCertificates = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxArrayLength
        {
            get
            {
                return m_maxArrayLength;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxArrayLength, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxArrayLength = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxStringLength
        {
            get
            {
                return m_maxStringLength;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxStringLength, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxStringLength = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxByteStringLength
        {
            get
            {
                return m_maxByteStringLength;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxByteStringLength, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxByteStringLength = value;
            }
        }

        /// <remarks />
        public OperationLimitsState OperationLimits
        {
            get
            {
                return m_operationLimits;
            }

            set
            {
                if (!Object.ReferenceEquals(m_operationLimits, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_operationLimits = value;
            }
        }

        /// <remarks />
        public FolderState ModellingRules
        {
            get
            {
                return m_modellingRules;
            }

            set
            {
                if (!Object.ReferenceEquals(m_modellingRules, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_modellingRules = value;
            }
        }

        /// <remarks />
        public FolderState AggregateFunctions
        {
            get
            {
                return m_aggregateFunctions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_aggregateFunctions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_aggregateFunctions = value;
            }
        }

        /// <remarks />
        public RoleSetState RoleSet
        {
            get
            {
                return m_roleSet;
            }

            set
            {
                if (!Object.ReferenceEquals(m_roleSet, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_roleSet = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxSessions
        {
            get
            {
                return m_maxSessions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxSessions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxSessions = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxSubscriptions
        {
            get
            {
                return m_maxSubscriptions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxSubscriptions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxSubscriptions = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxMonitoredItems
        {
            get
            {
                return m_maxMonitoredItems;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxMonitoredItems, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxMonitoredItems = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxSubscriptionsPerSession
        {
            get
            {
                return m_maxSubscriptionsPerSession;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxSubscriptionsPerSession, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxSubscriptionsPerSession = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxMonitoredItemsPerSubscription
        {
            get
            {
                return m_maxMonitoredItemsPerSubscription;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxMonitoredItemsPerSubscription, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxMonitoredItemsPerSubscription = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxSelectClauseParameters
        {
            get
            {
                return m_maxSelectClauseParameters;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxSelectClauseParameters, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxSelectClauseParameters = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxWhereClauseParameters
        {
            get
            {
                return m_maxWhereClauseParameters;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxWhereClauseParameters, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxWhereClauseParameters = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxMonitoredItemsQueueSize
        {
            get
            {
                return m_maxMonitoredItemsQueueSize;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxMonitoredItemsQueueSize, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxMonitoredItemsQueueSize = value;
            }
        }

        /// <remarks />
        public PropertyState<QualifiedName[]> ConformanceUnits
        {
            get
            {
                return m_conformanceUnits;
            }

            set
            {
                if (!Object.ReferenceEquals(m_conformanceUnits, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_conformanceUnits = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverProfileArray != null)
            {
                children.Add(m_serverProfileArray);
            }

            if (m_localeIdArray != null)
            {
                children.Add(m_localeIdArray);
            }

            if (m_minSupportedSampleRate != null)
            {
                children.Add(m_minSupportedSampleRate);
            }

            if (m_maxBrowseContinuationPoints != null)
            {
                children.Add(m_maxBrowseContinuationPoints);
            }

            if (m_maxQueryContinuationPoints != null)
            {
                children.Add(m_maxQueryContinuationPoints);
            }

            if (m_maxHistoryContinuationPoints != null)
            {
                children.Add(m_maxHistoryContinuationPoints);
            }

            if (m_softwareCertificates != null)
            {
                children.Add(m_softwareCertificates);
            }

            if (m_maxArrayLength != null)
            {
                children.Add(m_maxArrayLength);
            }

            if (m_maxStringLength != null)
            {
                children.Add(m_maxStringLength);
            }

            if (m_maxByteStringLength != null)
            {
                children.Add(m_maxByteStringLength);
            }

            if (m_operationLimits != null)
            {
                children.Add(m_operationLimits);
            }

            if (m_modellingRules != null)
            {
                children.Add(m_modellingRules);
            }

            if (m_aggregateFunctions != null)
            {
                children.Add(m_aggregateFunctions);
            }

            if (m_roleSet != null)
            {
                children.Add(m_roleSet);
            }

            if (m_maxSessions != null)
            {
                children.Add(m_maxSessions);
            }

            if (m_maxSubscriptions != null)
            {
                children.Add(m_maxSubscriptions);
            }

            if (m_maxMonitoredItems != null)
            {
                children.Add(m_maxMonitoredItems);
            }

            if (m_maxSubscriptionsPerSession != null)
            {
                children.Add(m_maxSubscriptionsPerSession);
            }

            if (m_maxMonitoredItemsPerSubscription != null)
            {
                children.Add(m_maxMonitoredItemsPerSubscription);
            }

            if (m_maxSelectClauseParameters != null)
            {
                children.Add(m_maxSelectClauseParameters);
            }

            if (m_maxWhereClauseParameters != null)
            {
                children.Add(m_maxWhereClauseParameters);
            }

            if (m_maxMonitoredItemsQueueSize != null)
            {
                children.Add(m_maxMonitoredItemsQueueSize);
            }

            if (m_conformanceUnits != null)
            {
                children.Add(m_conformanceUnits);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerProfileArray:
                {
                    if (createOrReplace)
                    {
                        if (ServerProfileArray == null)
                        {
                            if (replacement == null)
                            {
                                ServerProfileArray = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                ServerProfileArray = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = ServerProfileArray;
                    break;
                }

                case Opc.Ua.BrowseNames.LocaleIdArray:
                {
                    if (createOrReplace)
                    {
                        if (LocaleIdArray == null)
                        {
                            if (replacement == null)
                            {
                                LocaleIdArray = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                LocaleIdArray = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = LocaleIdArray;
                    break;
                }

                case Opc.Ua.BrowseNames.MinSupportedSampleRate:
                {
                    if (createOrReplace)
                    {
                        if (MinSupportedSampleRate == null)
                        {
                            if (replacement == null)
                            {
                                MinSupportedSampleRate = new PropertyState<double>(this);
                            }
                            else
                            {
                                MinSupportedSampleRate = (PropertyState<double>)replacement;
                            }
                        }
                    }

                    instance = MinSupportedSampleRate;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxBrowseContinuationPoints:
                {
                    if (createOrReplace)
                    {
                        if (MaxBrowseContinuationPoints == null)
                        {
                            if (replacement == null)
                            {
                                MaxBrowseContinuationPoints = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                MaxBrowseContinuationPoints = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = MaxBrowseContinuationPoints;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxQueryContinuationPoints:
                {
                    if (createOrReplace)
                    {
                        if (MaxQueryContinuationPoints == null)
                        {
                            if (replacement == null)
                            {
                                MaxQueryContinuationPoints = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                MaxQueryContinuationPoints = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = MaxQueryContinuationPoints;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxHistoryContinuationPoints:
                {
                    if (createOrReplace)
                    {
                        if (MaxHistoryContinuationPoints == null)
                        {
                            if (replacement == null)
                            {
                                MaxHistoryContinuationPoints = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                MaxHistoryContinuationPoints = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = MaxHistoryContinuationPoints;
                    break;
                }

                case Opc.Ua.BrowseNames.SoftwareCertificates:
                {
                    if (createOrReplace)
                    {
                        if (SoftwareCertificates == null)
                        {
                            if (replacement == null)
                            {
                                SoftwareCertificates = new PropertyState<SignedSoftwareCertificate[]>(this);
                            }
                            else
                            {
                                SoftwareCertificates = (PropertyState<SignedSoftwareCertificate[]>)replacement;
                            }
                        }
                    }

                    instance = SoftwareCertificates;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxArrayLength:
                {
                    if (createOrReplace)
                    {
                        if (MaxArrayLength == null)
                        {
                            if (replacement == null)
                            {
                                MaxArrayLength = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxArrayLength = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxArrayLength;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxStringLength:
                {
                    if (createOrReplace)
                    {
                        if (MaxStringLength == null)
                        {
                            if (replacement == null)
                            {
                                MaxStringLength = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxStringLength = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxStringLength;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxByteStringLength:
                {
                    if (createOrReplace)
                    {
                        if (MaxByteStringLength == null)
                        {
                            if (replacement == null)
                            {
                                MaxByteStringLength = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxByteStringLength = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxByteStringLength;
                    break;
                }

                case Opc.Ua.BrowseNames.OperationLimits:
                {
                    if (createOrReplace)
                    {
                        if (OperationLimits == null)
                        {
                            if (replacement == null)
                            {
                                OperationLimits = new OperationLimitsState(this);
                            }
                            else
                            {
                                OperationLimits = (OperationLimitsState)replacement;
                            }
                        }
                    }

                    instance = OperationLimits;
                    break;
                }

                case Opc.Ua.BrowseNames.ModellingRules:
                {
                    if (createOrReplace)
                    {
                        if (ModellingRules == null)
                        {
                            if (replacement == null)
                            {
                                ModellingRules = new FolderState(this);
                            }
                            else
                            {
                                ModellingRules = (FolderState)replacement;
                            }
                        }
                    }

                    instance = ModellingRules;
                    break;
                }

                case Opc.Ua.BrowseNames.AggregateFunctions:
                {
                    if (createOrReplace)
                    {
                        if (AggregateFunctions == null)
                        {
                            if (replacement == null)
                            {
                                AggregateFunctions = new FolderState(this);
                            }
                            else
                            {
                                AggregateFunctions = (FolderState)replacement;
                            }
                        }
                    }

                    instance = AggregateFunctions;
                    break;
                }

                case Opc.Ua.BrowseNames.RoleSet:
                {
                    if (createOrReplace)
                    {
                        if (RoleSet == null)
                        {
                            if (replacement == null)
                            {
                                RoleSet = new RoleSetState(this);
                            }
                            else
                            {
                                RoleSet = (RoleSetState)replacement;
                            }
                        }
                    }

                    instance = RoleSet;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxSessions:
                {
                    if (createOrReplace)
                    {
                        if (MaxSessions == null)
                        {
                            if (replacement == null)
                            {
                                MaxSessions = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxSessions = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxSessions;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxSubscriptions:
                {
                    if (createOrReplace)
                    {
                        if (MaxSubscriptions == null)
                        {
                            if (replacement == null)
                            {
                                MaxSubscriptions = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxSubscriptions = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxSubscriptions;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxMonitoredItems:
                {
                    if (createOrReplace)
                    {
                        if (MaxMonitoredItems == null)
                        {
                            if (replacement == null)
                            {
                                MaxMonitoredItems = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxMonitoredItems = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxMonitoredItems;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxSubscriptionsPerSession:
                {
                    if (createOrReplace)
                    {
                        if (MaxSubscriptionsPerSession == null)
                        {
                            if (replacement == null)
                            {
                                MaxSubscriptionsPerSession = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxSubscriptionsPerSession = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxSubscriptionsPerSession;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxMonitoredItemsPerSubscription:
                {
                    if (createOrReplace)
                    {
                        if (MaxMonitoredItemsPerSubscription == null)
                        {
                            if (replacement == null)
                            {
                                MaxMonitoredItemsPerSubscription = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxMonitoredItemsPerSubscription = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxMonitoredItemsPerSubscription;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxSelectClauseParameters:
                {
                    if (createOrReplace)
                    {
                        if (MaxSelectClauseParameters == null)
                        {
                            if (replacement == null)
                            {
                                MaxSelectClauseParameters = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxSelectClauseParameters = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxSelectClauseParameters;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxWhereClauseParameters:
                {
                    if (createOrReplace)
                    {
                        if (MaxWhereClauseParameters == null)
                        {
                            if (replacement == null)
                            {
                                MaxWhereClauseParameters = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxWhereClauseParameters = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxWhereClauseParameters;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxMonitoredItemsQueueSize:
                {
                    if (createOrReplace)
                    {
                        if (MaxMonitoredItemsQueueSize == null)
                        {
                            if (replacement == null)
                            {
                                MaxMonitoredItemsQueueSize = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxMonitoredItemsQueueSize = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxMonitoredItemsQueueSize;
                    break;
                }

                case Opc.Ua.BrowseNames.ConformanceUnits:
                {
                    if (createOrReplace)
                    {
                        if (ConformanceUnits == null)
                        {
                            if (replacement == null)
                            {
                                ConformanceUnits = new PropertyState<QualifiedName[]>(this);
                            }
                            else
                            {
                                ConformanceUnits = (PropertyState<QualifiedName[]>)replacement;
                            }
                        }
                    }

                    instance = ConformanceUnits;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string[]> m_serverProfileArray;
        private PropertyState<string[]> m_localeIdArray;
        private PropertyState<double> m_minSupportedSampleRate;
        private PropertyState<ushort> m_maxBrowseContinuationPoints;
        private PropertyState<ushort> m_maxQueryContinuationPoints;
        private PropertyState<ushort> m_maxHistoryContinuationPoints;
        private PropertyState<SignedSoftwareCertificate[]> m_softwareCertificates;
        private PropertyState<uint> m_maxArrayLength;
        private PropertyState<uint> m_maxStringLength;
        private PropertyState<uint> m_maxByteStringLength;
        private OperationLimitsState m_operationLimits;
        private FolderState m_modellingRules;
        private FolderState m_aggregateFunctions;
        private RoleSetState m_roleSet;
        private PropertyState<uint> m_maxSessions;
        private PropertyState<uint> m_maxSubscriptions;
        private PropertyState<uint> m_maxMonitoredItems;
        private PropertyState<uint> m_maxSubscriptionsPerSession;
        private PropertyState<uint> m_maxMonitoredItemsPerSubscription;
        private PropertyState<uint> m_maxSelectClauseParameters;
        private PropertyState<uint> m_maxWhereClauseParameters;
        private PropertyState<uint> m_maxMonitoredItemsQueueSize;
        private PropertyState<QualifiedName[]> m_conformanceUnits;
        #endregion
    }
    #endif
    #endregion

    #region ServerDiagnosticsState Class
    #if (!OPCUA_EXCLUDE_ServerDiagnosticsState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerDiagnosticsState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public ServerDiagnosticsState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ServerDiagnosticsType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (SamplingIntervalDiagnosticsArray != null)
            {
                SamplingIntervalDiagnosticsArray.Initialize(context, SamplingIntervalDiagnosticsArray_InitializationString);
            }
        }

        #region Initialization String
        private const string SamplingIntervalDiagnosticsArray_InitializationString =
           "//////////8XYIkKAgAAAAAAIAAAAFNhbXBsaW5nSW50ZXJ2YWxEaWFnbm9zdGljc0FycmF5AQDmBwAv" +
           "AQB0COYHAAABAFgDAQAAAAEAAAAAAAAAAQH/////AAAAAA==";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHQAAAFNlcnZlckRpYWdub3N0aWNzVHlwZUluc3RhbmNlAQDkBwEA5Afk" +
           "BwAA/////wUAAAAVYIkKAgAAAAAAGAAAAFNlcnZlckRpYWdub3N0aWNzU3VtbWFyeQEA5QcALwEAZgjl" +
           "BwAAAQBbA/////8BAf////8MAAAAFWCJCgIAAAAAAA8AAABTZXJ2ZXJWaWV3Q291bnQBACwMAC8APywM" +
           "AAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABMAAABDdXJyZW50U2Vzc2lvbkNvdW50AQAtDAAvAD8t" +
           "DAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAVAAAAQ3VtdWxhdGVkU2Vzc2lvbkNvdW50AQAuDAAv" +
           "AD8uDAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAcAAAAU2VjdXJpdHlSZWplY3RlZFNlc3Npb25D" +
           "b3VudAEALwwALwA/LwwAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFAAAAFJlamVjdGVkU2Vzc2lv" +
           "bkNvdW50AQAwDAAvAD8wDAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAATAAAAU2Vzc2lvblRpbWVv" +
           "dXRDb3VudAEAMQwALwA/MQwAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAFNlc3Npb25BYm9y" +
           "dENvdW50AQAyDAAvAD8yDAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAXAAAAUHVibGlzaGluZ0lu" +
           "dGVydmFsQ291bnQBADQMAC8APzQMAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABgAAABDdXJyZW50" +
           "U3Vic2NyaXB0aW9uQ291bnQBADUMAC8APzUMAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABD" +
           "dW11bGF0ZWRTdWJzY3JpcHRpb25Db3VudAEANgwALwA/NgwAAAAH/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAHQAAAFNlY3VyaXR5UmVqZWN0ZWRSZXF1ZXN0c0NvdW50AQA3DAAvAD83DAAAAAf/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAVAAAAUmVqZWN0ZWRSZXF1ZXN0c0NvdW50AQA4DAAvAD84DAAAAAf/////AQH/" +
           "////AAAAABdgiQoCAAAAAAAgAAAAU2FtcGxpbmdJbnRlcnZhbERpYWdub3N0aWNzQXJyYXkBAOYHAC8B" +
           "AHQI5gcAAAEAWAMBAAAAAQAAAAAAAAABAf////8AAAAAF2CJCgIAAAAAABwAAABTdWJzY3JpcHRpb25E" +
           "aWFnbm9zdGljc0FycmF5AQDnBwAvAQB7COcHAAABAGoDAQAAAAEAAAAAAAAAAQH/////AAAAAARggAoB" +
           "AAAAAAAaAAAAU2Vzc2lvbnNEaWFnbm9zdGljc1N1bW1hcnkBALgKAC8BAOoHuAoAAP////8CAAAAF2CJ" +
           "CgIAAAAAABcAAABTZXNzaW9uRGlhZ25vc3RpY3NBcnJheQEAOQwALwEAlAg5DAAAAQBhAwEAAAABAAAA" +
           "AAAAAAEB/////wAAAAAXYIkKAgAAAAAAHwAAAFNlc3Npb25TZWN1cml0eURpYWdub3N0aWNzQXJyYXkB" +
           "ADoMAC8BAMMIOgwAAAEAZAMBAAAAAQAAAAAAAAABAf////8AAAAAFWCJCgIAAAAAAAsAAABFbmFibGVk" +
           "RmxhZwEA6QcALgBE6QcAAAAB/////wMD/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public ServerDiagnosticsSummaryState ServerDiagnosticsSummary
        {
            get
            {
                return m_serverDiagnosticsSummary;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverDiagnosticsSummary, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverDiagnosticsSummary = value;
            }
        }

        /// <remarks />
        public SamplingIntervalDiagnosticsArrayState SamplingIntervalDiagnosticsArray
        {
            get
            {
                return m_samplingIntervalDiagnosticsArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_samplingIntervalDiagnosticsArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_samplingIntervalDiagnosticsArray = value;
            }
        }

        /// <remarks />
        public SubscriptionDiagnosticsArrayState SubscriptionDiagnosticsArray
        {
            get
            {
                return m_subscriptionDiagnosticsArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_subscriptionDiagnosticsArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_subscriptionDiagnosticsArray = value;
            }
        }

        /// <remarks />
        public SessionsDiagnosticsSummaryState SessionsDiagnosticsSummary
        {
            get
            {
                return m_sessionsDiagnosticsSummary;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionsDiagnosticsSummary, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionsDiagnosticsSummary = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> EnabledFlag
        {
            get
            {
                return m_enabledFlag;
            }

            set
            {
                if (!Object.ReferenceEquals(m_enabledFlag, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_enabledFlag = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverDiagnosticsSummary != null)
            {
                children.Add(m_serverDiagnosticsSummary);
            }

            if (m_samplingIntervalDiagnosticsArray != null)
            {
                children.Add(m_samplingIntervalDiagnosticsArray);
            }

            if (m_subscriptionDiagnosticsArray != null)
            {
                children.Add(m_subscriptionDiagnosticsArray);
            }

            if (m_sessionsDiagnosticsSummary != null)
            {
                children.Add(m_sessionsDiagnosticsSummary);
            }

            if (m_enabledFlag != null)
            {
                children.Add(m_enabledFlag);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerDiagnosticsSummary:
                {
                    if (createOrReplace)
                    {
                        if (ServerDiagnosticsSummary == null)
                        {
                            if (replacement == null)
                            {
                                ServerDiagnosticsSummary = new ServerDiagnosticsSummaryState(this);
                            }
                            else
                            {
                                ServerDiagnosticsSummary = (ServerDiagnosticsSummaryState)replacement;
                            }
                        }
                    }

                    instance = ServerDiagnosticsSummary;
                    break;
                }

                case Opc.Ua.BrowseNames.SamplingIntervalDiagnosticsArray:
                {
                    if (createOrReplace)
                    {
                        if (SamplingIntervalDiagnosticsArray == null)
                        {
                            if (replacement == null)
                            {
                                SamplingIntervalDiagnosticsArray = new SamplingIntervalDiagnosticsArrayState(this);
                            }
                            else
                            {
                                SamplingIntervalDiagnosticsArray = (SamplingIntervalDiagnosticsArrayState)replacement;
                            }
                        }
                    }

                    instance = SamplingIntervalDiagnosticsArray;
                    break;
                }

                case Opc.Ua.BrowseNames.SubscriptionDiagnosticsArray:
                {
                    if (createOrReplace)
                    {
                        if (SubscriptionDiagnosticsArray == null)
                        {
                            if (replacement == null)
                            {
                                SubscriptionDiagnosticsArray = new SubscriptionDiagnosticsArrayState(this);
                            }
                            else
                            {
                                SubscriptionDiagnosticsArray = (SubscriptionDiagnosticsArrayState)replacement;
                            }
                        }
                    }

                    instance = SubscriptionDiagnosticsArray;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionsDiagnosticsSummary:
                {
                    if (createOrReplace)
                    {
                        if (SessionsDiagnosticsSummary == null)
                        {
                            if (replacement == null)
                            {
                                SessionsDiagnosticsSummary = new SessionsDiagnosticsSummaryState(this);
                            }
                            else
                            {
                                SessionsDiagnosticsSummary = (SessionsDiagnosticsSummaryState)replacement;
                            }
                        }
                    }

                    instance = SessionsDiagnosticsSummary;
                    break;
                }

                case Opc.Ua.BrowseNames.EnabledFlag:
                {
                    if (createOrReplace)
                    {
                        if (EnabledFlag == null)
                        {
                            if (replacement == null)
                            {
                                EnabledFlag = new PropertyState<bool>(this);
                            }
                            else
                            {
                                EnabledFlag = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = EnabledFlag;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private ServerDiagnosticsSummaryState m_serverDiagnosticsSummary;
        private SamplingIntervalDiagnosticsArrayState m_samplingIntervalDiagnosticsArray;
        private SubscriptionDiagnosticsArrayState m_subscriptionDiagnosticsArray;
        private SessionsDiagnosticsSummaryState m_sessionsDiagnosticsSummary;
        private PropertyState<bool> m_enabledFlag;
        #endregion
    }
    #endif
    #endregion

    #region SessionsDiagnosticsSummaryState Class
    #if (!OPCUA_EXCLUDE_SessionsDiagnosticsSummaryState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionsDiagnosticsSummaryState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public SessionsDiagnosticsSummaryState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.SessionsDiagnosticsSummaryType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJgAAAFNlc3Npb25zRGlhZ25vc3RpY3NTdW1tYXJ5VHlwZUluc3RhbmNl" +
           "AQDqBwEA6gfqBwAA/////wMAAAAXYIkKAgAAAAAAFwAAAFNlc3Npb25EaWFnbm9zdGljc0FycmF5AQDr" +
           "BwAvAQCUCOsHAAABAGEDAQAAAAEAAAAAAAAAAQH/////AAAAABdgiQoCAAAAAAAfAAAAU2Vzc2lvblNl" +
           "Y3VyaXR5RGlhZ25vc3RpY3NBcnJheQEA7AcALwEAwwjsBwAAAQBkAwEAAAABAAAAAAAAAAEB/////wAA" +
           "AAAEYMAKAQAAABYAAABDbGllbnROYW1lX1BsYWNlaG9sZGVyAAAMAAAAPENsaWVudE5hbWU+AQBBLwAv" +
           "AQDtB0EvAAD/////AwAAABVgiQoCAAAAAAASAAAAU2Vzc2lvbkRpYWdub3N0aWNzAQBCLwAvAQCVCEIv" +
           "AAABAGED/////wEB/////ysAAAAVYIkKAgAAAAAACQAAAFNlc3Npb25JZAEAQy8ALwA/Qy8AAAAR////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACwAAAFNlc3Npb25OYW1lAQBELwAvAD9ELwAAAAz/////AQH/////" +
           "AAAAABVgiQoCAAAAAAARAAAAQ2xpZW50RGVzY3JpcHRpb24BAEUvAC8AP0UvAAABADQB/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAACQAAAFNlcnZlclVyaQEARi8ALwA/Ri8AAAAM/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACwAAAEVuZHBvaW50VXJsAQBHLwAvAD9HLwAAAAz/////AQH/////AAAAABdgiQoCAAAAAAAJ" +
           "AAAATG9jYWxlSWRzAQBILwAvAD9ILwAAAQAnAQEAAAABAAAAAAAAAAEB/////wAAAAAVYIkKAgAAAAAA" +
           "FAAAAEFjdHVhbFNlc3Npb25UaW1lb3V0AQBJLwAvAD9JLwAAAQAiAf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAABYAAABNYXhSZXNwb25zZU1lc3NhZ2VTaXplAQBKLwAvAD9KLwAAAAf/////AQH/////AAAAABVg" +
           "iQoCAAAAAAAUAAAAQ2xpZW50Q29ubmVjdGlvblRpbWUBAEsvAC8AP0svAAABACYB/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAAFQAAAENsaWVudExhc3RDb250YWN0VGltZQEATC8ALwA/TC8AAAEAJgH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAZAAAAQ3VycmVudFN1YnNjcmlwdGlvbnNDb3VudAEATS8ALwA/TS8AAAAH" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAAGgAAAEN1cnJlbnRNb25pdG9yZWRJdGVtc0NvdW50AQBOLwAv" +
           "AD9OLwAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAdAAAAQ3VycmVudFB1Ymxpc2hSZXF1ZXN0c0lu" +
           "UXVldWUBAE8vAC8AP08vAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABEAAABUb3RhbFJlcXVlc3RD" +
           "b3VudAEAUC8ALwA/UC8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAYAAAAVW5hdXRob3JpemVk" +
           "UmVxdWVzdENvdW50AQBRLwAvAD9RLwAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAUmVhZENv" +
           "dW50AQBSLwAvAD9SLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABAAAABIaXN0b3J5UmVhZENv" +
           "dW50AQBTLwAvAD9TLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABXcml0ZUNvdW50AQBU" +
           "LwAvAD9ULwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABIAAABIaXN0b3J5VXBkYXRlQ291bnQB" +
           "AFUvAC8AP1UvAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAACQAAAENhbGxDb3VudAEAVi8ALwA/" +
           "Vi8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAZAAAAQ3JlYXRlTW9uaXRvcmVkSXRlbXNDb3Vu" +
           "dAEAVy8ALwA/Vy8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAZAAAATW9kaWZ5TW9uaXRvcmVk" +
           "SXRlbXNDb3VudAEAWC8ALwA/WC8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAWAAAAU2V0TW9u" +
           "aXRvcmluZ01vZGVDb3VudAEAWS8ALwA/WS8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAASAAAA" +
           "U2V0VHJpZ2dlcmluZ0NvdW50AQBaLwAvAD9aLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABkA" +
           "AABEZWxldGVNb25pdG9yZWRJdGVtc0NvdW50AQBbLwAvAD9bLwAAAQBnA/////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAABcAAABDcmVhdGVTdWJzY3JpcHRpb25Db3VudAEAXC8ALwA/XC8AAAEAZwP/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAXAAAATW9kaWZ5U3Vic2NyaXB0aW9uQ291bnQBAF0vAC8AP10vAAABAGcD////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAFgAAAFNldFB1Ymxpc2hpbmdNb2RlQ291bnQBAF4vAC8AP14vAAAB" +
           "AGcD/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAFB1Ymxpc2hDb3VudAEAXy8ALwA/Xy8AAAEAZwP/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAOAAAAUmVwdWJsaXNoQ291bnQBAGAvAC8AP2AvAAABAGcD////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAGgAAAFRyYW5zZmVyU3Vic2NyaXB0aW9uc0NvdW50AQBhLwAvAD9h" +
           "LwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABgAAABEZWxldGVTdWJzY3JpcHRpb25zQ291bnQB" +
           "AGIvAC8AP2IvAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADQAAAEFkZE5vZGVzQ291bnQBAGMv" +
           "AC8AP2MvAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAEFkZFJlZmVyZW5jZXNDb3VudAEA" +
           "ZC8ALwA/ZC8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAQAAAARGVsZXRlTm9kZXNDb3VudAEA" +
           "ZS8ALwA/ZS8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAVAAAARGVsZXRlUmVmZXJlbmNlc0Nv" +
           "dW50AQBmLwAvAD9mLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABCcm93c2VDb3VudAEA" +
           "Zy8ALwA/Zy8AAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAPAAAAQnJvd3NlTmV4dENvdW50AQBo" +
           "LwAvAD9oLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAACIAAABUcmFuc2xhdGVCcm93c2VQYXRo" +
           "c1RvTm9kZUlkc0NvdW50AQBpLwAvAD9pLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAA8AAABR" +
           "dWVyeUZpcnN0Q291bnQBAGovAC8AP2ovAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADgAAAFF1" +
           "ZXJ5TmV4dENvdW50AQBrLwAvAD9rLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABIAAABSZWdp" +
           "c3Rlck5vZGVzQ291bnQBAGwvAC8AP2wvAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAFAAAAFVu" +
           "cmVnaXN0ZXJOb2Rlc0NvdW50AQBtLwAvAD9tLwAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABoA" +
           "AABTZXNzaW9uU2VjdXJpdHlEaWFnbm9zdGljcwEAbi8ALwEAxAhuLwAAAQBkA/////8BAf////8JAAAA" +
           "FWCJCgIAAAAAAAkAAABTZXNzaW9uSWQBAG8vAC8AP28vAAAAEf////8BAf////8AAAAAFWCJCgIAAAAA" +
           "ABUAAABDbGllbnRVc2VySWRPZlNlc3Npb24BAHAvAC8AP3AvAAAADP////8BAf////8AAAAAF2CJCgIA" +
           "AAAAABMAAABDbGllbnRVc2VySWRIaXN0b3J5AQBxLwAvAD9xLwAAAAwBAAAAAQAAAAAAAAABAf////8A" +
           "AAAAFWCJCgIAAAAAABcAAABBdXRoZW50aWNhdGlvbk1lY2hhbmlzbQEAci8ALwA/ci8AAAAM/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACAAAAEVuY29kaW5nAQBzLwAvAD9zLwAAAAz/////AQH/////AAAAABVg" +
           "iQoCAAAAAAARAAAAVHJhbnNwb3J0UHJvdG9jb2wBAHQvAC8AP3QvAAAADP////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAwAAABTZWN1cml0eU1vZGUBAHUvAC8AP3UvAAABAC4B/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAEQAAAFNlY3VyaXR5UG9saWN5VXJpAQB2LwAvAD92LwAAAAz/////AQH/////AAAAABVgiQoCAAAA" +
           "AAARAAAAQ2xpZW50Q2VydGlmaWNhdGUBAHcvAC8AP3cvAAAAD/////8BAf////8AAAAAF2CJCgIAAAAA" +
           "ABwAAABTdWJzY3JpcHRpb25EaWFnbm9zdGljc0FycmF5AQB4LwAvAQB7CHgvAAABAGoDAQAAAAEAAAAA" +
           "AAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public SessionDiagnosticsArrayState SessionDiagnosticsArray
        {
            get
            {
                return m_sessionDiagnosticsArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionDiagnosticsArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionDiagnosticsArray = value;
            }
        }

        /// <remarks />
        public SessionSecurityDiagnosticsArrayState SessionSecurityDiagnosticsArray
        {
            get
            {
                return m_sessionSecurityDiagnosticsArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionSecurityDiagnosticsArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionSecurityDiagnosticsArray = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionDiagnosticsArray != null)
            {
                children.Add(m_sessionDiagnosticsArray);
            }

            if (m_sessionSecurityDiagnosticsArray != null)
            {
                children.Add(m_sessionSecurityDiagnosticsArray);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionDiagnosticsArray:
                {
                    if (createOrReplace)
                    {
                        if (SessionDiagnosticsArray == null)
                        {
                            if (replacement == null)
                            {
                                SessionDiagnosticsArray = new SessionDiagnosticsArrayState(this);
                            }
                            else
                            {
                                SessionDiagnosticsArray = (SessionDiagnosticsArrayState)replacement;
                            }
                        }
                    }

                    instance = SessionDiagnosticsArray;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionSecurityDiagnosticsArray:
                {
                    if (createOrReplace)
                    {
                        if (SessionSecurityDiagnosticsArray == null)
                        {
                            if (replacement == null)
                            {
                                SessionSecurityDiagnosticsArray = new SessionSecurityDiagnosticsArrayState(this);
                            }
                            else
                            {
                                SessionSecurityDiagnosticsArray = (SessionSecurityDiagnosticsArrayState)replacement;
                            }
                        }
                    }

                    instance = SessionSecurityDiagnosticsArray;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private SessionDiagnosticsArrayState m_sessionDiagnosticsArray;
        private SessionSecurityDiagnosticsArrayState m_sessionSecurityDiagnosticsArray;
        #endregion
    }
    #endif
    #endregion

    #region SessionDiagnosticsObjectState Class
    #if (!OPCUA_EXCLUDE_SessionDiagnosticsObjectState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionDiagnosticsObjectState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public SessionDiagnosticsObjectState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.SessionDiagnosticsObjectType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJAAAAFNlc3Npb25EaWFnbm9zdGljc09iamVjdFR5cGVJbnN0YW5jZQEA" +
           "7QcBAO0H7QcAAP////8DAAAAFWCJCgIAAAAAABIAAABTZXNzaW9uRGlhZ25vc3RpY3MBAO4HAC8BAJUI" +
           "7gcAAAEAYQP/////AQH/////KwAAABVgiQoCAAAAAAAJAAAAU2Vzc2lvbklkAQA7DAAvAD87DAAAABH/" +
           "////AQH/////AAAAABVgiQoCAAAAAAALAAAAU2Vzc2lvbk5hbWUBADwMAC8APzwMAAAADP////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABEAAABDbGllbnREZXNjcmlwdGlvbgEAPQwALwA/PQwAAAEANAH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAJAAAAU2VydmVyVXJpAQA+DAAvAD8+DAAAAAz/////AQH/////AAAAABVg" +
           "iQoCAAAAAAALAAAARW5kcG9pbnRVcmwBAD8MAC8APz8MAAAADP////8BAf////8AAAAAF2CJCgIAAAAA" +
           "AAkAAABMb2NhbGVJZHMBAEAMAC8AP0AMAAABACcBAQAAAAEAAAAAAAAAAQH/////AAAAABVgiQoCAAAA" +
           "AAAUAAAAQWN0dWFsU2Vzc2lvblRpbWVvdXQBAEEMAC8AP0EMAAABACIB/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAAFgAAAE1heFJlc3BvbnNlTWVzc2FnZVNpemUBAEIMAC8AP0IMAAAAB/////8BAf////8AAAAA" +
           "FWCJCgIAAAAAABQAAABDbGllbnRDb25uZWN0aW9uVGltZQEAQwwALwA/QwwAAAEAJgH/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAVAAAAQ2xpZW50TGFzdENvbnRhY3RUaW1lAQBEDAAvAD9EDAAAAQAmAf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAABkAAABDdXJyZW50U3Vic2NyaXB0aW9uc0NvdW50AQBFDAAvAD9FDAAA" +
           "AAf/////AQH/////AAAAABVgiQoCAAAAAAAaAAAAQ3VycmVudE1vbml0b3JlZEl0ZW1zQ291bnQBAEYM" +
           "AC8AP0YMAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAB0AAABDdXJyZW50UHVibGlzaFJlcXVlc3Rz" +
           "SW5RdWV1ZQEARwwALwA/RwwAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAFRvdGFsUmVxdWVz" +
           "dENvdW50AQDCIgAvAD/CIgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABgAAABVbmF1dGhvcml6" +
           "ZWRSZXF1ZXN0Q291bnQBAHMuAC8AP3MuAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABSZWFk" +
           "Q291bnQBAE8MAC8AP08MAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEAAAAEhpc3RvcnlSZWFk" +
           "Q291bnQBAFAMAC8AP1AMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFdyaXRlQ291bnQB" +
           "AFEMAC8AP1EMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAEhpc3RvcnlVcGRhdGVDb3Vu" +
           "dAEAUgwALwA/UgwAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAQ2FsbENvdW50AQBTDAAv" +
           "AD9TDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABkAAABDcmVhdGVNb25pdG9yZWRJdGVtc0Nv" +
           "dW50AQBUDAAvAD9UDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABkAAABNb2RpZnlNb25pdG9y" +
           "ZWRJdGVtc0NvdW50AQBVDAAvAD9VDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABYAAABTZXRN" +
           "b25pdG9yaW5nTW9kZUNvdW50AQBWDAAvAD9WDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABIA" +
           "AABTZXRUcmlnZ2VyaW5nQ291bnQBAFcMAC8AP1cMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "GQAAAERlbGV0ZU1vbml0b3JlZEl0ZW1zQ291bnQBAFgMAC8AP1gMAAABAGcD/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAAFwAAAENyZWF0ZVN1YnNjcmlwdGlvbkNvdW50AQBZDAAvAD9ZDAAAAQBnA/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABcAAABNb2RpZnlTdWJzY3JpcHRpb25Db3VudAEAWgwALwA/WgwAAAEAZwP/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAWAAAAU2V0UHVibGlzaGluZ01vZGVDb3VudAEAWwwALwA/WwwA" +
           "AAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAUHVibGlzaENvdW50AQBcDAAvAD9cDAAAAQBn" +
           "A/////8BAf////8AAAAAFWCJCgIAAAAAAA4AAABSZXB1Ymxpc2hDb3VudAEAXQwALwA/XQwAAAEAZwP/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAaAAAAVHJhbnNmZXJTdWJzY3JpcHRpb25zQ291bnQBAF4MAC8A" +
           "P14MAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAGAAAAERlbGV0ZVN1YnNjcmlwdGlvbnNDb3Vu" +
           "dAEAXwwALwA/XwwAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAANAAAAQWRkTm9kZXNDb3VudAEA" +
           "YAwALwA/YAwAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQWRkUmVmZXJlbmNlc0NvdW50" +
           "AQBhDAAvAD9hDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABAAAABEZWxldGVOb2Rlc0NvdW50" +
           "AQBiDAAvAD9iDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABUAAABEZWxldGVSZWZlcmVuY2Vz" +
           "Q291bnQBAGMMAC8AP2MMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAEJyb3dzZUNvdW50" +
           "AQBkDAAvAD9kDAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAA8AAABCcm93c2VOZXh0Q291bnQB" +
           "AGUMAC8AP2UMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAIgAAAFRyYW5zbGF0ZUJyb3dzZVBh" +
           "dGhzVG9Ob2RlSWRzQ291bnQBAGYMAC8AP2YMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADwAA" +
           "AFF1ZXJ5Rmlyc3RDb3VudAEAZwwALwA/ZwwAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAOAAAA" +
           "UXVlcnlOZXh0Q291bnQBAGgMAC8AP2gMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAFJl" +
           "Z2lzdGVyTm9kZXNDb3VudAEAaQwALwA/aQwAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAUAAAA" +
           "VW5yZWdpc3Rlck5vZGVzQ291bnQBAGoMAC8AP2oMAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "GgAAAFNlc3Npb25TZWN1cml0eURpYWdub3N0aWNzAQDvBwAvAQDECO8HAAABAGQD/////wEB/////wkA" +
           "AAAVYIkKAgAAAAAACQAAAFNlc3Npb25JZAEAawwALwA/awwAAAAR/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAFQAAAENsaWVudFVzZXJJZE9mU2Vzc2lvbgEAbAwALwA/bAwAAAAM/////wEB/////wAAAAAXYIkK" +
           "AgAAAAAAEwAAAENsaWVudFVzZXJJZEhpc3RvcnkBAG0MAC8AP20MAAAADAEAAAABAAAAAAAAAAEB////" +
           "/wAAAAAVYIkKAgAAAAAAFwAAAEF1dGhlbnRpY2F0aW9uTWVjaGFuaXNtAQBuDAAvAD9uDAAAAAz/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAIAAAARW5jb2RpbmcBAG8MAC8AP28MAAAADP////8BAf////8AAAAA" +
           "FWCJCgIAAAAAABEAAABUcmFuc3BvcnRQcm90b2NvbAEAcAwALwA/cAwAAAAM/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAADAAAAFNlY3VyaXR5TW9kZQEAcQwALwA/cQwAAAEALgH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAARAAAAU2VjdXJpdHlQb2xpY3lVcmkBAHIMAC8AP3IMAAAADP////8BAf////8AAAAAFWCJCgIA" +
           "AAAAABEAAABDbGllbnRDZXJ0aWZpY2F0ZQEAcwwALwA/cwwAAAAP/////wEB/////wAAAAAXYIkKAgAA" +
           "AAAAHAAAAFN1YnNjcmlwdGlvbkRpYWdub3N0aWNzQXJyYXkBAPAHAC8BAHsI8AcAAAEAagMBAAAAAQAA" +
           "AAAAAAABAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public SessionDiagnosticsVariableState SessionDiagnostics
        {
            get
            {
                return m_sessionDiagnostics;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionDiagnostics, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionDiagnostics = value;
            }
        }

        /// <remarks />
        public SessionSecurityDiagnosticsState SessionSecurityDiagnostics
        {
            get
            {
                return m_sessionSecurityDiagnostics;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionSecurityDiagnostics, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionSecurityDiagnostics = value;
            }
        }

        /// <remarks />
        public SubscriptionDiagnosticsArrayState SubscriptionDiagnosticsArray
        {
            get
            {
                return m_subscriptionDiagnosticsArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_subscriptionDiagnosticsArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_subscriptionDiagnosticsArray = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionDiagnostics != null)
            {
                children.Add(m_sessionDiagnostics);
            }

            if (m_sessionSecurityDiagnostics != null)
            {
                children.Add(m_sessionSecurityDiagnostics);
            }

            if (m_subscriptionDiagnosticsArray != null)
            {
                children.Add(m_subscriptionDiagnosticsArray);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionDiagnostics:
                {
                    if (createOrReplace)
                    {
                        if (SessionDiagnostics == null)
                        {
                            if (replacement == null)
                            {
                                SessionDiagnostics = new SessionDiagnosticsVariableState(this);
                            }
                            else
                            {
                                SessionDiagnostics = (SessionDiagnosticsVariableState)replacement;
                            }
                        }
                    }

                    instance = SessionDiagnostics;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionSecurityDiagnostics:
                {
                    if (createOrReplace)
                    {
                        if (SessionSecurityDiagnostics == null)
                        {
                            if (replacement == null)
                            {
                                SessionSecurityDiagnostics = new SessionSecurityDiagnosticsState(this);
                            }
                            else
                            {
                                SessionSecurityDiagnostics = (SessionSecurityDiagnosticsState)replacement;
                            }
                        }
                    }

                    instance = SessionSecurityDiagnostics;
                    break;
                }

                case Opc.Ua.BrowseNames.SubscriptionDiagnosticsArray:
                {
                    if (createOrReplace)
                    {
                        if (SubscriptionDiagnosticsArray == null)
                        {
                            if (replacement == null)
                            {
                                SubscriptionDiagnosticsArray = new SubscriptionDiagnosticsArrayState(this);
                            }
                            else
                            {
                                SubscriptionDiagnosticsArray = (SubscriptionDiagnosticsArrayState)replacement;
                            }
                        }
                    }

                    instance = SubscriptionDiagnosticsArray;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private SessionDiagnosticsVariableState m_sessionDiagnostics;
        private SessionSecurityDiagnosticsState m_sessionSecurityDiagnostics;
        private SubscriptionDiagnosticsArrayState m_subscriptionDiagnosticsArray;
        #endregion
    }
    #endif
    #endregion

    #region VendorServerInfoState Class
    #if (!OPCUA_EXCLUDE_VendorServerInfoState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class VendorServerInfoState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public VendorServerInfoState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.VendorServerInfoType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAFZlbmRvclNlcnZlckluZm9UeXBlSW5zdGFuY2UBAPEHAQDxB/EH" +
           "AAD/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ServerRedundancyState Class
    #if (!OPCUA_EXCLUDE_ServerRedundancyState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerRedundancyState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public ServerRedundancyState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ServerRedundancyType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAFNlcnZlclJlZHVuZGFuY3lUeXBlSW5zdGFuY2UBAPIHAQDyB/IH" +
           "AAD/////AQAAABVgiQoCAAAAAAARAAAAUmVkdW5kYW5jeVN1cHBvcnQBAPMHAC4ARPMHAAABAFMD////" +
           "/wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<RedundancySupport> RedundancySupport
        {
            get
            {
                return m_redundancySupport;
            }

            set
            {
                if (!Object.ReferenceEquals(m_redundancySupport, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_redundancySupport = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_redundancySupport != null)
            {
                children.Add(m_redundancySupport);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.RedundancySupport:
                {
                    if (createOrReplace)
                    {
                        if (RedundancySupport == null)
                        {
                            if (replacement == null)
                            {
                                RedundancySupport = new PropertyState<RedundancySupport>(this);
                            }
                            else
                            {
                                RedundancySupport = (PropertyState<RedundancySupport>)replacement;
                            }
                        }
                    }

                    instance = RedundancySupport;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<RedundancySupport> m_redundancySupport;
        #endregion
    }
    #endif
    #endregion

    #region TransparentRedundancyState Class
    #if (!OPCUA_EXCLUDE_TransparentRedundancyState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class TransparentRedundancyState : ServerRedundancyState
    {
        #region Constructors
        /// <remarks />
        public TransparentRedundancyState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.TransparentRedundancyType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIQAAAFRyYW5zcGFyZW50UmVkdW5kYW5jeVR5cGVJbnN0YW5jZQEA9AcB" +
           "APQH9AcAAP////8DAAAAFWCJCgIAAAAAABEAAABSZWR1bmRhbmN5U3VwcG9ydAEAdAwALgBEdAwAAAEA" +
           "UwP/////AQH/////AAAAABVgiQoCAAAAAAAPAAAAQ3VycmVudFNlcnZlcklkAQD1BwAuAET1BwAAAAz/" +
           "////AQH/////AAAAABdgiQoCAAAAAAAUAAAAUmVkdW5kYW50U2VydmVyQXJyYXkBAPYHAC4ARPYHAAAB" +
           "AFUDAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> CurrentServerId
        {
            get
            {
                return m_currentServerId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentServerId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentServerId = value;
            }
        }

        /// <remarks />
        public PropertyState<RedundantServerDataType[]> RedundantServerArray
        {
            get
            {
                return m_redundantServerArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_redundantServerArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_redundantServerArray = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_currentServerId != null)
            {
                children.Add(m_currentServerId);
            }

            if (m_redundantServerArray != null)
            {
                children.Add(m_redundantServerArray);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.CurrentServerId:
                {
                    if (createOrReplace)
                    {
                        if (CurrentServerId == null)
                        {
                            if (replacement == null)
                            {
                                CurrentServerId = new PropertyState<string>(this);
                            }
                            else
                            {
                                CurrentServerId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = CurrentServerId;
                    break;
                }

                case Opc.Ua.BrowseNames.RedundantServerArray:
                {
                    if (createOrReplace)
                    {
                        if (RedundantServerArray == null)
                        {
                            if (replacement == null)
                            {
                                RedundantServerArray = new PropertyState<RedundantServerDataType[]>(this);
                            }
                            else
                            {
                                RedundantServerArray = (PropertyState<RedundantServerDataType[]>)replacement;
                            }
                        }
                    }

                    instance = RedundantServerArray;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_currentServerId;
        private PropertyState<RedundantServerDataType[]> m_redundantServerArray;
        #endregion
    }
    #endif
    #endregion

    #region NonTransparentRedundancyState Class
    #if (!OPCUA_EXCLUDE_NonTransparentRedundancyState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class NonTransparentRedundancyState : ServerRedundancyState
    {
        #region Constructors
        /// <remarks />
        public NonTransparentRedundancyState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.NonTransparentRedundancyType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJAAAAE5vblRyYW5zcGFyZW50UmVkdW5kYW5jeVR5cGVJbnN0YW5jZQEA" +
           "9wcBAPcH9wcAAP////8CAAAAFWCJCgIAAAAAABEAAABSZWR1bmRhbmN5U3VwcG9ydAEAdQwALgBEdQwA" +
           "AAEAUwP/////AQH/////AAAAABdgiQoCAAAAAAAOAAAAU2VydmVyVXJpQXJyYXkBAPgHAC4ARPgHAAAA" +
           "DAEAAAABAAAAAAAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string[]> ServerUriArray
        {
            get
            {
                return m_serverUriArray;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverUriArray, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverUriArray = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverUriArray != null)
            {
                children.Add(m_serverUriArray);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerUriArray:
                {
                    if (createOrReplace)
                    {
                        if (ServerUriArray == null)
                        {
                            if (replacement == null)
                            {
                                ServerUriArray = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                ServerUriArray = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = ServerUriArray;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string[]> m_serverUriArray;
        #endregion
    }
    #endif
    #endregion

    #region NonTransparentNetworkRedundancyState Class
    #if (!OPCUA_EXCLUDE_NonTransparentNetworkRedundancyState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class NonTransparentNetworkRedundancyState : NonTransparentRedundancyState
    {
        #region Constructors
        /// <remarks />
        public NonTransparentNetworkRedundancyState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.NonTransparentNetworkRedundancyType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKwAAAE5vblRyYW5zcGFyZW50TmV0d29ya1JlZHVuZGFuY3lUeXBlSW5z" +
           "dGFuY2UBAKkuAQCpLqkuAAD/////AwAAABVgiQoCAAAAAAARAAAAUmVkdW5kYW5jeVN1cHBvcnQBAKou" +
           "AC4ARKouAAABAFMD/////wEB/////wAAAAAXYIkKAgAAAAAADgAAAFNlcnZlclVyaUFycmF5AQCrLgAu" +
           "AESrLgAAAAwBAAAAAQAAAAAAAAABAf////8AAAAAF2CJCgIAAAAAABMAAABTZXJ2ZXJOZXR3b3JrR3Jv" +
           "dXBzAQCsLgAuAESsLgAAAQCoLgEAAAABAAAAAAAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<NetworkGroupDataType[]> ServerNetworkGroups
        {
            get
            {
                return m_serverNetworkGroups;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverNetworkGroups, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverNetworkGroups = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverNetworkGroups != null)
            {
                children.Add(m_serverNetworkGroups);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerNetworkGroups:
                {
                    if (createOrReplace)
                    {
                        if (ServerNetworkGroups == null)
                        {
                            if (replacement == null)
                            {
                                ServerNetworkGroups = new PropertyState<NetworkGroupDataType[]>(this);
                            }
                            else
                            {
                                ServerNetworkGroups = (PropertyState<NetworkGroupDataType[]>)replacement;
                            }
                        }
                    }

                    instance = ServerNetworkGroups;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<NetworkGroupDataType[]> m_serverNetworkGroups;
        #endregion
    }
    #endif
    #endregion

    #region OperationLimitsState Class
    #if (!OPCUA_EXCLUDE_OperationLimitsState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class OperationLimitsState : FolderState
    {
        #region Constructors
        /// <remarks />
        public OperationLimitsState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.OperationLimitsType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (MaxNodesPerRead != null)
            {
                MaxNodesPerRead.Initialize(context, MaxNodesPerRead_InitializationString);
            }

            if (MaxNodesPerHistoryReadData != null)
            {
                MaxNodesPerHistoryReadData.Initialize(context, MaxNodesPerHistoryReadData_InitializationString);
            }

            if (MaxNodesPerHistoryReadEvents != null)
            {
                MaxNodesPerHistoryReadEvents.Initialize(context, MaxNodesPerHistoryReadEvents_InitializationString);
            }

            if (MaxNodesPerWrite != null)
            {
                MaxNodesPerWrite.Initialize(context, MaxNodesPerWrite_InitializationString);
            }

            if (MaxNodesPerHistoryUpdateData != null)
            {
                MaxNodesPerHistoryUpdateData.Initialize(context, MaxNodesPerHistoryUpdateData_InitializationString);
            }

            if (MaxNodesPerHistoryUpdateEvents != null)
            {
                MaxNodesPerHistoryUpdateEvents.Initialize(context, MaxNodesPerHistoryUpdateEvents_InitializationString);
            }

            if (MaxNodesPerMethodCall != null)
            {
                MaxNodesPerMethodCall.Initialize(context, MaxNodesPerMethodCall_InitializationString);
            }

            if (MaxNodesPerBrowse != null)
            {
                MaxNodesPerBrowse.Initialize(context, MaxNodesPerBrowse_InitializationString);
            }

            if (MaxNodesPerRegisterNodes != null)
            {
                MaxNodesPerRegisterNodes.Initialize(context, MaxNodesPerRegisterNodes_InitializationString);
            }

            if (MaxNodesPerTranslateBrowsePathsToNodeIds != null)
            {
                MaxNodesPerTranslateBrowsePathsToNodeIds.Initialize(context, MaxNodesPerTranslateBrowsePathsToNodeIds_InitializationString);
            }

            if (MaxNodesPerNodeManagement != null)
            {
                MaxNodesPerNodeManagement.Initialize(context, MaxNodesPerNodeManagement_InitializationString);
            }

            if (MaxMonitoredItemsPerCall != null)
            {
                MaxMonitoredItemsPerCall.Initialize(context, MaxMonitoredItemsPerCall_InitializationString);
            }
        }

        #region Initialization String
        private const string MaxNodesPerRead_InitializationString =
           "//////////8VYIkKAgAAAAAADwAAAE1heE5vZGVzUGVyUmVhZAEALS0ALgBELS0AAAAH/////wEB////" +
           "/wAAAAA=";

        private const string MaxNodesPerHistoryReadData_InitializationString =
           "//////////8VYIkKAgAAAAAAGgAAAE1heE5vZGVzUGVySGlzdG9yeVJlYWREYXRhAQCBLwAuAESBLwAA" +
           "AAf/////AQH/////AAAAAA==";

        private const string MaxNodesPerHistoryReadEvents_InitializationString =
           "//////////8VYIkKAgAAAAAAHAAAAE1heE5vZGVzUGVySGlzdG9yeVJlYWRFdmVudHMBAIIvAC4ARIIv" +
           "AAAAB/////8BAf////8AAAAA";

        private const string MaxNodesPerWrite_InitializationString =
           "//////////8VYIkKAgAAAAAAEAAAAE1heE5vZGVzUGVyV3JpdGUBAC8tAC4ARC8tAAAAB/////8BAf//" +
           "//8AAAAA";

        private const string MaxNodesPerHistoryUpdateData_InitializationString =
           "//////////8VYIkKAgAAAAAAHAAAAE1heE5vZGVzUGVySGlzdG9yeVVwZGF0ZURhdGEBAIMvAC4ARIMv" +
           "AAAAB/////8BAf////8AAAAA";

        private const string MaxNodesPerHistoryUpdateEvents_InitializationString =
           "//////////8VYIkKAgAAAAAAHgAAAE1heE5vZGVzUGVySGlzdG9yeVVwZGF0ZUV2ZW50cwEAhC8ALgBE" +
           "hC8AAAAH/////wEB/////wAAAAA=";

        private const string MaxNodesPerMethodCall_InitializationString =
           "//////////8VYIkKAgAAAAAAFQAAAE1heE5vZGVzUGVyTWV0aG9kQ2FsbAEAMS0ALgBEMS0AAAAH////" +
           "/wEB/////wAAAAA=";

        private const string MaxNodesPerBrowse_InitializationString =
           "//////////8VYIkKAgAAAAAAEQAAAE1heE5vZGVzUGVyQnJvd3NlAQAyLQAuAEQyLQAAAAf/////AQH/" +
           "////AAAAAA==";

        private const string MaxNodesPerRegisterNodes_InitializationString =
           "//////////8VYIkKAgAAAAAAGAAAAE1heE5vZGVzUGVyUmVnaXN0ZXJOb2RlcwEAMy0ALgBEMy0AAAAH" +
           "/////wEB/////wAAAAA=";

        private const string MaxNodesPerTranslateBrowsePathsToNodeIds_InitializationString =
           "//////////8VYIkKAgAAAAAAKAAAAE1heE5vZGVzUGVyVHJhbnNsYXRlQnJvd3NlUGF0aHNUb05vZGVJ" +
           "ZHMBADQtAC4ARDQtAAAAB/////8BAf////8AAAAA";

        private const string MaxNodesPerNodeManagement_InitializationString =
           "//////////8VYIkKAgAAAAAAGQAAAE1heE5vZGVzUGVyTm9kZU1hbmFnZW1lbnQBADUtAC4ARDUtAAAA" +
           "B/////8BAf////8AAAAA";

        private const string MaxMonitoredItemsPerCall_InitializationString =
           "//////////8VYIkKAgAAAAAAGAAAAE1heE1vbml0b3JlZEl0ZW1zUGVyQ2FsbAEANi0ALgBENi0AAAAH" +
           "/////wEB/////wAAAAA=";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAGwAAAE9wZXJhdGlvbkxpbWl0c1R5cGVJbnN0YW5jZQEALC0BACwtLC0A" +
           "AP////8MAAAAFWCJCgIAAAAAAA8AAABNYXhOb2Rlc1BlclJlYWQBAC0tAC4ARC0tAAAAB/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABoAAABNYXhOb2Rlc1Blckhpc3RvcnlSZWFkRGF0YQEAgS8ALgBEgS8AAAAH" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAAHAAAAE1heE5vZGVzUGVySGlzdG9yeVJlYWRFdmVudHMBAIIv" +
           "AC4ARIIvAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABAAAABNYXhOb2Rlc1BlcldyaXRlAQAvLQAu" +
           "AEQvLQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAcAAAATWF4Tm9kZXNQZXJIaXN0b3J5VXBkYXRl" +
           "RGF0YQEAgy8ALgBEgy8AAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAHgAAAE1heE5vZGVzUGVySGlz" +
           "dG9yeVVwZGF0ZUV2ZW50cwEAhC8ALgBEhC8AAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFQAAAE1h" +
           "eE5vZGVzUGVyTWV0aG9kQ2FsbAEAMS0ALgBEMS0AAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEQAA" +
           "AE1heE5vZGVzUGVyQnJvd3NlAQAyLQAuAEQyLQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAYAAAA" +
           "TWF4Tm9kZXNQZXJSZWdpc3Rlck5vZGVzAQAzLQAuAEQzLQAAAAf/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAoAAAATWF4Tm9kZXNQZXJUcmFuc2xhdGVCcm93c2VQYXRoc1RvTm9kZUlkcwEANC0ALgBENC0AAAAH" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAE1heE5vZGVzUGVyTm9kZU1hbmFnZW1lbnQBADUtAC4A" +
           "RDUtAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABgAAABNYXhNb25pdG9yZWRJdGVtc1BlckNhbGwB" +
           "ADYtAC4ARDYtAAAAB/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<uint> MaxNodesPerRead
        {
            get
            {
                return m_maxNodesPerRead;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerRead, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerRead = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerHistoryReadData
        {
            get
            {
                return m_maxNodesPerHistoryReadData;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerHistoryReadData, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerHistoryReadData = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerHistoryReadEvents
        {
            get
            {
                return m_maxNodesPerHistoryReadEvents;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerHistoryReadEvents, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerHistoryReadEvents = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerWrite
        {
            get
            {
                return m_maxNodesPerWrite;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerWrite, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerWrite = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerHistoryUpdateData
        {
            get
            {
                return m_maxNodesPerHistoryUpdateData;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerHistoryUpdateData, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerHistoryUpdateData = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerHistoryUpdateEvents
        {
            get
            {
                return m_maxNodesPerHistoryUpdateEvents;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerHistoryUpdateEvents, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerHistoryUpdateEvents = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerMethodCall
        {
            get
            {
                return m_maxNodesPerMethodCall;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerMethodCall, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerMethodCall = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerBrowse
        {
            get
            {
                return m_maxNodesPerBrowse;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerBrowse, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerBrowse = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerRegisterNodes
        {
            get
            {
                return m_maxNodesPerRegisterNodes;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerRegisterNodes, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerRegisterNodes = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerTranslateBrowsePathsToNodeIds
        {
            get
            {
                return m_maxNodesPerTranslateBrowsePathsToNodeIds;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerTranslateBrowsePathsToNodeIds, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerTranslateBrowsePathsToNodeIds = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxNodesPerNodeManagement
        {
            get
            {
                return m_maxNodesPerNodeManagement;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNodesPerNodeManagement, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNodesPerNodeManagement = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxMonitoredItemsPerCall
        {
            get
            {
                return m_maxMonitoredItemsPerCall;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxMonitoredItemsPerCall, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxMonitoredItemsPerCall = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_maxNodesPerRead != null)
            {
                children.Add(m_maxNodesPerRead);
            }

            if (m_maxNodesPerHistoryReadData != null)
            {
                children.Add(m_maxNodesPerHistoryReadData);
            }

            if (m_maxNodesPerHistoryReadEvents != null)
            {
                children.Add(m_maxNodesPerHistoryReadEvents);
            }

            if (m_maxNodesPerWrite != null)
            {
                children.Add(m_maxNodesPerWrite);
            }

            if (m_maxNodesPerHistoryUpdateData != null)
            {
                children.Add(m_maxNodesPerHistoryUpdateData);
            }

            if (m_maxNodesPerHistoryUpdateEvents != null)
            {
                children.Add(m_maxNodesPerHistoryUpdateEvents);
            }

            if (m_maxNodesPerMethodCall != null)
            {
                children.Add(m_maxNodesPerMethodCall);
            }

            if (m_maxNodesPerBrowse != null)
            {
                children.Add(m_maxNodesPerBrowse);
            }

            if (m_maxNodesPerRegisterNodes != null)
            {
                children.Add(m_maxNodesPerRegisterNodes);
            }

            if (m_maxNodesPerTranslateBrowsePathsToNodeIds != null)
            {
                children.Add(m_maxNodesPerTranslateBrowsePathsToNodeIds);
            }

            if (m_maxNodesPerNodeManagement != null)
            {
                children.Add(m_maxNodesPerNodeManagement);
            }

            if (m_maxMonitoredItemsPerCall != null)
            {
                children.Add(m_maxMonitoredItemsPerCall);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.MaxNodesPerRead:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerRead == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerRead = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerRead = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerRead;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerHistoryReadData:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerHistoryReadData == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerHistoryReadData = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerHistoryReadData = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerHistoryReadData;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerHistoryReadEvents:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerHistoryReadEvents == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerHistoryReadEvents = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerHistoryReadEvents = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerHistoryReadEvents;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerWrite:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerWrite == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerWrite = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerWrite = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerWrite;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerHistoryUpdateData:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerHistoryUpdateData == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerHistoryUpdateData = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerHistoryUpdateData = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerHistoryUpdateData;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerHistoryUpdateEvents:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerHistoryUpdateEvents == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerHistoryUpdateEvents = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerHistoryUpdateEvents = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerHistoryUpdateEvents;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerMethodCall:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerMethodCall == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerMethodCall = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerMethodCall = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerMethodCall;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerBrowse:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerBrowse == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerBrowse = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerBrowse = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerBrowse;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerRegisterNodes:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerRegisterNodes == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerRegisterNodes = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerRegisterNodes = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerRegisterNodes;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerTranslateBrowsePathsToNodeIds:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerTranslateBrowsePathsToNodeIds == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerTranslateBrowsePathsToNodeIds = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerTranslateBrowsePathsToNodeIds = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerTranslateBrowsePathsToNodeIds;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNodesPerNodeManagement:
                {
                    if (createOrReplace)
                    {
                        if (MaxNodesPerNodeManagement == null)
                        {
                            if (replacement == null)
                            {
                                MaxNodesPerNodeManagement = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxNodesPerNodeManagement = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNodesPerNodeManagement;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxMonitoredItemsPerCall:
                {
                    if (createOrReplace)
                    {
                        if (MaxMonitoredItemsPerCall == null)
                        {
                            if (replacement == null)
                            {
                                MaxMonitoredItemsPerCall = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxMonitoredItemsPerCall = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxMonitoredItemsPerCall;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<uint> m_maxNodesPerRead;
        private PropertyState<uint> m_maxNodesPerHistoryReadData;
        private PropertyState<uint> m_maxNodesPerHistoryReadEvents;
        private PropertyState<uint> m_maxNodesPerWrite;
        private PropertyState<uint> m_maxNodesPerHistoryUpdateData;
        private PropertyState<uint> m_maxNodesPerHistoryUpdateEvents;
        private PropertyState<uint> m_maxNodesPerMethodCall;
        private PropertyState<uint> m_maxNodesPerBrowse;
        private PropertyState<uint> m_maxNodesPerRegisterNodes;
        private PropertyState<uint> m_maxNodesPerTranslateBrowsePathsToNodeIds;
        private PropertyState<uint> m_maxNodesPerNodeManagement;
        private PropertyState<uint> m_maxMonitoredItemsPerCall;
        #endregion
    }
    #endif
    #endregion

    #region FileState Class
    #if (!OPCUA_EXCLUDE_FileState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class FileState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public FileState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.FileType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (MimeType != null)
            {
                MimeType.Initialize(context, MimeType_InitializationString);
            }

            if (MaxByteStringLength != null)
            {
                MaxByteStringLength.Initialize(context, MaxByteStringLength_InitializationString);
            }

            if (LastModifiedTime != null)
            {
                LastModifiedTime.Initialize(context, LastModifiedTime_InitializationString);
            }
        }

        #region Initialization String
        private const string MimeType_InitializationString =
           "//////////8VYIkKAgAAAAAACAAAAE1pbWVUeXBlAQAdNAAuAEQdNAAAAAz/////AQH/////AAAAAA==";

        private const string MaxByteStringLength_InitializationString =
           "//////////8VYIkKAgAAAAAAEwAAAE1heEJ5dGVTdHJpbmdMZW5ndGgBALReAC4ARLReAAAAB/////8B" +
           "Af////8AAAAA";

        private const string LastModifiedTime_InitializationString =
           "//////////8VYIkKAgAAAAAAEAAAAExhc3RNb2RpZmllZFRpbWUBAHBiAC4ARHBiAAAADf////8BAf//" +
           "//8AAAAA";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAEAAAAEZpbGVUeXBlSW5zdGFuY2UBADctAQA3LTctAAD/////DQAAABVg" +
           "iQoCAAAAAAAEAAAAU2l6ZQEAOC0ALgBEOC0AAAAJ/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFdy" +
           "aXRhYmxlAQCOMQAuAESOMQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAVXNlcldyaXRhYmxl" +
           "AQCPMQAuAESPMQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAT3BlbkNvdW50AQA7LQAuAEQ7" +
           "LQAAAAX/////AQH/////AAAAABVgiQoCAAAAAAAIAAAATWltZVR5cGUBAB00AC4ARB00AAAADP////8B" +
           "Af////8AAAAAFWCJCgIAAAAAABMAAABNYXhCeXRlU3RyaW5nTGVuZ3RoAQC0XgAuAES0XgAAAAf/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAQAAAATGFzdE1vZGlmaWVkVGltZQEAcGIALgBEcGIAAAAN/////wEB" +
           "/////wAAAAAEYYIKBAAAAAAABAAAAE9wZW4BADwtAC8BADwtPC0AAAEB/////wIAAAAXYKkKAgAAAAAA" +
           "DgAAAElucHV0QXJndW1lbnRzAQA9LQAuAEQ9LQAAlgEAAAABACoBARMAAAAEAAAATW9kZQAD/////wAA" +
           "AAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAXYKkKAgAAAAAADwAAAE91dHB1dEFyZ3VtZW50cwEA" +
           "Pi0ALgBEPi0AAJYBAAAAAQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8AAAAAAAEAKAEBAAAAAQAA" +
           "AAEAAAABAf////8AAAAABGGCCgQAAAAAAAUAAABDbG9zZQEAPy0ALwEAPy0/LQAAAQH/////AQAAABdg" +
           "qQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAEAtAC4AREAtAACWAQAAAAEAKgEBGQAAAAoAAABGaWxl" +
           "SGFuZGxlAAf/////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAAARhggoEAAAAAAAEAAAAUmVh" +
           "ZAEAQS0ALwEAQS1BLQAAAQH/////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAEItAC4A" +
           "REItAACWAgAAAAEAKgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////AAAAAAABACoBARUAAAAGAAAATGVu" +
           "Z3RoAAb/////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAABdgqQoCAAAAAAAPAAAAT3V0cHV0" +
           "QXJndW1lbnRzAQBDLQAuAERDLQAAlgEAAAABACoBARMAAAAEAAAARGF0YQAP/////wAAAAAAAQAoAQEA" +
           "AAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAAAAAABQAAAFdyaXRlAQBELQAvAQBELUQtAAABAf////8B" +
           "AAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEARS0ALgBERS0AAJYCAAAAAQAqAQEZAAAACgAA" +
           "AEZpbGVIYW5kbGUAB/////8AAAAAAAEAKgEBEwAAAAQAAABEYXRhAA//////AAAAAAABACgBAQAAAAEA" +
           "AAACAAAAAQH/////AAAAAARhggoEAAAAAAALAAAAR2V0UG9zaXRpb24BAEYtAC8BAEYtRi0AAAEB////" +
           "/wIAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBHLQAuAERHLQAAlgEAAAABACoBARkAAAAK" +
           "AAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAXYKkKAgAAAAAA" +
           "DwAAAE91dHB1dEFyZ3VtZW50cwEASC0ALgBESC0AAJYBAAAAAQAqAQEXAAAACAAAAFBvc2l0aW9uAAn/" +
           "////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAAARhggoEAAAAAAALAAAAU2V0UG9zaXRpb24B" +
           "AEktAC8BAEktSS0AAAEB/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBKLQAuAERK" +
           "LQAAlgIAAAABACoBARkAAAAKAAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAqAQEXAAAACAAAAFBvc2l0" +
           "aW9uAAn/////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<ulong> Size
        {
            get
            {
                return m_size;
            }

            set
            {
                if (!Object.ReferenceEquals(m_size, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_size = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> Writable
        {
            get
            {
                return m_writable;
            }

            set
            {
                if (!Object.ReferenceEquals(m_writable, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_writable = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> UserWritable
        {
            get
            {
                return m_userWritable;
            }

            set
            {
                if (!Object.ReferenceEquals(m_userWritable, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_userWritable = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> OpenCount
        {
            get
            {
                return m_openCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_openCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_openCount = value;
            }
        }

        /// <remarks />
        public PropertyState<string> MimeType
        {
            get
            {
                return m_mimeType;
            }

            set
            {
                if (!Object.ReferenceEquals(m_mimeType, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_mimeType = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> MaxByteStringLength
        {
            get
            {
                return m_maxByteStringLength;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxByteStringLength, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxByteStringLength = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> LastModifiedTime
        {
            get
            {
                return m_lastModifiedTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_lastModifiedTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_lastModifiedTime = value;
            }
        }

        /// <remarks />
        public OpenMethodState Open
        {
            get
            {
                return m_openMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_openMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_openMethod = value;
            }
        }

        /// <remarks />
        public CloseMethodState Close
        {
            get
            {
                return m_closeMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_closeMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_closeMethod = value;
            }
        }

        /// <remarks />
        public ReadMethodState Read
        {
            get
            {
                return m_readMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_readMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_readMethod = value;
            }
        }

        /// <remarks />
        public WriteMethodState Write
        {
            get
            {
                return m_writeMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_writeMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_writeMethod = value;
            }
        }

        /// <remarks />
        public GetPositionMethodState GetPosition
        {
            get
            {
                return m_getPositionMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_getPositionMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_getPositionMethod = value;
            }
        }

        /// <remarks />
        public SetPositionMethodState SetPosition
        {
            get
            {
                return m_setPositionMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_setPositionMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_setPositionMethod = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_size != null)
            {
                children.Add(m_size);
            }

            if (m_writable != null)
            {
                children.Add(m_writable);
            }

            if (m_userWritable != null)
            {
                children.Add(m_userWritable);
            }

            if (m_openCount != null)
            {
                children.Add(m_openCount);
            }

            if (m_mimeType != null)
            {
                children.Add(m_mimeType);
            }

            if (m_maxByteStringLength != null)
            {
                children.Add(m_maxByteStringLength);
            }

            if (m_lastModifiedTime != null)
            {
                children.Add(m_lastModifiedTime);
            }

            if (m_openMethod != null)
            {
                children.Add(m_openMethod);
            }

            if (m_closeMethod != null)
            {
                children.Add(m_closeMethod);
            }

            if (m_readMethod != null)
            {
                children.Add(m_readMethod);
            }

            if (m_writeMethod != null)
            {
                children.Add(m_writeMethod);
            }

            if (m_getPositionMethod != null)
            {
                children.Add(m_getPositionMethod);
            }

            if (m_setPositionMethod != null)
            {
                children.Add(m_setPositionMethod);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Size:
                {
                    if (createOrReplace)
                    {
                        if (Size == null)
                        {
                            if (replacement == null)
                            {
                                Size = new PropertyState<ulong>(this);
                            }
                            else
                            {
                                Size = (PropertyState<ulong>)replacement;
                            }
                        }
                    }

                    instance = Size;
                    break;
                }

                case Opc.Ua.BrowseNames.Writable:
                {
                    if (createOrReplace)
                    {
                        if (Writable == null)
                        {
                            if (replacement == null)
                            {
                                Writable = new PropertyState<bool>(this);
                            }
                            else
                            {
                                Writable = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = Writable;
                    break;
                }

                case Opc.Ua.BrowseNames.UserWritable:
                {
                    if (createOrReplace)
                    {
                        if (UserWritable == null)
                        {
                            if (replacement == null)
                            {
                                UserWritable = new PropertyState<bool>(this);
                            }
                            else
                            {
                                UserWritable = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = UserWritable;
                    break;
                }

                case Opc.Ua.BrowseNames.OpenCount:
                {
                    if (createOrReplace)
                    {
                        if (OpenCount == null)
                        {
                            if (replacement == null)
                            {
                                OpenCount = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                OpenCount = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = OpenCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MimeType:
                {
                    if (createOrReplace)
                    {
                        if (MimeType == null)
                        {
                            if (replacement == null)
                            {
                                MimeType = new PropertyState<string>(this);
                            }
                            else
                            {
                                MimeType = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = MimeType;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxByteStringLength:
                {
                    if (createOrReplace)
                    {
                        if (MaxByteStringLength == null)
                        {
                            if (replacement == null)
                            {
                                MaxByteStringLength = new PropertyState<uint>(this);
                            }
                            else
                            {
                                MaxByteStringLength = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxByteStringLength;
                    break;
                }

                case Opc.Ua.BrowseNames.LastModifiedTime:
                {
                    if (createOrReplace)
                    {
                        if (LastModifiedTime == null)
                        {
                            if (replacement == null)
                            {
                                LastModifiedTime = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                LastModifiedTime = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = LastModifiedTime;
                    break;
                }

                case Opc.Ua.BrowseNames.Open:
                {
                    if (createOrReplace)
                    {
                        if (Open == null)
                        {
                            if (replacement == null)
                            {
                                Open = new OpenMethodState(this);
                            }
                            else
                            {
                                Open = (OpenMethodState)replacement;
                            }
                        }
                    }

                    instance = Open;
                    break;
                }

                case Opc.Ua.BrowseNames.Close:
                {
                    if (createOrReplace)
                    {
                        if (Close == null)
                        {
                            if (replacement == null)
                            {
                                Close = new CloseMethodState(this);
                            }
                            else
                            {
                                Close = (CloseMethodState)replacement;
                            }
                        }
                    }

                    instance = Close;
                    break;
                }

                case Opc.Ua.BrowseNames.Read:
                {
                    if (createOrReplace)
                    {
                        if (Read == null)
                        {
                            if (replacement == null)
                            {
                                Read = new ReadMethodState(this);
                            }
                            else
                            {
                                Read = (ReadMethodState)replacement;
                            }
                        }
                    }

                    instance = Read;
                    break;
                }

                case Opc.Ua.BrowseNames.Write:
                {
                    if (createOrReplace)
                    {
                        if (Write == null)
                        {
                            if (replacement == null)
                            {
                                Write = new WriteMethodState(this);
                            }
                            else
                            {
                                Write = (WriteMethodState)replacement;
                            }
                        }
                    }

                    instance = Write;
                    break;
                }

                case Opc.Ua.BrowseNames.GetPosition:
                {
                    if (createOrReplace)
                    {
                        if (GetPosition == null)
                        {
                            if (replacement == null)
                            {
                                GetPosition = new GetPositionMethodState(this);
                            }
                            else
                            {
                                GetPosition = (GetPositionMethodState)replacement;
                            }
                        }
                    }

                    instance = GetPosition;
                    break;
                }

                case Opc.Ua.BrowseNames.SetPosition:
                {
                    if (createOrReplace)
                    {
                        if (SetPosition == null)
                        {
                            if (replacement == null)
                            {
                                SetPosition = new SetPositionMethodState(this);
                            }
                            else
                            {
                                SetPosition = (SetPositionMethodState)replacement;
                            }
                        }
                    }

                    instance = SetPosition;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<ulong> m_size;
        private PropertyState<bool> m_writable;
        private PropertyState<bool> m_userWritable;
        private PropertyState<ushort> m_openCount;
        private PropertyState<string> m_mimeType;
        private PropertyState<uint> m_maxByteStringLength;
        private PropertyState<DateTime> m_lastModifiedTime;
        private OpenMethodState m_openMethod;
        private CloseMethodState m_closeMethod;
        private ReadMethodState m_readMethod;
        private WriteMethodState m_writeMethod;
        private GetPositionMethodState m_getPositionMethod;
        private SetPositionMethodState m_setPositionMethod;
        #endregion
    }
    #endif
    #endregion

    #region AddressSpaceFileState Class
    #if (!OPCUA_EXCLUDE_AddressSpaceFileState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AddressSpaceFileState : FileState
    {
        #region Constructors
        /// <remarks />
        public AddressSpaceFileState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AddressSpaceFileType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (ExportNamespace != null)
            {
                ExportNamespace.Initialize(context, ExportNamespace_InitializationString);
            }
        }

        #region Initialization String
        private const string ExportNamespace_InitializationString =
           "//////////8EYYIKBAAAAAAADwAAAEV4cG9ydE5hbWVzcGFjZQEAXy0ALwEAXy1fLQAAAQH/////AAAA" +
           "AA==";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAEFkZHJlc3NTcGFjZUZpbGVUeXBlSW5zdGFuY2UBAEstAQBLLUst" +
           "AAD/////CwAAABVgiQoCAAAAAAAEAAAAU2l6ZQEATC0ALgBETC0AAAAJ/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACAAAAFdyaXRhYmxlAQCQMQAuAESQMQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAMAAAA" +
           "VXNlcldyaXRhYmxlAQCRMQAuAESRMQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAT3BlbkNv" +
           "dW50AQBPLQAuAERPLQAAAAX/////AQH/////AAAAAARhggoEAAAAAAAEAAAAT3BlbgEAUC0ALwEAPC1Q" +
           "LQAAAQH/////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAFEtAC4ARFEtAACWAQAAAAEA" +
           "KgEBEwAAAAQAAABNb2RlAAP/////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAABdgqQoCAAAA" +
           "AAAPAAAAT3V0cHV0QXJndW1lbnRzAQBSLQAuAERSLQAAlgEAAAABACoBARkAAAAKAAAARmlsZUhhbmRs" +
           "ZQAH/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAAAAAABQAAAENsb3NlAQBT" +
           "LQAvAQA/LVMtAAABAf////8BAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEAVC0ALgBEVC0A" +
           "AJYBAAAAAQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf//" +
           "//8AAAAABGGCCgQAAAAAAAQAAABSZWFkAQBVLQAvAQBBLVUtAAABAf////8CAAAAF2CpCgIAAAAAAA4A" +
           "AABJbnB1dEFyZ3VtZW50cwEAVi0ALgBEVi0AAJYCAAAAAQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB///" +
           "//8AAAAAAAEAKgEBFQAAAAYAAABMZW5ndGgABv////8AAAAAAAEAKAEBAAAAAQAAAAIAAAABAf////8A" +
           "AAAAF2CpCgIAAAAAAA8AAABPdXRwdXRBcmd1bWVudHMBAFctAC4ARFctAACWAQAAAAEAKgEBEwAAAAQA" +
           "AABEYXRhAA//////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAAARhggoEAAAAAAAFAAAAV3Jp" +
           "dGUBAFgtAC8BAEQtWC0AAAEB/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBZLQAu" +
           "AERZLQAAlgIAAAABACoBARkAAAAKAAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAqAQETAAAABAAAAERh" +
           "dGEAD/////8AAAAAAAEAKAEBAAAAAQAAAAIAAAABAf////8AAAAABGGCCgQAAAAAAAsAAABHZXRQb3Np" +
           "dGlvbgEAWi0ALwEARi1aLQAAAQH/////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAFst" +
           "AC4ARFstAACWAQAAAAEAKgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////AAAAAAABACgBAQAAAAEAAAAB" +
           "AAAAAQH/////AAAAABdgqQoCAAAAAAAPAAAAT3V0cHV0QXJndW1lbnRzAQBcLQAuAERcLQAAlgEAAAAB" +
           "ACoBARcAAAAIAAAAUG9zaXRpb24ACf////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAABGGC" +
           "CgQAAAAAAAsAAABTZXRQb3NpdGlvbgEAXS0ALwEASS1dLQAAAQH/////AQAAABdgqQoCAAAAAAAOAAAA" +
           "SW5wdXRBcmd1bWVudHMBAF4tAC4ARF4tAACWAgAAAAEAKgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////" +
           "AAAAAAABACoBARcAAAAIAAAAUG9zaXRpb24ACf////8AAAAAAAEAKAEBAAAAAQAAAAIAAAABAf////8A" +
           "AAAABGGCCgQAAAAAAA8AAABFeHBvcnROYW1lc3BhY2UBAF8tAC8BAF8tXy0AAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public MethodState ExportNamespace
        {
            get
            {
                return m_exportNamespaceMethod;
            }

            set
            {
                if (!Object.ReferenceEquals(m_exportNamespaceMethod, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_exportNamespaceMethod = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_exportNamespaceMethod != null)
            {
                children.Add(m_exportNamespaceMethod);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ExportNamespace:
                {
                    if (createOrReplace)
                    {
                        if (ExportNamespace == null)
                        {
                            if (replacement == null)
                            {
                                ExportNamespace = new MethodState(this);
                            }
                            else
                            {
                                ExportNamespace = (MethodState)replacement;
                            }
                        }
                    }

                    instance = ExportNamespace;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private MethodState m_exportNamespaceMethod;
        #endregion
    }
    #endif
    #endregion

    #region NamespaceMetadataState Class
    #if (!OPCUA_EXCLUDE_NamespaceMetadataState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class NamespaceMetadataState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public NamespaceMetadataState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.NamespaceMetadataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (NamespaceFile != null)
            {
                NamespaceFile.Initialize(context, NamespaceFile_InitializationString);
            }

            if (DefaultRolePermissions != null)
            {
                DefaultRolePermissions.Initialize(context, DefaultRolePermissions_InitializationString);
            }

            if (DefaultUserRolePermissions != null)
            {
                DefaultUserRolePermissions.Initialize(context, DefaultUserRolePermissions_InitializationString);
            }

            if (DefaultAccessRestrictions != null)
            {
                DefaultAccessRestrictions.Initialize(context, DefaultAccessRestrictions_InitializationString);
            }

            if (ConfigurationVersion != null)
            {
                ConfigurationVersion.Initialize(context, ConfigurationVersion_InitializationString);
            }
        }

        #region Initialization String
        private const string NamespaceFile_InitializationString =
           "//////////8EYIAKAQAAAAAADQAAAE5hbWVzcGFjZUZpbGUBAGgtAC8BAEstaC0AAP////8KAAAAFWCJ" +
           "CgIAAAAAAAQAAABTaXplAQBpLQAuAERpLQAAAAn/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAV3Jp" +
           "dGFibGUBAJIxAC4ARJIxAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABVc2VyV3JpdGFibGUB" +
           "AJMxAC4ARJMxAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABPcGVuQ291bnQBAGwtAC4ARGwt" +
           "AAAABf////8BAf////8AAAAABGGCCgQAAAAAAAQAAABPcGVuAQBtLQAvAQA8LW0tAAABAf////8CAAAA" +
           "F2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEAbi0ALgBEbi0AAJYBAAAAAQAqAQETAAAABAAAAE1v" +
           "ZGUAA/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAAF2CpCgIAAAAAAA8AAABPdXRwdXRB" +
           "cmd1bWVudHMBAG8tAC4ARG8tAACWAQAAAAEAKgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////AAAAAAAB" +
           "ACgBAQAAAAEAAAABAAAAAQH/////AAAAAARhggoEAAAAAAAFAAAAQ2xvc2UBAHAtAC8BAD8tcC0AAAEB" +
           "/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBxLQAuAERxLQAAlgEAAAABACoBARkA" +
           "AAAKAAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAA" +
           "AAAABAAAAFJlYWQBAHItAC8BAEEtci0AAAEB/////wIAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1l" +
           "bnRzAQBzLQAuAERzLQAAlgIAAAABACoBARkAAAAKAAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAqAQEV" +
           "AAAABgAAAExlbmd0aAAG/////wAAAAAAAQAoAQEAAAABAAAAAgAAAAEB/////wAAAAAXYKkKAgAAAAAA" +
           "DwAAAE91dHB1dEFyZ3VtZW50cwEAdC0ALgBEdC0AAJYBAAAAAQAqAQETAAAABAAAAERhdGEAD/////8A" +
           "AAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAABGGCCgQAAAAAAAUAAABXcml0ZQEAdS0ALwEARC11" +
           "LQAAAQH/////AQAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAHYtAC4ARHYtAACWAgAAAAEA" +
           "KgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////AAAAAAABACoBARMAAAAEAAAARGF0YQAP/////wAAAAAA" +
           "AQAoAQEAAAABAAAAAgAAAAEB/////wAAAAAEYYIKBAAAAAAACwAAAEdldFBvc2l0aW9uAQB3LQAvAQBG" +
           "LXctAAABAf////8CAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEAeC0ALgBEeC0AAJYBAAAA" +
           "AQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAA" +
           "F2CpCgIAAAAAAA8AAABPdXRwdXRBcmd1bWVudHMBAHktAC4ARHktAACWAQAAAAEAKgEBFwAAAAgAAABQ" +
           "b3NpdGlvbgAJ/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAEYYIKBAAAAAAACwAAAFNl" +
           "dFBvc2l0aW9uAQB6LQAvAQBJLXotAAABAf////8BAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50" +
           "cwEAey0ALgBEey0AAJYCAAAAAQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8AAAAAAAEAKgEBFwAA" +
           "AAgAAABQb3NpdGlvbgAJ/////wAAAAAAAQAoAQEAAAABAAAAAgAAAAEB/////wAAAAA=";

        private const string DefaultRolePermissions_InitializationString =
           "//////////8XYIkKAgAAAAAAFgAAAERlZmF1bHRSb2xlUGVybWlzc2lvbnMBAAk/AC4ARAk/AAAAYAEA" +
           "AAABAAAAAAAAAAEB/////wAAAAA=";

        private const string DefaultUserRolePermissions_InitializationString =
           "//////////8XYIkKAgAAAAAAGgAAAERlZmF1bHRVc2VyUm9sZVBlcm1pc3Npb25zAQAKPwAuAEQKPwAA" +
           "AGABAAAAAQAAAAAAAAABAf////8AAAAA";

        private const string DefaultAccessRestrictions_InitializationString =
           "//////////8VYIkKAgAAAAAAGQAAAERlZmF1bHRBY2Nlc3NSZXN0cmljdGlvbnMBAAs/AC4ARAs/AAAA" +
           "X/////8BAf////8AAAAA";

        private const string ConfigurationVersion_InitializationString =
           "//////////8VYIkKAgAAAAAAFAAAAENvbmZpZ3VyYXRpb25WZXJzaW9uAQCzYgAuAESzYgAAAQAGUv//" +
           "//8BAf////8AAAAA";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHQAAAE5hbWVzcGFjZU1ldGFkYXRhVHlwZUluc3RhbmNlAQBgLQEAYC1g" +
           "LQAA/////wwAAAAVYIkKAgAAAAAADAAAAE5hbWVzcGFjZVVyaQEAYS0ALgBEYS0AAAAM/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAEAAAAE5hbWVzcGFjZVZlcnNpb24BAGItAC4ARGItAAAADP////8BAf////8A" +
           "AAAAFWCJCgIAAAAAABgAAABOYW1lc3BhY2VQdWJsaWNhdGlvbkRhdGUBAGMtAC4ARGMtAAAADf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAABEAAABJc05hbWVzcGFjZVN1YnNldAEAZC0ALgBEZC0AAAAB/////wEB" +
           "/////wAAAAAXYIkKAgAAAAAAEQAAAFN0YXRpY05vZGVJZFR5cGVzAQBlLQAuAERlLQAAAQAAAQEAAAAB" +
           "AAAAAAAAAAEB/////wAAAAAXYIkKAgAAAAAAGAAAAFN0YXRpY051bWVyaWNOb2RlSWRSYW5nZQEAZi0A" +
           "LgBEZi0AAAEAIwEBAAAAAQAAAAAAAAABAf////8AAAAAFWCJCgIAAAAAABkAAABTdGF0aWNTdHJpbmdO" +
           "b2RlSWRQYXR0ZXJuAQBnLQAuAERnLQAAAAz/////AQH/////AAAAAARggAoBAAAAAAANAAAATmFtZXNw" +
           "YWNlRmlsZQEAaC0ALwEASy1oLQAA/////woAAAAVYIkKAgAAAAAABAAAAFNpemUBAGktAC4ARGktAAAA" +
           "Cf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABXcml0YWJsZQEAkjEALgBEkjEAAAAB/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAADAAAAFVzZXJXcml0YWJsZQEAkzEALgBEkzEAAAAB/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACQAAAE9wZW5Db3VudAEAbC0ALgBEbC0AAAAF/////wEB/////wAAAAAEYYIKBAAAAAAA" +
           "BAAAAE9wZW4BAG0tAC8BADwtbS0AAAEB/////wIAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRz" +
           "AQBuLQAuAERuLQAAlgEAAAABACoBARMAAAAEAAAATW9kZQAD/////wAAAAAAAQAoAQEAAAABAAAAAQAA" +
           "AAEB/////wAAAAAXYKkKAgAAAAAADwAAAE91dHB1dEFyZ3VtZW50cwEAby0ALgBEby0AAJYBAAAAAQAq" +
           "AQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8AAAAAAAEAKAEBAAAAAQAAAAEAAAABAf////8AAAAABGGC" +
           "CgQAAAAAAAUAAABDbG9zZQEAcC0ALwEAPy1wLQAAAQH/////AQAAABdgqQoCAAAAAAAOAAAASW5wdXRB" +
           "cmd1bWVudHMBAHEtAC4ARHEtAACWAQAAAAEAKgEBGQAAAAoAAABGaWxlSGFuZGxlAAf/////AAAAAAAB" +
           "ACgBAQAAAAEAAAABAAAAAQH/////AAAAAARhggoEAAAAAAAEAAAAUmVhZAEAci0ALwEAQS1yLQAAAQH/" +
           "////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAHMtAC4ARHMtAACWAgAAAAEAKgEBGQAA" +
           "AAoAAABGaWxlSGFuZGxlAAf/////AAAAAAABACoBARUAAAAGAAAATGVuZ3RoAAb/////AAAAAAABACgB" +
           "AQAAAAEAAAACAAAAAQH/////AAAAABdgqQoCAAAAAAAPAAAAT3V0cHV0QXJndW1lbnRzAQB0LQAuAER0" +
           "LQAAlgEAAAABACoBARMAAAAEAAAARGF0YQAP/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAA" +
           "AAAEYYIKBAAAAAAABQAAAFdyaXRlAQB1LQAvAQBELXUtAAABAf////8BAAAAF2CpCgIAAAAAAA4AAABJ" +
           "bnB1dEFyZ3VtZW50cwEAdi0ALgBEdi0AAJYCAAAAAQAqAQEZAAAACgAAAEZpbGVIYW5kbGUAB/////8A" +
           "AAAAAAEAKgEBEwAAAAQAAABEYXRhAA//////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAAARh" +
           "ggoEAAAAAAALAAAAR2V0UG9zaXRpb24BAHctAC8BAEYtdy0AAAEB/////wIAAAAXYKkKAgAAAAAADgAA" +
           "AElucHV0QXJndW1lbnRzAQB4LQAuAER4LQAAlgEAAAABACoBARkAAAAKAAAARmlsZUhhbmRsZQAH////" +
           "/wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAAXYKkKAgAAAAAADwAAAE91dHB1dEFyZ3VtZW50" +
           "cwEAeS0ALgBEeS0AAJYBAAAAAQAqAQEXAAAACAAAAFBvc2l0aW9uAAn/////AAAAAAABACgBAQAAAAEA" +
           "AAABAAAAAQH/////AAAAAARhggoEAAAAAAALAAAAU2V0UG9zaXRpb24BAHotAC8BAEktei0AAAEB////" +
           "/wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQB7LQAuAER7LQAAlgIAAAABACoBARkAAAAK" +
           "AAAARmlsZUhhbmRsZQAH/////wAAAAAAAQAqAQEXAAAACAAAAFBvc2l0aW9uAAn/////AAAAAAABACgB" +
           "AQAAAAEAAAACAAAAAQH/////AAAAABdgiQoCAAAAAAAWAAAARGVmYXVsdFJvbGVQZXJtaXNzaW9ucwEA" +
           "CT8ALgBECT8AAABgAQAAAAEAAAAAAAAAAQH/////AAAAABdgiQoCAAAAAAAaAAAARGVmYXVsdFVzZXJS" +
           "b2xlUGVybWlzc2lvbnMBAAo/AC4ARAo/AAAAYAEAAAABAAAAAAAAAAEB/////wAAAAAVYIkKAgAAAAAA" +
           "GQAAAERlZmF1bHRBY2Nlc3NSZXN0cmljdGlvbnMBAAs/AC4ARAs/AAAAX/////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAABQAAABDb25maWd1cmF0aW9uVmVyc2lvbgEAs2IALgBEs2IAAAEABlL/////AQH/////AAAA" +
           "AA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> NamespaceUri
        {
            get
            {
                return m_namespaceUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespaceUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespaceUri = value;
            }
        }

        /// <remarks />
        public PropertyState<string> NamespaceVersion
        {
            get
            {
                return m_namespaceVersion;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespaceVersion, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespaceVersion = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> NamespacePublicationDate
        {
            get
            {
                return m_namespacePublicationDate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespacePublicationDate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespacePublicationDate = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> IsNamespaceSubset
        {
            get
            {
                return m_isNamespaceSubset;
            }

            set
            {
                if (!Object.ReferenceEquals(m_isNamespaceSubset, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_isNamespaceSubset = value;
            }
        }

        /// <remarks />
        public PropertyState<IdType[]> StaticNodeIdTypes
        {
            get
            {
                return m_staticNodeIdTypes;
            }

            set
            {
                if (!Object.ReferenceEquals(m_staticNodeIdTypes, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_staticNodeIdTypes = value;
            }
        }

        /// <remarks />
        public PropertyState<string[]> StaticNumericNodeIdRange
        {
            get
            {
                return m_staticNumericNodeIdRange;
            }

            set
            {
                if (!Object.ReferenceEquals(m_staticNumericNodeIdRange, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_staticNumericNodeIdRange = value;
            }
        }

        /// <remarks />
        public PropertyState<string> StaticStringNodeIdPattern
        {
            get
            {
                return m_staticStringNodeIdPattern;
            }

            set
            {
                if (!Object.ReferenceEquals(m_staticStringNodeIdPattern, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_staticStringNodeIdPattern = value;
            }
        }

        /// <remarks />
        public AddressSpaceFileState NamespaceFile
        {
            get
            {
                return m_namespaceFile;
            }

            set
            {
                if (!Object.ReferenceEquals(m_namespaceFile, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_namespaceFile = value;
            }
        }

        /// <remarks />
        public PropertyState<RolePermissionType[]> DefaultRolePermissions
        {
            get
            {
                return m_defaultRolePermissions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_defaultRolePermissions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_defaultRolePermissions = value;
            }
        }

        /// <remarks />
        public PropertyState<RolePermissionType[]> DefaultUserRolePermissions
        {
            get
            {
                return m_defaultUserRolePermissions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_defaultUserRolePermissions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_defaultUserRolePermissions = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> DefaultAccessRestrictions
        {
            get
            {
                return m_defaultAccessRestrictions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_defaultAccessRestrictions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_defaultAccessRestrictions = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> ConfigurationVersion
        {
            get
            {
                return m_configurationVersion;
            }

            set
            {
                if (!Object.ReferenceEquals(m_configurationVersion, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_configurationVersion = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_namespaceUri != null)
            {
                children.Add(m_namespaceUri);
            }

            if (m_namespaceVersion != null)
            {
                children.Add(m_namespaceVersion);
            }

            if (m_namespacePublicationDate != null)
            {
                children.Add(m_namespacePublicationDate);
            }

            if (m_isNamespaceSubset != null)
            {
                children.Add(m_isNamespaceSubset);
            }

            if (m_staticNodeIdTypes != null)
            {
                children.Add(m_staticNodeIdTypes);
            }

            if (m_staticNumericNodeIdRange != null)
            {
                children.Add(m_staticNumericNodeIdRange);
            }

            if (m_staticStringNodeIdPattern != null)
            {
                children.Add(m_staticStringNodeIdPattern);
            }

            if (m_namespaceFile != null)
            {
                children.Add(m_namespaceFile);
            }

            if (m_defaultRolePermissions != null)
            {
                children.Add(m_defaultRolePermissions);
            }

            if (m_defaultUserRolePermissions != null)
            {
                children.Add(m_defaultUserRolePermissions);
            }

            if (m_defaultAccessRestrictions != null)
            {
                children.Add(m_defaultAccessRestrictions);
            }

            if (m_configurationVersion != null)
            {
                children.Add(m_configurationVersion);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.NamespaceUri:
                {
                    if (createOrReplace)
                    {
                        if (NamespaceUri == null)
                        {
                            if (replacement == null)
                            {
                                NamespaceUri = new PropertyState<string>(this);
                            }
                            else
                            {
                                NamespaceUri = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = NamespaceUri;
                    break;
                }

                case Opc.Ua.BrowseNames.NamespaceVersion:
                {
                    if (createOrReplace)
                    {
                        if (NamespaceVersion == null)
                        {
                            if (replacement == null)
                            {
                                NamespaceVersion = new PropertyState<string>(this);
                            }
                            else
                            {
                                NamespaceVersion = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = NamespaceVersion;
                    break;
                }

                case Opc.Ua.BrowseNames.NamespacePublicationDate:
                {
                    if (createOrReplace)
                    {
                        if (NamespacePublicationDate == null)
                        {
                            if (replacement == null)
                            {
                                NamespacePublicationDate = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                NamespacePublicationDate = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = NamespacePublicationDate;
                    break;
                }

                case Opc.Ua.BrowseNames.IsNamespaceSubset:
                {
                    if (createOrReplace)
                    {
                        if (IsNamespaceSubset == null)
                        {
                            if (replacement == null)
                            {
                                IsNamespaceSubset = new PropertyState<bool>(this);
                            }
                            else
                            {
                                IsNamespaceSubset = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = IsNamespaceSubset;
                    break;
                }

                case Opc.Ua.BrowseNames.StaticNodeIdTypes:
                {
                    if (createOrReplace)
                    {
                        if (StaticNodeIdTypes == null)
                        {
                            if (replacement == null)
                            {
                                StaticNodeIdTypes = new PropertyState<IdType[]>(this);
                            }
                            else
                            {
                                StaticNodeIdTypes = (PropertyState<IdType[]>)replacement;
                            }
                        }
                    }

                    instance = StaticNodeIdTypes;
                    break;
                }

                case Opc.Ua.BrowseNames.StaticNumericNodeIdRange:
                {
                    if (createOrReplace)
                    {
                        if (StaticNumericNodeIdRange == null)
                        {
                            if (replacement == null)
                            {
                                StaticNumericNodeIdRange = new PropertyState<string[]>(this);
                            }
                            else
                            {
                                StaticNumericNodeIdRange = (PropertyState<string[]>)replacement;
                            }
                        }
                    }

                    instance = StaticNumericNodeIdRange;
                    break;
                }

                case Opc.Ua.BrowseNames.StaticStringNodeIdPattern:
                {
                    if (createOrReplace)
                    {
                        if (StaticStringNodeIdPattern == null)
                        {
                            if (replacement == null)
                            {
                                StaticStringNodeIdPattern = new PropertyState<string>(this);
                            }
                            else
                            {
                                StaticStringNodeIdPattern = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = StaticStringNodeIdPattern;
                    break;
                }

                case Opc.Ua.BrowseNames.NamespaceFile:
                {
                    if (createOrReplace)
                    {
                        if (NamespaceFile == null)
                        {
                            if (replacement == null)
                            {
                                NamespaceFile = new AddressSpaceFileState(this);
                            }
                            else
                            {
                                NamespaceFile = (AddressSpaceFileState)replacement;
                            }
                        }
                    }

                    instance = NamespaceFile;
                    break;
                }

                case Opc.Ua.BrowseNames.DefaultRolePermissions:
                {
                    if (createOrReplace)
                    {
                        if (DefaultRolePermissions == null)
                        {
                            if (replacement == null)
                            {
                                DefaultRolePermissions = new PropertyState<RolePermissionType[]>(this);
                            }
                            else
                            {
                                DefaultRolePermissions = (PropertyState<RolePermissionType[]>)replacement;
                            }
                        }
                    }

                    instance = DefaultRolePermissions;
                    break;
                }

                case Opc.Ua.BrowseNames.DefaultUserRolePermissions:
                {
                    if (createOrReplace)
                    {
                        if (DefaultUserRolePermissions == null)
                        {
                            if (replacement == null)
                            {
                                DefaultUserRolePermissions = new PropertyState<RolePermissionType[]>(this);
                            }
                            else
                            {
                                DefaultUserRolePermissions = (PropertyState<RolePermissionType[]>)replacement;
                            }
                        }
                    }

                    instance = DefaultUserRolePermissions;
                    break;
                }

                case Opc.Ua.BrowseNames.DefaultAccessRestrictions:
                {
                    if (createOrReplace)
                    {
                        if (DefaultAccessRestrictions == null)
                        {
                            if (replacement == null)
                            {
                                DefaultAccessRestrictions = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                DefaultAccessRestrictions = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = DefaultAccessRestrictions;
                    break;
                }

                case Opc.Ua.BrowseNames.ConfigurationVersion:
                {
                    if (createOrReplace)
                    {
                        if (ConfigurationVersion == null)
                        {
                            if (replacement == null)
                            {
                                ConfigurationVersion = new PropertyState<uint>(this);
                            }
                            else
                            {
                                ConfigurationVersion = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = ConfigurationVersion;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_namespaceUri;
        private PropertyState<string> m_namespaceVersion;
        private PropertyState<DateTime> m_namespacePublicationDate;
        private PropertyState<bool> m_isNamespaceSubset;
        private PropertyState<IdType[]> m_staticNodeIdTypes;
        private PropertyState<string[]> m_staticNumericNodeIdRange;
        private PropertyState<string> m_staticStringNodeIdPattern;
        private AddressSpaceFileState m_namespaceFile;
        private PropertyState<RolePermissionType[]> m_defaultRolePermissions;
        private PropertyState<RolePermissionType[]> m_defaultUserRolePermissions;
        private PropertyState<ushort> m_defaultAccessRestrictions;
        private PropertyState<uint> m_configurationVersion;
        #endregion
    }
    #endif
    #endregion

    #region NamespacesState Class
    #if (!OPCUA_EXCLUDE_NamespacesState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class NamespacesState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public NamespacesState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.NamespacesType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFgAAAE5hbWVzcGFjZXNUeXBlSW5zdGFuY2UBAH0tAQB9LX0tAAD/////" +
           "AQAAAARgwAoBAAAAHwAAAE5hbWVzcGFjZUlkZW50aWZpZXJfUGxhY2Vob2xkZXIAABUAAAA8TmFtZXNw" +
           "YWNlSWRlbnRpZmllcj4BAH4tAC8BAGAtfi0AAP////8HAAAAFWCJCgIAAAAAAAwAAABOYW1lc3BhY2VV" +
           "cmkBAH8tAC4ARH8tAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABAAAABOYW1lc3BhY2VWZXJzaW9u" +
           "AQCALQAuAESALQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAYAAAATmFtZXNwYWNlUHVibGljYXRp" +
           "b25EYXRlAQCBLQAuAESBLQAAAA3/////AQH/////AAAAABVgiQoCAAAAAAARAAAASXNOYW1lc3BhY2VT" +
           "dWJzZXQBAIItAC4ARIItAAAAAf////8BAf////8AAAAAF2CJCgIAAAAAABEAAABTdGF0aWNOb2RlSWRU" +
           "eXBlcwEAgy0ALgBEgy0AAAEAAAEBAAAAAQAAAAAAAAABAf////8AAAAAF2CJCgIAAAAAABgAAABTdGF0" +
           "aWNOdW1lcmljTm9kZUlkUmFuZ2UBAIQtAC4ARIQtAAABACMBAQAAAAEAAAAAAAAAAQH/////AAAAABVg" +
           "iQoCAAAAAAAZAAAAU3RhdGljU3RyaW5nTm9kZUlkUGF0dGVybgEAhS0ALgBEhS0AAAAM/////wEB////" +
           "/wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region BaseEventState Class
    #if (!OPCUA_EXCLUDE_BaseEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class BaseEventState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public BaseEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.BaseEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (LocalTime != null)
            {
                LocalTime.Initialize(context, LocalTime_InitializationString);
            }

            if (ConditionClassId != null)
            {
                ConditionClassId.Initialize(context, ConditionClassId_InitializationString);
            }

            if (ConditionClassName != null)
            {
                ConditionClassName.Initialize(context, ConditionClassName_InitializationString);
            }

            if (ConditionSubClassId != null)
            {
                ConditionSubClassId.Initialize(context, ConditionSubClassId_InitializationString);
            }

            if (ConditionSubClassName != null)
            {
                ConditionSubClassName.Initialize(context, ConditionSubClassName_InitializationString);
            }
        }

        #region Initialization String
        private const string LocalTime_InitializationString =
           "//////////8VYIkKAgAAAAAACQAAAExvY2FsVGltZQEAdgwALgBEdgwAAAEA0CL/////AQH/////AAAA" +
           "AA==";

        private const string ConditionClassId_InitializationString =
           "//////////8VYIkKAgAAAAAAEAAAAENvbmRpdGlvbkNsYXNzSWQBABt8AC4ARBt8AAAAEf////8BAf//" +
           "//8AAAAA";

        private const string ConditionClassName_InitializationString =
           "//////////8VYIkKAgAAAAAAEgAAAENvbmRpdGlvbkNsYXNzTmFtZQEAHHwALgBEHHwAAAAV/////wEB" +
           "/////wAAAAA=";

        private const string ConditionSubClassId_InitializationString =
           "//////////8XYIkKAgAAAAAAEwAAAENvbmRpdGlvblN1YkNsYXNzSWQBAB18AC4ARB18AAAAEQEAAAAB" +
           "AAAAAAAAAAEB/////wAAAAA=";

        private const string ConditionSubClassName_InitializationString =
           "//////////8XYIkKAgAAAAAAFQAAAENvbmRpdGlvblN1YkNsYXNzTmFtZQEAHnwALgBEHnwAAAAVAQAA" +
           "AAEAAAAAAAAAAQH/////AAAAAA==";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFQAAAEJhc2VFdmVudFR5cGVJbnN0YW5jZQEA+QcBAPkH+QcAAP////8N" +
           "AAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQD6BwAuAET6BwAAAA//////AQH/////AAAAABVgiQoCAAAA" +
           "AAAJAAAARXZlbnRUeXBlAQD7BwAuAET7BwAAABH/////AQH/////AAAAABVgiQoCAAAAAAAKAAAAU291" +
           "cmNlTm9kZQEA/AcALgBE/AcAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFNvdXJjZU5hbWUB" +
           "AP0HAC4ARP0HAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQD+BwAuAET+BwAAAQAm" +
           "Af////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEA/wcALgBE/wcAAAEAJgH/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAJAAAATG9jYWxUaW1lAQB2DAAuAER2DAAAAQDQIv////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQACCAAuAEQCCAAAABX/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAIAAAAU2V2ZXJpdHkBAAMIAC4ARAMIAAAABf////8BAf////8AAAAAFWCJCgIAAAAAABAAAABDb25k" +
           "aXRpb25DbGFzc0lkAQAbfAAuAEQbfAAAABH/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ29uZGl0" +
           "aW9uQ2xhc3NOYW1lAQAcfAAuAEQcfAAAABX/////AQH/////AAAAABdgiQoCAAAAAAATAAAAQ29uZGl0" +
           "aW9uU3ViQ2xhc3NJZAEAHXwALgBEHXwAAAARAQAAAAEAAAAAAAAAAQH/////AAAAABdgiQoCAAAAAAAV" +
           "AAAAQ29uZGl0aW9uU3ViQ2xhc3NOYW1lAQAefAAuAEQefAAAABUBAAAAAQAAAAAAAAABAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<byte[]> EventId
        {
            get
            {
                return m_eventId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_eventId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_eventId = value;
            }
        }

        /// <remarks />
        public PropertyState<NodeId> EventType
        {
            get
            {
                return m_eventType;
            }

            set
            {
                if (!Object.ReferenceEquals(m_eventType, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_eventType = value;
            }
        }

        /// <remarks />
        public PropertyState<NodeId> SourceNode
        {
            get
            {
                return m_sourceNode;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sourceNode, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sourceNode = value;
            }
        }

        /// <remarks />
        public PropertyState<string> SourceName
        {
            get
            {
                return m_sourceName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sourceName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sourceName = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> Time
        {
            get
            {
                return m_time;
            }

            set
            {
                if (!Object.ReferenceEquals(m_time, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_time = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> ReceiveTime
        {
            get
            {
                return m_receiveTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_receiveTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_receiveTime = value;
            }
        }

        /// <remarks />
        public PropertyState<TimeZoneDataType> LocalTime
        {
            get
            {
                return m_localTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_localTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_localTime = value;
            }
        }

        /// <remarks />
        public PropertyState<LocalizedText> Message
        {
            get
            {
                return m_message;
            }

            set
            {
                if (!Object.ReferenceEquals(m_message, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_message = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> Severity
        {
            get
            {
                return m_severity;
            }

            set
            {
                if (!Object.ReferenceEquals(m_severity, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_severity = value;
            }
        }

        /// <remarks />
        public PropertyState<NodeId> ConditionClassId
        {
            get
            {
                return m_conditionClassId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_conditionClassId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_conditionClassId = value;
            }
        }

        /// <remarks />
        public PropertyState<LocalizedText> ConditionClassName
        {
            get
            {
                return m_conditionClassName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_conditionClassName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_conditionClassName = value;
            }
        }

        /// <remarks />
        public PropertyState<NodeId[]> ConditionSubClassId
        {
            get
            {
                return m_conditionSubClassId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_conditionSubClassId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_conditionSubClassId = value;
            }
        }

        /// <remarks />
        public PropertyState<LocalizedText[]> ConditionSubClassName
        {
            get
            {
                return m_conditionSubClassName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_conditionSubClassName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_conditionSubClassName = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_eventId != null)
            {
                children.Add(m_eventId);
            }

            if (m_eventType != null)
            {
                children.Add(m_eventType);
            }

            if (m_sourceNode != null)
            {
                children.Add(m_sourceNode);
            }

            if (m_sourceName != null)
            {
                children.Add(m_sourceName);
            }

            if (m_time != null)
            {
                children.Add(m_time);
            }

            if (m_receiveTime != null)
            {
                children.Add(m_receiveTime);
            }

            if (m_localTime != null)
            {
                children.Add(m_localTime);
            }

            if (m_message != null)
            {
                children.Add(m_message);
            }

            if (m_severity != null)
            {
                children.Add(m_severity);
            }

            if (m_conditionClassId != null)
            {
                children.Add(m_conditionClassId);
            }

            if (m_conditionClassName != null)
            {
                children.Add(m_conditionClassName);
            }

            if (m_conditionSubClassId != null)
            {
                children.Add(m_conditionSubClassId);
            }

            if (m_conditionSubClassName != null)
            {
                children.Add(m_conditionSubClassName);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.EventId:
                {
                    if (createOrReplace)
                    {
                        if (EventId == null)
                        {
                            if (replacement == null)
                            {
                                EventId = new PropertyState<byte[]>(this);
                            }
                            else
                            {
                                EventId = (PropertyState<byte[]>)replacement;
                            }
                        }
                    }

                    instance = EventId;
                    break;
                }

                case Opc.Ua.BrowseNames.EventType:
                {
                    if (createOrReplace)
                    {
                        if (EventType == null)
                        {
                            if (replacement == null)
                            {
                                EventType = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                EventType = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = EventType;
                    break;
                }

                case Opc.Ua.BrowseNames.SourceNode:
                {
                    if (createOrReplace)
                    {
                        if (SourceNode == null)
                        {
                            if (replacement == null)
                            {
                                SourceNode = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                SourceNode = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = SourceNode;
                    break;
                }

                case Opc.Ua.BrowseNames.SourceName:
                {
                    if (createOrReplace)
                    {
                        if (SourceName == null)
                        {
                            if (replacement == null)
                            {
                                SourceName = new PropertyState<string>(this);
                            }
                            else
                            {
                                SourceName = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = SourceName;
                    break;
                }

                case Opc.Ua.BrowseNames.Time:
                {
                    if (createOrReplace)
                    {
                        if (Time == null)
                        {
                            if (replacement == null)
                            {
                                Time = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                Time = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = Time;
                    break;
                }

                case Opc.Ua.BrowseNames.ReceiveTime:
                {
                    if (createOrReplace)
                    {
                        if (ReceiveTime == null)
                        {
                            if (replacement == null)
                            {
                                ReceiveTime = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                ReceiveTime = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = ReceiveTime;
                    break;
                }

                case Opc.Ua.BrowseNames.LocalTime:
                {
                    if (createOrReplace)
                    {
                        if (LocalTime == null)
                        {
                            if (replacement == null)
                            {
                                LocalTime = new PropertyState<TimeZoneDataType>(this);
                            }
                            else
                            {
                                LocalTime = (PropertyState<TimeZoneDataType>)replacement;
                            }
                        }
                    }

                    instance = LocalTime;
                    break;
                }

                case Opc.Ua.BrowseNames.Message:
                {
                    if (createOrReplace)
                    {
                        if (Message == null)
                        {
                            if (replacement == null)
                            {
                                Message = new PropertyState<LocalizedText>(this);
                            }
                            else
                            {
                                Message = (PropertyState<LocalizedText>)replacement;
                            }
                        }
                    }

                    instance = Message;
                    break;
                }

                case Opc.Ua.BrowseNames.Severity:
                {
                    if (createOrReplace)
                    {
                        if (Severity == null)
                        {
                            if (replacement == null)
                            {
                                Severity = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                Severity = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = Severity;
                    break;
                }

                case Opc.Ua.BrowseNames.ConditionClassId:
                {
                    if (createOrReplace)
                    {
                        if (ConditionClassId == null)
                        {
                            if (replacement == null)
                            {
                                ConditionClassId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                ConditionClassId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = ConditionClassId;
                    break;
                }

                case Opc.Ua.BrowseNames.ConditionClassName:
                {
                    if (createOrReplace)
                    {
                        if (ConditionClassName == null)
                        {
                            if (replacement == null)
                            {
                                ConditionClassName = new PropertyState<LocalizedText>(this);
                            }
                            else
                            {
                                ConditionClassName = (PropertyState<LocalizedText>)replacement;
                            }
                        }
                    }

                    instance = ConditionClassName;
                    break;
                }

                case Opc.Ua.BrowseNames.ConditionSubClassId:
                {
                    if (createOrReplace)
                    {
                        if (ConditionSubClassId == null)
                        {
                            if (replacement == null)
                            {
                                ConditionSubClassId = new PropertyState<NodeId[]>(this);
                            }
                            else
                            {
                                ConditionSubClassId = (PropertyState<NodeId[]>)replacement;
                            }
                        }
                    }

                    instance = ConditionSubClassId;
                    break;
                }

                case Opc.Ua.BrowseNames.ConditionSubClassName:
                {
                    if (createOrReplace)
                    {
                        if (ConditionSubClassName == null)
                        {
                            if (replacement == null)
                            {
                                ConditionSubClassName = new PropertyState<LocalizedText[]>(this);
                            }
                            else
                            {
                                ConditionSubClassName = (PropertyState<LocalizedText[]>)replacement;
                            }
                        }
                    }

                    instance = ConditionSubClassName;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<byte[]> m_eventId;
        private PropertyState<NodeId> m_eventType;
        private PropertyState<NodeId> m_sourceNode;
        private PropertyState<string> m_sourceName;
        private PropertyState<DateTime> m_time;
        private PropertyState<DateTime> m_receiveTime;
        private PropertyState<TimeZoneDataType> m_localTime;
        private PropertyState<LocalizedText> m_message;
        private PropertyState<ushort> m_severity;
        private PropertyState<NodeId> m_conditionClassId;
        private PropertyState<LocalizedText> m_conditionClassName;
        private PropertyState<NodeId[]> m_conditionSubClassId;
        private PropertyState<LocalizedText[]> m_conditionSubClassName;
        #endregion
    }
    #endif
    #endregion

    #region AuditEventState Class
    #if (!OPCUA_EXCLUDE_AuditEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public AuditEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFgAAAEF1ZGl0RXZlbnRUeXBlSW5zdGFuY2UBAAQIAQAECAQIAAD/////" +
           "DQAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAgAwALgBEgAwAAAAP/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAACQAAAEV2ZW50VHlwZQEAgQwALgBEgQwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFNv" +
           "dXJjZU5vZGUBAIIMAC4ARIIMAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABTb3VyY2VOYW1l" +
           "AQCDDAAuAESDDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEAhAwALgBEhAwAAAEA" +
           "JgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBAIUMAC4ARIUMAAABACYB////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAIcMAC4ARIcMAAAAFf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAgAAABTZXZlcml0eQEAiAwALgBEiAwAAAAF/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "DwAAAEFjdGlvblRpbWVTdGFtcAEABQgALgBEBQgAAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAG" +
           "AAAAU3RhdHVzAQAGCAAuAEQGCAAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2VydmVySWQB" +
           "AAcIAC4ARAcIAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGllbnRBdWRpdEVudHJ5SWQB" +
           "AAgIAC4ARAgIAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGllbnRVc2VySWQBAAkIAC4A" +
           "RAkIAAAADP////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<DateTime> ActionTimeStamp
        {
            get
            {
                return m_actionTimeStamp;
            }

            set
            {
                if (!Object.ReferenceEquals(m_actionTimeStamp, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_actionTimeStamp = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> Status
        {
            get
            {
                return m_status;
            }

            set
            {
                if (!Object.ReferenceEquals(m_status, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_status = value;
            }
        }

        /// <remarks />
        public PropertyState<string> ServerId
        {
            get
            {
                return m_serverId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverId = value;
            }
        }

        /// <remarks />
        public PropertyState<string> ClientAuditEntryId
        {
            get
            {
                return m_clientAuditEntryId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientAuditEntryId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientAuditEntryId = value;
            }
        }

        /// <remarks />
        public PropertyState<string> ClientUserId
        {
            get
            {
                return m_clientUserId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientUserId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientUserId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_actionTimeStamp != null)
            {
                children.Add(m_actionTimeStamp);
            }

            if (m_status != null)
            {
                children.Add(m_status);
            }

            if (m_serverId != null)
            {
                children.Add(m_serverId);
            }

            if (m_clientAuditEntryId != null)
            {
                children.Add(m_clientAuditEntryId);
            }

            if (m_clientUserId != null)
            {
                children.Add(m_clientUserId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ActionTimeStamp:
                {
                    if (createOrReplace)
                    {
                        if (ActionTimeStamp == null)
                        {
                            if (replacement == null)
                            {
                                ActionTimeStamp = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                ActionTimeStamp = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = ActionTimeStamp;
                    break;
                }

                case Opc.Ua.BrowseNames.Status:
                {
                    if (createOrReplace)
                    {
                        if (Status == null)
                        {
                            if (replacement == null)
                            {
                                Status = new PropertyState<bool>(this);
                            }
                            else
                            {
                                Status = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = Status;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerId:
                {
                    if (createOrReplace)
                    {
                        if (ServerId == null)
                        {
                            if (replacement == null)
                            {
                                ServerId = new PropertyState<string>(this);
                            }
                            else
                            {
                                ServerId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ServerId;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientAuditEntryId:
                {
                    if (createOrReplace)
                    {
                        if (ClientAuditEntryId == null)
                        {
                            if (replacement == null)
                            {
                                ClientAuditEntryId = new PropertyState<string>(this);
                            }
                            else
                            {
                                ClientAuditEntryId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ClientAuditEntryId;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientUserId:
                {
                    if (createOrReplace)
                    {
                        if (ClientUserId == null)
                        {
                            if (replacement == null)
                            {
                                ClientUserId = new PropertyState<string>(this);
                            }
                            else
                            {
                                ClientUserId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ClientUserId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<DateTime> m_actionTimeStamp;
        private PropertyState<bool> m_status;
        private PropertyState<string> m_serverId;
        private PropertyState<string> m_clientAuditEntryId;
        private PropertyState<string> m_clientUserId;
        #endregion
    }
    #endif
    #endregion

    #region AuditSecurityEventState Class
    #if (!OPCUA_EXCLUDE_AuditSecurityEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditSecurityEventState : AuditEventState
    {
        #region Constructors
        /// <remarks />
        public AuditSecurityEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditSecurityEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (StatusCodeId != null)
            {
                StatusCodeId.Initialize(context, StatusCodeId_InitializationString);
            }
        }

        #region Initialization String
        private const string StatusCodeId_InitializationString =
           "//////////8VYIkKAgAAAAAADAAAAFN0YXR1c0NvZGVJZAEAz0QALgBEz0QAAAAT/////wEB/////wAA" +
           "AAA=";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHgAAAEF1ZGl0U2VjdXJpdHlFdmVudFR5cGVJbnN0YW5jZQEACggBAAoI" +
           "CggAAP////8OAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQCJDAAuAESJDAAAAA//////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQCKDAAuAESKDAAAABH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAKAAAAU291cmNlTm9kZQEAiwwALgBEiwwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFNv" +
           "dXJjZU5hbWUBAIwMAC4ARIwMAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQCNDAAu" +
           "AESNDAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAjgwALgBEjgwA" +
           "AAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAkAwALgBEkAwAAAAV/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQCRDAAuAESRDAAAAAX/////AQH/////AAAAABVg" +
           "iQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQCSDAAuAESSDAAAAQAmAf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAYAAABTdGF0dXMBAJMMAC4ARJMMAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABT" +
           "ZXJ2ZXJJZAEAlAwALgBElAwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1ZGl0" +
           "RW50cnlJZAEAlQwALgBElQwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVzZXJJ" +
           "ZAEAlgwALgBElgwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAFN0YXR1c0NvZGVJZAEAz0QA" +
           "LgBEz0QAAAAT/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<StatusCode> StatusCodeId
        {
            get
            {
                return m_statusCodeId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_statusCodeId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_statusCodeId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_statusCodeId != null)
            {
                children.Add(m_statusCodeId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.StatusCodeId:
                {
                    if (createOrReplace)
                    {
                        if (StatusCodeId == null)
                        {
                            if (replacement == null)
                            {
                                StatusCodeId = new PropertyState<StatusCode>(this);
                            }
                            else
                            {
                                StatusCodeId = (PropertyState<StatusCode>)replacement;
                            }
                        }
                    }

                    instance = StatusCodeId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<StatusCode> m_statusCodeId;
        #endregion
    }
    #endif
    #endregion

    #region AuditChannelEventState Class
    #if (!OPCUA_EXCLUDE_AuditChannelEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditChannelEventState : AuditSecurityEventState
    {
        #region Constructors
        /// <remarks />
        public AuditChannelEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditChannelEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHQAAAEF1ZGl0Q2hhbm5lbEV2ZW50VHlwZUluc3RhbmNlAQALCAEACwgL" +
           "CAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAJcMAC4ARJcMAAAAD/////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAJgMAC4ARJgMAAAAEf////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAoAAABTb3VyY2VOb2RlAQCZDAAuAESZDAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAKAAAAU291" +
           "cmNlTmFtZQEAmgwALgBEmgwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUBAJsMAC4A" +
           "RJsMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQCcDAAuAEScDAAA" +
           "AQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQCeDAAuAESeDAAAABX/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAJ8MAC4ARJ8MAAAABf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAKAMAC4ARKAMAAABACYB/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAABgAAAFN0YXR1cwEAoQwALgBEoQwAAAAB/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNl" +
           "cnZlcklkAQCiDAAuAESiDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xpZW50QXVkaXRF" +
           "bnRyeUlkAQCjDAAuAESjDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xpZW50VXNlcklk" +
           "AQCkDAAuAESkDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAPAAAAU2VjdXJlQ2hhbm5lbElkAQC5" +
           "CgAuAES5CgAAAAz/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> SecureChannelId
        {
            get
            {
                return m_secureChannelId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_secureChannelId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_secureChannelId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_secureChannelId != null)
            {
                children.Add(m_secureChannelId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SecureChannelId:
                {
                    if (createOrReplace)
                    {
                        if (SecureChannelId == null)
                        {
                            if (replacement == null)
                            {
                                SecureChannelId = new PropertyState<string>(this);
                            }
                            else
                            {
                                SecureChannelId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = SecureChannelId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_secureChannelId;
        #endregion
    }
    #endif
    #endregion

    #region AuditOpenSecureChannelEventState Class
    #if (!OPCUA_EXCLUDE_AuditOpenSecureChannelEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditOpenSecureChannelEventState : AuditChannelEventState
    {
        #region Constructors
        /// <remarks />
        public AuditOpenSecureChannelEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditOpenSecureChannelEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (CertificateErrorEventId != null)
            {
                CertificateErrorEventId.Initialize(context, CertificateErrorEventId_InitializationString);
            }
        }

        #region Initialization String
        private const string CertificateErrorEventId_InitializationString =
           "//////////8VYIkKAgAAAAAAFwAAAENlcnRpZmljYXRlRXJyb3JFdmVudElkAQBHXgAuAERHXgAAAAz/" +
           "////AQH/////AAAAAA==";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJwAAAEF1ZGl0T3BlblNlY3VyZUNoYW5uZWxFdmVudFR5cGVJbnN0YW5j" +
           "ZQEADAgBAAwIDAgAAP////8VAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQClDAAuAESlDAAAAA//////" +
           "AQH/////AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQCmDAAuAESmDAAAABH/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAKAAAAU291cmNlTm9kZQEApwwALgBEpwwAAAAR/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAACgAAAFNvdXJjZU5hbWUBAKgMAC4ARKgMAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABU" +
           "aW1lAQCpDAAuAESpDAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEA" +
           "qgwALgBEqgwAAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEArAwALgBErAwA" +
           "AAAV/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQCtDAAuAEStDAAAAAX/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQCuDAAuAESuDAAAAQAmAf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAYAAABTdGF0dXMBAK8MAC4ARK8MAAAAAf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAgAAABTZXJ2ZXJJZAEAsAwALgBEsAwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENs" +
           "aWVudEF1ZGl0RW50cnlJZAEAsQwALgBEsQwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENs" +
           "aWVudFVzZXJJZAEAsgwALgBEsgwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADwAAAFNlY3VyZUNo" +
           "YW5uZWxJZAEAswwALgBEswwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAENsaWVudENlcnRp" +
           "ZmljYXRlAQANCAAuAEQNCAAAAA//////AQH/////AAAAABVgiQoCAAAAAAAbAAAAQ2xpZW50Q2VydGlm" +
           "aWNhdGVUaHVtYnByaW50AQC6CgAuAES6CgAAAAz/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVx" +
           "dWVzdFR5cGUBAA4IAC4ARA4IAAABADsB/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAFNlY3VyaXR5" +
           "UG9saWN5VXJpAQAPCAAuAEQPCAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAU2VjdXJpdHlN" +
           "b2RlAQARCAAuAEQRCAAAAQAuAf////8BAf////8AAAAAFWCJCgIAAAAAABEAAABSZXF1ZXN0ZWRMaWZl" +
           "dGltZQEAEggALgBEEggAAAEAIgH/////AQH/////AAAAABVgiQoCAAAAAAAXAAAAQ2VydGlmaWNhdGVF" +
           "cnJvckV2ZW50SWQBAEdeAC4AREdeAAAADP////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<byte[]> ClientCertificate
        {
            get
            {
                return m_clientCertificate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientCertificate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientCertificate = value;
            }
        }

        /// <remarks />
        public PropertyState<string> ClientCertificateThumbprint
        {
            get
            {
                return m_clientCertificateThumbprint;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientCertificateThumbprint, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientCertificateThumbprint = value;
            }
        }

        /// <remarks />
        public PropertyState<SecurityTokenRequestType> RequestType
        {
            get
            {
                return m_requestType;
            }

            set
            {
                if (!Object.ReferenceEquals(m_requestType, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_requestType = value;
            }
        }

        /// <remarks />
        public PropertyState<string> SecurityPolicyUri
        {
            get
            {
                return m_securityPolicyUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityPolicyUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityPolicyUri = value;
            }
        }

        /// <remarks />
        public PropertyState<MessageSecurityMode> SecurityMode
        {
            get
            {
                return m_securityMode;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityMode, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityMode = value;
            }
        }

        /// <remarks />
        public PropertyState<double> RequestedLifetime
        {
            get
            {
                return m_requestedLifetime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_requestedLifetime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_requestedLifetime = value;
            }
        }

        /// <remarks />
        public PropertyState<string> CertificateErrorEventId
        {
            get
            {
                return m_certificateErrorEventId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_certificateErrorEventId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_certificateErrorEventId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_clientCertificate != null)
            {
                children.Add(m_clientCertificate);
            }

            if (m_clientCertificateThumbprint != null)
            {
                children.Add(m_clientCertificateThumbprint);
            }

            if (m_requestType != null)
            {
                children.Add(m_requestType);
            }

            if (m_securityPolicyUri != null)
            {
                children.Add(m_securityPolicyUri);
            }

            if (m_securityMode != null)
            {
                children.Add(m_securityMode);
            }

            if (m_requestedLifetime != null)
            {
                children.Add(m_requestedLifetime);
            }

            if (m_certificateErrorEventId != null)
            {
                children.Add(m_certificateErrorEventId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ClientCertificate:
                {
                    if (createOrReplace)
                    {
                        if (ClientCertificate == null)
                        {
                            if (replacement == null)
                            {
                                ClientCertificate = new PropertyState<byte[]>(this);
                            }
                            else
                            {
                                ClientCertificate = (PropertyState<byte[]>)replacement;
                            }
                        }
                    }

                    instance = ClientCertificate;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientCertificateThumbprint:
                {
                    if (createOrReplace)
                    {
                        if (ClientCertificateThumbprint == null)
                        {
                            if (replacement == null)
                            {
                                ClientCertificateThumbprint = new PropertyState<string>(this);
                            }
                            else
                            {
                                ClientCertificateThumbprint = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ClientCertificateThumbprint;
                    break;
                }

                case Opc.Ua.BrowseNames.RequestType:
                {
                    if (createOrReplace)
                    {
                        if (RequestType == null)
                        {
                            if (replacement == null)
                            {
                                RequestType = new PropertyState<SecurityTokenRequestType>(this);
                            }
                            else
                            {
                                RequestType = (PropertyState<SecurityTokenRequestType>)replacement;
                            }
                        }
                    }

                    instance = RequestType;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityPolicyUri:
                {
                    if (createOrReplace)
                    {
                        if (SecurityPolicyUri == null)
                        {
                            if (replacement == null)
                            {
                                SecurityPolicyUri = new PropertyState<string>(this);
                            }
                            else
                            {
                                SecurityPolicyUri = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = SecurityPolicyUri;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityMode:
                {
                    if (createOrReplace)
                    {
                        if (SecurityMode == null)
                        {
                            if (replacement == null)
                            {
                                SecurityMode = new PropertyState<MessageSecurityMode>(this);
                            }
                            else
                            {
                                SecurityMode = (PropertyState<MessageSecurityMode>)replacement;
                            }
                        }
                    }

                    instance = SecurityMode;
                    break;
                }

                case Opc.Ua.BrowseNames.RequestedLifetime:
                {
                    if (createOrReplace)
                    {
                        if (RequestedLifetime == null)
                        {
                            if (replacement == null)
                            {
                                RequestedLifetime = new PropertyState<double>(this);
                            }
                            else
                            {
                                RequestedLifetime = (PropertyState<double>)replacement;
                            }
                        }
                    }

                    instance = RequestedLifetime;
                    break;
                }

                case Opc.Ua.BrowseNames.CertificateErrorEventId:
                {
                    if (createOrReplace)
                    {
                        if (CertificateErrorEventId == null)
                        {
                            if (replacement == null)
                            {
                                CertificateErrorEventId = new PropertyState<string>(this);
                            }
                            else
                            {
                                CertificateErrorEventId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = CertificateErrorEventId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<byte[]> m_clientCertificate;
        private PropertyState<string> m_clientCertificateThumbprint;
        private PropertyState<SecurityTokenRequestType> m_requestType;
        private PropertyState<string> m_securityPolicyUri;
        private PropertyState<MessageSecurityMode> m_securityMode;
        private PropertyState<double> m_requestedLifetime;
        private PropertyState<string> m_certificateErrorEventId;
        #endregion
    }
    #endif
    #endregion

    #region AuditSessionEventState Class
    #if (!OPCUA_EXCLUDE_AuditSessionEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditSessionEventState : AuditSecurityEventState
    {
        #region Constructors
        /// <remarks />
        public AuditSessionEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditSessionEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHQAAAEF1ZGl0U2Vzc2lvbkV2ZW50VHlwZUluc3RhbmNlAQAVCAEAFQgV" +
           "CAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBALQMAC4ARLQMAAAAD/////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAkAAABFdmVudFR5cGUBALUMAC4ARLUMAAAAEf////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAoAAABTb3VyY2VOb2RlAQC2DAAuAES2DAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAKAAAAU291" +
           "cmNlTmFtZQEAtwwALgBEtwwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUBALgMAC4A" +
           "RLgMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQC5DAAuAES5DAAA" +
           "AQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQC7DAAuAES7DAAAABX/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBALwMAC4ARLwMAAAABf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAL0MAC4ARL0MAAABACYB/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAABgAAAFN0YXR1cwEAvgwALgBEvgwAAAAB/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNl" +
           "cnZlcklkAQC/DAAuAES/DAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xpZW50QXVkaXRF" +
           "bnRyeUlkAQDADAAuAETADAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xpZW50VXNlcklk" +
           "AQDBDAAuAETBDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAU2Vzc2lvbklkAQAWCAAuAEQW" +
           "CAAAABH/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<NodeId> SessionId
        {
            get
            {
                return m_sessionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionId != null)
            {
                children.Add(m_sessionId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionId:
                {
                    if (createOrReplace)
                    {
                        if (SessionId == null)
                        {
                            if (replacement == null)
                            {
                                SessionId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                SessionId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = SessionId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<NodeId> m_sessionId;
        #endregion
    }
    #endif
    #endregion

    #region AuditCreateSessionEventState Class
    #if (!OPCUA_EXCLUDE_AuditCreateSessionEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCreateSessionEventState : AuditSessionEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCreateSessionEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCreateSessionEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAEF1ZGl0Q3JlYXRlU2Vzc2lvbkV2ZW50VHlwZUluc3RhbmNlAQAX" +
           "CAEAFwgXCAAA/////xIAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAMIMAC4ARMIMAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAMMMAC4ARMMMAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQDEDAAuAETEDAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAxQwALgBExQwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "AMYMAC4ARMYMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQDHDAAu" +
           "AETHDAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQDJDAAuAETJDAAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAMoMAC4ARMoMAAAABf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAMsMAC4ARMsMAAABACYB/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAABgAAAFN0YXR1cwEAzAwALgBEzAwAAAAB/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CAAAAFNlcnZlcklkAQDNDAAuAETNDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xpZW50" +
           "QXVkaXRFbnRyeUlkAQDODAAuAETODAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xpZW50" +
           "VXNlcklkAQDPDAAuAETPDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAJAAAAU2Vzc2lvbklkAQBN" +
           "OAAuAERNOAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAPAAAAU2VjdXJlQ2hhbm5lbElkAQAYCAAu" +
           "AEQYCAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAARAAAAQ2xpZW50Q2VydGlmaWNhdGUBABkIAC4A" +
           "RBkIAAAAD/////8BAf////8AAAAAFWCJCgIAAAAAABsAAABDbGllbnRDZXJ0aWZpY2F0ZVRodW1icHJp" +
           "bnQBALsKAC4ARLsKAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABUAAABSZXZpc2VkU2Vzc2lvblRp" +
           "bWVvdXQBABoIAC4ARBoIAAABACIB/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> SecureChannelId
        {
            get
            {
                return m_secureChannelId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_secureChannelId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_secureChannelId = value;
            }
        }

        /// <remarks />
        public PropertyState<byte[]> ClientCertificate
        {
            get
            {
                return m_clientCertificate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientCertificate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientCertificate = value;
            }
        }

        /// <remarks />
        public PropertyState<string> ClientCertificateThumbprint
        {
            get
            {
                return m_clientCertificateThumbprint;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientCertificateThumbprint, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientCertificateThumbprint = value;
            }
        }

        /// <remarks />
        public PropertyState<double> RevisedSessionTimeout
        {
            get
            {
                return m_revisedSessionTimeout;
            }

            set
            {
                if (!Object.ReferenceEquals(m_revisedSessionTimeout, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_revisedSessionTimeout = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_secureChannelId != null)
            {
                children.Add(m_secureChannelId);
            }

            if (m_clientCertificate != null)
            {
                children.Add(m_clientCertificate);
            }

            if (m_clientCertificateThumbprint != null)
            {
                children.Add(m_clientCertificateThumbprint);
            }

            if (m_revisedSessionTimeout != null)
            {
                children.Add(m_revisedSessionTimeout);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SecureChannelId:
                {
                    if (createOrReplace)
                    {
                        if (SecureChannelId == null)
                        {
                            if (replacement == null)
                            {
                                SecureChannelId = new PropertyState<string>(this);
                            }
                            else
                            {
                                SecureChannelId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = SecureChannelId;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientCertificate:
                {
                    if (createOrReplace)
                    {
                        if (ClientCertificate == null)
                        {
                            if (replacement == null)
                            {
                                ClientCertificate = new PropertyState<byte[]>(this);
                            }
                            else
                            {
                                ClientCertificate = (PropertyState<byte[]>)replacement;
                            }
                        }
                    }

                    instance = ClientCertificate;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientCertificateThumbprint:
                {
                    if (createOrReplace)
                    {
                        if (ClientCertificateThumbprint == null)
                        {
                            if (replacement == null)
                            {
                                ClientCertificateThumbprint = new PropertyState<string>(this);
                            }
                            else
                            {
                                ClientCertificateThumbprint = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ClientCertificateThumbprint;
                    break;
                }

                case Opc.Ua.BrowseNames.RevisedSessionTimeout:
                {
                    if (createOrReplace)
                    {
                        if (RevisedSessionTimeout == null)
                        {
                            if (replacement == null)
                            {
                                RevisedSessionTimeout = new PropertyState<double>(this);
                            }
                            else
                            {
                                RevisedSessionTimeout = (PropertyState<double>)replacement;
                            }
                        }
                    }

                    instance = RevisedSessionTimeout;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_secureChannelId;
        private PropertyState<byte[]> m_clientCertificate;
        private PropertyState<string> m_clientCertificateThumbprint;
        private PropertyState<double> m_revisedSessionTimeout;
        #endregion
    }
    #endif
    #endregion

    #region AuditUrlMismatchEventState Class
    #if (!OPCUA_EXCLUDE_AuditUrlMismatchEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditUrlMismatchEventState : AuditCreateSessionEventState
    {
        #region Constructors
        /// <remarks />
        public AuditUrlMismatchEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditUrlMismatchEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIQAAAEF1ZGl0VXJsTWlzbWF0Y2hFdmVudFR5cGVJbnN0YW5jZQEAvAoB" +
           "ALwKvAoAAP////8TAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQDRDAAuAETRDAAAAA//////AQH/////" +
           "AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQDSDAAuAETSDAAAABH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAKAAAAU291cmNlTm9kZQEA0wwALgBE0wwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAA" +
           "AFNvdXJjZU5hbWUBANQMAC4ARNQMAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQDV" +
           "DAAuAETVDAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEA1gwALgBE" +
           "1gwAAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEA2AwALgBE2AwAAAAV////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQDZDAAuAETZDAAAAAX/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQDaDAAuAETaDAAAAQAmAf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAYAAABTdGF0dXMBANsMAC4ARNsMAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgA" +
           "AABTZXJ2ZXJJZAEA3AwALgBE3AwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1" +
           "ZGl0RW50cnlJZAEA3QwALgBE3QwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVz" +
           "ZXJJZAEA3gwALgBE3gwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACQAAAFNlc3Npb25JZAEATjgA" +
           "LgBETjgAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAADwAAAFNlY3VyZUNoYW5uZWxJZAEA4AwALgBE" +
           "4AwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAENsaWVudENlcnRpZmljYXRlAQDhDAAuAETh" +
           "DAAAAA//////AQH/////AAAAABVgiQoCAAAAAAAbAAAAQ2xpZW50Q2VydGlmaWNhdGVUaHVtYnByaW50" +
           "AQDiDAAuAETiDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAVAAAAUmV2aXNlZFNlc3Npb25UaW1l" +
           "b3V0AQDjDAAuAETjDAAAAQAiAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABFbmRwb2ludFVybAEA" +
           "vQoALgBEvQoAAAAM/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> EndpointUrl
        {
            get
            {
                return m_endpointUrl;
            }

            set
            {
                if (!Object.ReferenceEquals(m_endpointUrl, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_endpointUrl = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_endpointUrl != null)
            {
                children.Add(m_endpointUrl);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.EndpointUrl:
                {
                    if (createOrReplace)
                    {
                        if (EndpointUrl == null)
                        {
                            if (replacement == null)
                            {
                                EndpointUrl = new PropertyState<string>(this);
                            }
                            else
                            {
                                EndpointUrl = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = EndpointUrl;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_endpointUrl;
        #endregion
    }
    #endif
    #endregion

    #region AuditActivateSessionEventState Class
    #if (!OPCUA_EXCLUDE_AuditActivateSessionEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditActivateSessionEventState : AuditSessionEventState
    {
        #region Constructors
        /// <remarks />
        public AuditActivateSessionEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditActivateSessionEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJQAAAEF1ZGl0QWN0aXZhdGVTZXNzaW9uRXZlbnRUeXBlSW5zdGFuY2UB" +
           "ABsIAQAbCBsIAAD/////EQAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEA5AwALgBE5AwAAAAP/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEA5QwALgBE5QwAAAAR/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACgAAAFNvdXJjZU5vZGUBAOYMAC4AROYMAAAAEf////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAoAAABTb3VyY2VOYW1lAQDnDAAuAETnDAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGlt" +
           "ZQEA6AwALgBE6AwAAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBAOkM" +
           "AC4AROkMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAOsMAC4AROsMAAAA" +
           "Ff////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEA7AwALgBE7AwAAAAF/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEA7QwALgBE7QwAAAEAJgH/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAGAAAAU3RhdHVzAQDuDAAuAETuDAAAAAH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAIAAAAU2VydmVySWQBAO8MAC4ARO8MAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGll" +
           "bnRBdWRpdEVudHJ5SWQBAPAMAC4ARPAMAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGll" +
           "bnRVc2VySWQBAPEMAC4ARPEMAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9uSWQB" +
           "APIMAC4ARPIMAAAAEf////8BAf////8AAAAAF2CJCgIAAAAAABoAAABDbGllbnRTb2Z0d2FyZUNlcnRp" +
           "ZmljYXRlcwEAHAgALgBEHAgAAAEAWAEBAAAAAQAAAAAAAAABAf////8AAAAAFWCJCgIAAAAAABEAAABV" +
           "c2VySWRlbnRpdHlUb2tlbgEAHQgALgBEHQgAAAEAPAH/////AQH/////AAAAABVgiQoCAAAAAAAPAAAA" +
           "U2VjdXJlQ2hhbm5lbElkAQDdLAAuAETdLAAAAAz/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<SignedSoftwareCertificate[]> ClientSoftwareCertificates
        {
            get
            {
                return m_clientSoftwareCertificates;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientSoftwareCertificates, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientSoftwareCertificates = value;
            }
        }

        /// <remarks />
        public PropertyState<UserIdentityToken> UserIdentityToken
        {
            get
            {
                return m_userIdentityToken;
            }

            set
            {
                if (!Object.ReferenceEquals(m_userIdentityToken, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_userIdentityToken = value;
            }
        }

        /// <remarks />
        public PropertyState<string> SecureChannelId
        {
            get
            {
                return m_secureChannelId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_secureChannelId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_secureChannelId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_clientSoftwareCertificates != null)
            {
                children.Add(m_clientSoftwareCertificates);
            }

            if (m_userIdentityToken != null)
            {
                children.Add(m_userIdentityToken);
            }

            if (m_secureChannelId != null)
            {
                children.Add(m_secureChannelId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ClientSoftwareCertificates:
                {
                    if (createOrReplace)
                    {
                        if (ClientSoftwareCertificates == null)
                        {
                            if (replacement == null)
                            {
                                ClientSoftwareCertificates = new PropertyState<SignedSoftwareCertificate[]>(this);
                            }
                            else
                            {
                                ClientSoftwareCertificates = (PropertyState<SignedSoftwareCertificate[]>)replacement;
                            }
                        }
                    }

                    instance = ClientSoftwareCertificates;
                    break;
                }

                case Opc.Ua.BrowseNames.UserIdentityToken:
                {
                    if (createOrReplace)
                    {
                        if (UserIdentityToken == null)
                        {
                            if (replacement == null)
                            {
                                UserIdentityToken = new PropertyState<UserIdentityToken>(this);
                            }
                            else
                            {
                                UserIdentityToken = (PropertyState<UserIdentityToken>)replacement;
                            }
                        }
                    }

                    instance = UserIdentityToken;
                    break;
                }

                case Opc.Ua.BrowseNames.SecureChannelId:
                {
                    if (createOrReplace)
                    {
                        if (SecureChannelId == null)
                        {
                            if (replacement == null)
                            {
                                SecureChannelId = new PropertyState<string>(this);
                            }
                            else
                            {
                                SecureChannelId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = SecureChannelId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<SignedSoftwareCertificate[]> m_clientSoftwareCertificates;
        private PropertyState<UserIdentityToken> m_userIdentityToken;
        private PropertyState<string> m_secureChannelId;
        #endregion
    }
    #endif
    #endregion

    #region AuditCancelEventState Class
    #if (!OPCUA_EXCLUDE_AuditCancelEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCancelEventState : AuditSessionEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCancelEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCancelEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAEF1ZGl0Q2FuY2VsRXZlbnRUeXBlSW5zdGFuY2UBAB4IAQAeCB4I" +
           "AAD/////DwAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEA8wwALgBE8wwAAAAP/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACQAAAEV2ZW50VHlwZQEA9AwALgBE9AwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CgAAAFNvdXJjZU5vZGUBAPUMAC4ARPUMAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABTb3Vy" +
           "Y2VOYW1lAQD2DAAuAET2DAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEA9wwALgBE" +
           "9wwAAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBAPgMAC4ARPgMAAAB" +
           "ACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAPoMAC4ARPoMAAAAFf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEA+wwALgBE+wwAAAAF/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEA/AwALgBE/AwAAAEAJgH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAGAAAAU3RhdHVzAQD9DAAuAET9DAAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2Vy" +
           "dmVySWQBAP4MAC4ARP4MAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGllbnRBdWRpdEVu" +
           "dHJ5SWQBAP8MAC4ARP8MAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGllbnRVc2VySWQB" +
           "AAANAC4ARAANAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9uSWQBAAENAC4ARAEN" +
           "AAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAA0AAABSZXF1ZXN0SGFuZGxlAQAfCAAuAEQfCAAAAAf/" +
           "////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<uint> RequestHandle
        {
            get
            {
                return m_requestHandle;
            }

            set
            {
                if (!Object.ReferenceEquals(m_requestHandle, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_requestHandle = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_requestHandle != null)
            {
                children.Add(m_requestHandle);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.RequestHandle:
                {
                    if (createOrReplace)
                    {
                        if (RequestHandle == null)
                        {
                            if (replacement == null)
                            {
                                RequestHandle = new PropertyState<uint>(this);
                            }
                            else
                            {
                                RequestHandle = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = RequestHandle;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<uint> m_requestHandle;
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateEventState : AuditSecurityEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIQAAAEF1ZGl0Q2VydGlmaWNhdGVFdmVudFR5cGVJbnN0YW5jZQEAIAgB" +
           "ACAIIAgAAP////8OAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQACDQAuAEQCDQAAAA//////AQH/////" +
           "AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQADDQAuAEQDDQAAABH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAKAAAAU291cmNlTm9kZQEABA0ALgBEBA0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAA" +
           "AFNvdXJjZU5hbWUBAAUNAC4ARAUNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQAG" +
           "DQAuAEQGDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEABw0ALgBE" +
           "Bw0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEACQ0ALgBECQ0AAAAV////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQAKDQAuAEQKDQAAAAX/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQALDQAuAEQLDQAAAQAmAf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAYAAABTdGF0dXMBAAwNAC4ARAwNAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgA" +
           "AABTZXJ2ZXJJZAEADQ0ALgBEDQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1" +
           "ZGl0RW50cnlJZAEADg0ALgBEDg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVz" +
           "ZXJJZAEADw0ALgBEDw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAENlcnRpZmljYXRlAQAh" +
           "CAAuAEQhCAAAAA//////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<byte[]> Certificate
        {
            get
            {
                return m_certificate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_certificate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_certificate = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_certificate != null)
            {
                children.Add(m_certificate);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Certificate:
                {
                    if (createOrReplace)
                    {
                        if (Certificate == null)
                        {
                            if (replacement == null)
                            {
                                Certificate = new PropertyState<byte[]>(this);
                            }
                            else
                            {
                                Certificate = (PropertyState<byte[]>)replacement;
                            }
                        }
                    }

                    instance = Certificate;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<byte[]> m_certificate;
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateDataMismatchEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateDataMismatchEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateDataMismatchEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateDataMismatchEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateDataMismatchEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAALQAAAEF1ZGl0Q2VydGlmaWNhdGVEYXRhTWlzbWF0Y2hFdmVudFR5cGVJ" +
           "bnN0YW5jZQEAIggBACIIIggAAP////8QAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQAQDQAuAEQQDQAA" +
           "AA//////AQH/////AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQARDQAuAEQRDQAAABH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAKAAAAU291cmNlTm9kZQEAEg0ALgBEEg0AAAAR/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACgAAAFNvdXJjZU5hbWUBABMNAC4ARBMNAAAADP////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAQAAABUaW1lAQAUDQAuAEQUDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZl" +
           "VGltZQEAFQ0ALgBEFQ0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAFw0A" +
           "LgBEFw0AAAAV/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQAYDQAuAEQYDQAAAAX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQAZDQAuAEQZDQAAAQAmAf//" +
           "//8BAf////8AAAAAFWCJCgIAAAAAAAYAAABTdGF0dXMBABoNAC4ARBoNAAAAAf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAgAAABTZXJ2ZXJJZAEAGw0ALgBEGw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "EgAAAENsaWVudEF1ZGl0RW50cnlJZAEAHA0ALgBEHA0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "DAAAAENsaWVudFVzZXJJZAEAHQ0ALgBEHQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAENl" +
           "cnRpZmljYXRlAQAeDQAuAEQeDQAAAA//////AQH/////AAAAABVgiQoCAAAAAAAPAAAASW52YWxpZEhv" +
           "c3RuYW1lAQAjCAAuAEQjCAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAKAAAASW52YWxpZFVyaQEA" +
           "JAgALgBEJAgAAAAM/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> InvalidHostname
        {
            get
            {
                return m_invalidHostname;
            }

            set
            {
                if (!Object.ReferenceEquals(m_invalidHostname, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_invalidHostname = value;
            }
        }

        /// <remarks />
        public PropertyState<string> InvalidUri
        {
            get
            {
                return m_invalidUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_invalidUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_invalidUri = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_invalidHostname != null)
            {
                children.Add(m_invalidHostname);
            }

            if (m_invalidUri != null)
            {
                children.Add(m_invalidUri);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.InvalidHostname:
                {
                    if (createOrReplace)
                    {
                        if (InvalidHostname == null)
                        {
                            if (replacement == null)
                            {
                                InvalidHostname = new PropertyState<string>(this);
                            }
                            else
                            {
                                InvalidHostname = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = InvalidHostname;
                    break;
                }

                case Opc.Ua.BrowseNames.InvalidUri:
                {
                    if (createOrReplace)
                    {
                        if (InvalidUri == null)
                        {
                            if (replacement == null)
                            {
                                InvalidUri = new PropertyState<string>(this);
                            }
                            else
                            {
                                InvalidUri = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = InvalidUri;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_invalidHostname;
        private PropertyState<string> m_invalidUri;
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateExpiredEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateExpiredEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateExpiredEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateExpiredEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateExpiredEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKAAAAEF1ZGl0Q2VydGlmaWNhdGVFeHBpcmVkRXZlbnRUeXBlSW5zdGFu" +
           "Y2UBACUIAQAlCCUIAAD/////DgAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAHw0ALgBEHw0AAAAP////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAIA0ALgBEIA0AAAAR/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAACgAAAFNvdXJjZU5vZGUBACENAC4ARCENAAAAEf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAoAAABTb3VyY2VOYW1lAQAiDQAuAEQiDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAA" +
           "VGltZQEAIw0ALgBEIw0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUB" +
           "ACQNAC4ARCQNAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBACYNAC4ARCYN" +
           "AAAAFf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAJw0ALgBEJw0AAAAF/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAKA0ALgBEKA0AAAEAJgH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAGAAAAU3RhdHVzAQApDQAuAEQpDQAAAAH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAIAAAAU2VydmVySWQBACoNAC4ARCoNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABD" +
           "bGllbnRBdWRpdEVudHJ5SWQBACsNAC4ARCsNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABD" +
           "bGllbnRVc2VySWQBACwNAC4ARCwNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABDZXJ0aWZp" +
           "Y2F0ZQEALQ0ALgBELQ0AAAAP/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateInvalidEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateInvalidEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateInvalidEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateInvalidEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateInvalidEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKAAAAEF1ZGl0Q2VydGlmaWNhdGVJbnZhbGlkRXZlbnRUeXBlSW5zdGFu" +
           "Y2UBACYIAQAmCCYIAAD/////DgAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEALg0ALgBELg0AAAAP////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEALw0ALgBELw0AAAAR/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAACgAAAFNvdXJjZU5vZGUBADANAC4ARDANAAAAEf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAoAAABTb3VyY2VOYW1lAQAxDQAuAEQxDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAA" +
           "VGltZQEAMg0ALgBEMg0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUB" +
           "ADMNAC4ARDMNAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBADUNAC4ARDUN" +
           "AAAAFf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEANg0ALgBENg0AAAAF/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEANw0ALgBENw0AAAEAJgH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAGAAAAU3RhdHVzAQA4DQAuAEQ4DQAAAAH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAIAAAAU2VydmVySWQBADkNAC4ARDkNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABD" +
           "bGllbnRBdWRpdEVudHJ5SWQBADoNAC4ARDoNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABD" +
           "bGllbnRVc2VySWQBADsNAC4ARDsNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABDZXJ0aWZp" +
           "Y2F0ZQEAPA0ALgBEPA0AAAAP/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateUntrustedEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateUntrustedEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateUntrustedEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateUntrustedEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateUntrustedEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKgAAAEF1ZGl0Q2VydGlmaWNhdGVVbnRydXN0ZWRFdmVudFR5cGVJbnN0" +
           "YW5jZQEAJwgBACcIJwgAAP////8OAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQA9DQAuAEQ9DQAAAA//" +
           "////AQH/////AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQA+DQAuAEQ+DQAAABH/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAKAAAAU291cmNlTm9kZQEAPw0ALgBEPw0AAAAR/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACgAAAFNvdXJjZU5hbWUBAEANAC4AREANAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQA" +
           "AABUaW1lAQBBDQAuAERBDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGlt" +
           "ZQEAQg0ALgBEQg0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEARA0ALgBE" +
           "RA0AAAAV/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQBFDQAuAERFDQAAAAX/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQBGDQAuAERGDQAAAQAmAf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAYAAABTdGF0dXMBAEcNAC4AREcNAAAAAf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAgAAABTZXJ2ZXJJZAEASA0ALgBESA0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAA" +
           "AENsaWVudEF1ZGl0RW50cnlJZAEASQ0ALgBESQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAA" +
           "AENsaWVudFVzZXJJZAEASg0ALgBESg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAENlcnRp" +
           "ZmljYXRlAQBLDQAuAERLDQAAAA//////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateRevokedEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateRevokedEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateRevokedEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateRevokedEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateRevokedEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKAAAAEF1ZGl0Q2VydGlmaWNhdGVSZXZva2VkRXZlbnRUeXBlSW5zdGFu" +
           "Y2UBACgIAQAoCCgIAAD/////DgAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEATA0ALgBETA0AAAAP////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEATQ0ALgBETQ0AAAAR/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAACgAAAFNvdXJjZU5vZGUBAE4NAC4ARE4NAAAAEf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAoAAABTb3VyY2VOYW1lAQBPDQAuAERPDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAA" +
           "VGltZQEAUA0ALgBEUA0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUB" +
           "AFENAC4ARFENAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAFMNAC4ARFMN" +
           "AAAAFf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAVA0ALgBEVA0AAAAF/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAVQ0ALgBEVQ0AAAEAJgH/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAGAAAAU3RhdHVzAQBWDQAuAERWDQAAAAH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAIAAAAU2VydmVySWQBAFcNAC4ARFcNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABD" +
           "bGllbnRBdWRpdEVudHJ5SWQBAFgNAC4ARFgNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABD" +
           "bGllbnRVc2VySWQBAFkNAC4ARFkNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABDZXJ0aWZp" +
           "Y2F0ZQEAWg0ALgBEWg0AAAAP/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditCertificateMismatchEventState Class
    #if (!OPCUA_EXCLUDE_AuditCertificateMismatchEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditCertificateMismatchEventState : AuditCertificateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditCertificateMismatchEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditCertificateMismatchEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAKQAAAEF1ZGl0Q2VydGlmaWNhdGVNaXNtYXRjaEV2ZW50VHlwZUluc3Rh" +
           "bmNlAQApCAEAKQgpCAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAFsNAC4ARFsNAAAAD///" +
           "//8BAf////8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAFwNAC4ARFwNAAAAEf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAoAAABTb3VyY2VOb2RlAQBdDQAuAERdDQAAABH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAKAAAAU291cmNlTmFtZQEAXg0ALgBEXg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAA" +
           "AFRpbWUBAF8NAC4ARF8NAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1l" +
           "AQBgDQAuAERgDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQBiDQAuAERi" +
           "DQAAABX/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAGMNAC4ARGMNAAAABf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAGQNAC4ARGQNAAABACYB/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAABgAAAFN0YXR1cwEAZQ0ALgBEZQ0AAAAB/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACAAAAFNlcnZlcklkAQBmDQAuAERmDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAA" +
           "Q2xpZW50QXVkaXRFbnRyeUlkAQBnDQAuAERnDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAA" +
           "Q2xpZW50VXNlcklkAQBoDQAuAERoDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAALAAAAQ2VydGlm" +
           "aWNhdGUBAGkNAC4ARGkNAAAAD/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditNodeManagementEventState Class
    #if (!OPCUA_EXCLUDE_AuditNodeManagementEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditNodeManagementEventState : AuditEventState
    {
        #region Constructors
        /// <remarks />
        public AuditNodeManagementEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditNodeManagementEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJAAAAEF1ZGl0Tm9kZU1hbmFnZW1lbnRFdmVudFR5cGVJbnN0YW5jZQEA" +
           "KggBACoIKggAAP////8NAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQBqDQAuAERqDQAAAA//////AQH/" +
           "////AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQBrDQAuAERrDQAAABH/////AQH/////AAAAABVg" +
           "iQoCAAAAAAAKAAAAU291cmNlTm9kZQEAbA0ALgBEbA0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CgAAAFNvdXJjZU5hbWUBAG0NAC4ARG0NAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1l" +
           "AQBuDQAuAERuDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAbw0A" +
           "LgBEbw0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAcQ0ALgBEcQ0AAAAV" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQByDQAuAERyDQAAAAX/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQBzDQAuAERzDQAAAQAmAf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAYAAABTdGF0dXMBAHQNAC4ARHQNAAAAAf////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAgAAABTZXJ2ZXJJZAEAdQ0ALgBEdQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVu" +
           "dEF1ZGl0RW50cnlJZAEAdg0ALgBEdg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVu" +
           "dFVzZXJJZAEAdw0ALgBEdw0AAAAM/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditAddNodesEventState Class
    #if (!OPCUA_EXCLUDE_AuditAddNodesEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditAddNodesEventState : AuditNodeManagementEventState
    {
        #region Constructors
        /// <remarks />
        public AuditAddNodesEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditAddNodesEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHgAAAEF1ZGl0QWRkTm9kZXNFdmVudFR5cGVJbnN0YW5jZQEAKwgBACsI" +
           "KwgAAP////8OAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQB4DQAuAER4DQAAAA//////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQB5DQAuAER5DQAAABH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAKAAAAU291cmNlTm9kZQEAeg0ALgBEeg0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFNv" +
           "dXJjZU5hbWUBAHsNAC4ARHsNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQB8DQAu" +
           "AER8DQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAfQ0ALgBEfQ0A" +
           "AAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAfw0ALgBEfw0AAAAV/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQCADQAuAESADQAAAAX/////AQH/////AAAAABVg" +
           "iQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQCBDQAuAESBDQAAAQAmAf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAYAAABTdGF0dXMBAIINAC4ARIINAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABT" +
           "ZXJ2ZXJJZAEAgw0ALgBEgw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1ZGl0" +
           "RW50cnlJZAEAhA0ALgBEhA0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVzZXJJ" +
           "ZAEAhQ0ALgBEhQ0AAAAM/////wEB/////wAAAAAXYIkKAgAAAAAACgAAAE5vZGVzVG9BZGQBACwIAC4A" +
           "RCwIAAABAHgBAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<AddNodesItem[]> NodesToAdd
        {
            get
            {
                return m_nodesToAdd;
            }

            set
            {
                if (!Object.ReferenceEquals(m_nodesToAdd, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_nodesToAdd = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_nodesToAdd != null)
            {
                children.Add(m_nodesToAdd);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.NodesToAdd:
                {
                    if (createOrReplace)
                    {
                        if (NodesToAdd == null)
                        {
                            if (replacement == null)
                            {
                                NodesToAdd = new PropertyState<AddNodesItem[]>(this);
                            }
                            else
                            {
                                NodesToAdd = (PropertyState<AddNodesItem[]>)replacement;
                            }
                        }
                    }

                    instance = NodesToAdd;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<AddNodesItem[]> m_nodesToAdd;
        #endregion
    }
    #endif
    #endregion

    #region AuditDeleteNodesEventState Class
    #if (!OPCUA_EXCLUDE_AuditDeleteNodesEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditDeleteNodesEventState : AuditNodeManagementEventState
    {
        #region Constructors
        /// <remarks />
        public AuditDeleteNodesEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditDeleteNodesEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIQAAAEF1ZGl0RGVsZXRlTm9kZXNFdmVudFR5cGVJbnN0YW5jZQEALQgB" +
           "AC0ILQgAAP////8OAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQCGDQAuAESGDQAAAA//////AQH/////" +
           "AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQCHDQAuAESHDQAAABH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAKAAAAU291cmNlTm9kZQEAiA0ALgBEiA0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAA" +
           "AFNvdXJjZU5hbWUBAIkNAC4ARIkNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQCK" +
           "DQAuAESKDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAiw0ALgBE" +
           "iw0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAjQ0ALgBEjQ0AAAAV////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQCODQAuAESODQAAAAX/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQCPDQAuAESPDQAAAQAmAf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAYAAABTdGF0dXMBAJANAC4ARJANAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgA" +
           "AABTZXJ2ZXJJZAEAkQ0ALgBEkQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1" +
           "ZGl0RW50cnlJZAEAkg0ALgBEkg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVz" +
           "ZXJJZAEAkw0ALgBEkw0AAAAM/////wEB/////wAAAAAXYIkKAgAAAAAADQAAAE5vZGVzVG9EZWxldGUB" +
           "AC4IAC4ARC4IAAABAH4BAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<DeleteNodesItem[]> NodesToDelete
        {
            get
            {
                return m_nodesToDelete;
            }

            set
            {
                if (!Object.ReferenceEquals(m_nodesToDelete, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_nodesToDelete = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_nodesToDelete != null)
            {
                children.Add(m_nodesToDelete);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.NodesToDelete:
                {
                    if (createOrReplace)
                    {
                        if (NodesToDelete == null)
                        {
                            if (replacement == null)
                            {
                                NodesToDelete = new PropertyState<DeleteNodesItem[]>(this);
                            }
                            else
                            {
                                NodesToDelete = (PropertyState<DeleteNodesItem[]>)replacement;
                            }
                        }
                    }

                    instance = NodesToDelete;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<DeleteNodesItem[]> m_nodesToDelete;
        #endregion
    }
    #endif
    #endregion

    #region AuditAddReferencesEventState Class
    #if (!OPCUA_EXCLUDE_AuditAddReferencesEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditAddReferencesEventState : AuditNodeManagementEventState
    {
        #region Constructors
        /// <remarks />
        public AuditAddReferencesEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditAddReferencesEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAEF1ZGl0QWRkUmVmZXJlbmNlc0V2ZW50VHlwZUluc3RhbmNlAQAv" +
           "CAEALwgvCAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAJQNAC4ARJQNAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAJUNAC4ARJUNAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQCWDQAuAESWDQAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAlw0ALgBElw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "AJgNAC4ARJgNAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQCZDQAu" +
           "AESZDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQCbDQAuAESbDQAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAJwNAC4ARJwNAAAABf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAJ0NAC4ARJ0NAAABACYB/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAABgAAAFN0YXR1cwEAng0ALgBEng0AAAAB/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CAAAAFNlcnZlcklkAQCfDQAuAESfDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xpZW50" +
           "QXVkaXRFbnRyeUlkAQCgDQAuAESgDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xpZW50" +
           "VXNlcklkAQChDQAuAEShDQAAAAz/////AQH/////AAAAABdgiQoCAAAAAAAPAAAAUmVmZXJlbmNlc1Rv" +
           "QWRkAQAwCAAuAEQwCAAAAQB7AQEAAAABAAAAAAAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<AddReferencesItem[]> ReferencesToAdd
        {
            get
            {
                return m_referencesToAdd;
            }

            set
            {
                if (!Object.ReferenceEquals(m_referencesToAdd, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_referencesToAdd = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_referencesToAdd != null)
            {
                children.Add(m_referencesToAdd);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ReferencesToAdd:
                {
                    if (createOrReplace)
                    {
                        if (ReferencesToAdd == null)
                        {
                            if (replacement == null)
                            {
                                ReferencesToAdd = new PropertyState<AddReferencesItem[]>(this);
                            }
                            else
                            {
                                ReferencesToAdd = (PropertyState<AddReferencesItem[]>)replacement;
                            }
                        }
                    }

                    instance = ReferencesToAdd;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<AddReferencesItem[]> m_referencesToAdd;
        #endregion
    }
    #endif
    #endregion

    #region AuditDeleteReferencesEventState Class
    #if (!OPCUA_EXCLUDE_AuditDeleteReferencesEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditDeleteReferencesEventState : AuditNodeManagementEventState
    {
        #region Constructors
        /// <remarks />
        public AuditDeleteReferencesEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditDeleteReferencesEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAJgAAAEF1ZGl0RGVsZXRlUmVmZXJlbmNlc0V2ZW50VHlwZUluc3RhbmNl" +
           "AQAxCAEAMQgxCAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAKINAC4ARKINAAAAD/////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAKMNAC4ARKMNAAAAEf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAoAAABTb3VyY2VOb2RlAQCkDQAuAESkDQAAABH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAKAAAAU291cmNlTmFtZQEApQ0ALgBEpQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRp" +
           "bWUBAKYNAC4ARKYNAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQCn" +
           "DQAuAESnDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQCpDQAuAESpDQAA" +
           "ABX/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAKoNAC4ARKoNAAAABf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABAKsNAC4ARKsNAAABACYB/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAABgAAAFN0YXR1cwEArA0ALgBErA0AAAAB/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAACAAAAFNlcnZlcklkAQCtDQAuAEStDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xp" +
           "ZW50QXVkaXRFbnRyeUlkAQCuDQAuAESuDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xp" +
           "ZW50VXNlcklkAQCvDQAuAESvDQAAAAz/////AQH/////AAAAABdgiQoCAAAAAAASAAAAUmVmZXJlbmNl" +
           "c1RvRGVsZXRlAQAyCAAuAEQyCAAAAQCBAQEAAAABAAAAAAAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<DeleteReferencesItem[]> ReferencesToDelete
        {
            get
            {
                return m_referencesToDelete;
            }

            set
            {
                if (!Object.ReferenceEquals(m_referencesToDelete, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_referencesToDelete = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_referencesToDelete != null)
            {
                children.Add(m_referencesToDelete);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ReferencesToDelete:
                {
                    if (createOrReplace)
                    {
                        if (ReferencesToDelete == null)
                        {
                            if (replacement == null)
                            {
                                ReferencesToDelete = new PropertyState<DeleteReferencesItem[]>(this);
                            }
                            else
                            {
                                ReferencesToDelete = (PropertyState<DeleteReferencesItem[]>)replacement;
                            }
                        }
                    }

                    instance = ReferencesToDelete;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<DeleteReferencesItem[]> m_referencesToDelete;
        #endregion
    }
    #endif
    #endregion

    #region AuditUpdateEventState Class
    #if (!OPCUA_EXCLUDE_AuditUpdateEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditUpdateEventState : AuditEventState
    {
        #region Constructors
        /// <remarks />
        public AuditUpdateEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditUpdateEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAEF1ZGl0VXBkYXRlRXZlbnRUeXBlSW5zdGFuY2UBADMIAQAzCDMI" +
           "AAD/////DQAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAsA0ALgBEsA0AAAAP/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAsQ0ALgBEsQ0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CgAAAFNvdXJjZU5vZGUBALINAC4ARLINAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABTb3Vy" +
           "Y2VOYW1lAQCzDQAuAESzDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEAtA0ALgBE" +
           "tA0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBALUNAC4ARLUNAAAB" +
           "ACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBALcNAC4ARLcNAAAAFf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAuA0ALgBEuA0AAAAF/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAuQ0ALgBEuQ0AAAEAJgH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAGAAAAU3RhdHVzAQC6DQAuAES6DQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2Vy" +
           "dmVySWQBALsNAC4ARLsNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGllbnRBdWRpdEVu" +
           "dHJ5SWQBALwNAC4ARLwNAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGllbnRVc2VySWQB" +
           "AL0NAC4ARL0NAAAADP////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region AuditWriteUpdateEventState Class
    #if (!OPCUA_EXCLUDE_AuditWriteUpdateEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditWriteUpdateEventState : AuditUpdateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditWriteUpdateEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditWriteUpdateEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIQAAAEF1ZGl0V3JpdGVVcGRhdGVFdmVudFR5cGVJbnN0YW5jZQEANAgB" +
           "ADQINAgAAP////8RAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQC+DQAuAES+DQAAAA//////AQH/////" +
           "AAAAABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQC/DQAuAES/DQAAABH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAKAAAAU291cmNlTm9kZQEAwA0ALgBEwA0AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAA" +
           "AFNvdXJjZU5hbWUBAMENAC4ARMENAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQDC" +
           "DQAuAETCDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAww0ALgBE" +
           "ww0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAxQ0ALgBExQ0AAAAV////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQDGDQAuAETGDQAAAAX/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAPAAAAQWN0aW9uVGltZVN0YW1wAQDHDQAuAETHDQAAAQAmAf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAYAAABTdGF0dXMBAMgNAC4ARMgNAAAAAf////8BAf////8AAAAAFWCJCgIAAAAAAAgA" +
           "AABTZXJ2ZXJJZAEAyQ0ALgBEyQ0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAENsaWVudEF1" +
           "ZGl0RW50cnlJZAEAyg0ALgBEyg0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAADAAAAENsaWVudFVz" +
           "ZXJJZAEAyw0ALgBEyw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAEF0dHJpYnV0ZUlkAQC+" +
           "CgAuAES+CgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAKAAAASW5kZXhSYW5nZQEANQgALgBENQgA" +
           "AAEAIwH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAT2xkVmFsdWUBADYIAC4ARDYIAAAAGP////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAgAAABOZXdWYWx1ZQEANwgALgBENwgAAAAY/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<uint> AttributeId
        {
            get
            {
                return m_attributeId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_attributeId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_attributeId = value;
            }
        }

        /// <remarks />
        public PropertyState<string> IndexRange
        {
            get
            {
                return m_indexRange;
            }

            set
            {
                if (!Object.ReferenceEquals(m_indexRange, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_indexRange = value;
            }
        }

        /// <remarks />
        public PropertyState OldValue
        {
            get
            {
                return m_oldValue;
            }

            set
            {
                if (!Object.ReferenceEquals(m_oldValue, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_oldValue = value;
            }
        }

        /// <remarks />
        public PropertyState NewValue
        {
            get
            {
                return m_newValue;
            }

            set
            {
                if (!Object.ReferenceEquals(m_newValue, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_newValue = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_attributeId != null)
            {
                children.Add(m_attributeId);
            }

            if (m_indexRange != null)
            {
                children.Add(m_indexRange);
            }

            if (m_oldValue != null)
            {
                children.Add(m_oldValue);
            }

            if (m_newValue != null)
            {
                children.Add(m_newValue);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.AttributeId:
                {
                    if (createOrReplace)
                    {
                        if (AttributeId == null)
                        {
                            if (replacement == null)
                            {
                                AttributeId = new PropertyState<uint>(this);
                            }
                            else
                            {
                                AttributeId = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = AttributeId;
                    break;
                }

                case Opc.Ua.BrowseNames.IndexRange:
                {
                    if (createOrReplace)
                    {
                        if (IndexRange == null)
                        {
                            if (replacement == null)
                            {
                                IndexRange = new PropertyState<string>(this);
                            }
                            else
                            {
                                IndexRange = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = IndexRange;
                    break;
                }

                case Opc.Ua.BrowseNames.OldValue:
                {
                    if (createOrReplace)
                    {
                        if (OldValue == null)
                        {
                            if (replacement == null)
                            {
                                OldValue = new PropertyState(this);
                            }
                            else
                            {
                                OldValue = (PropertyState)replacement;
                            }
                        }
                    }

                    instance = OldValue;
                    break;
                }

                case Opc.Ua.BrowseNames.NewValue:
                {
                    if (createOrReplace)
                    {
                        if (NewValue == null)
                        {
                            if (replacement == null)
                            {
                                NewValue = new PropertyState(this);
                            }
                            else
                            {
                                NewValue = (PropertyState)replacement;
                            }
                        }
                    }

                    instance = NewValue;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<uint> m_attributeId;
        private PropertyState<string> m_indexRange;
        private PropertyState m_oldValue;
        private PropertyState m_newValue;
        #endregion
    }
    #endif
    #endregion

    #region AuditHistoryUpdateEventState Class
    #if (!OPCUA_EXCLUDE_AuditHistoryUpdateEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditHistoryUpdateEventState : AuditUpdateEventState
    {
        #region Constructors
        /// <remarks />
        public AuditHistoryUpdateEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditHistoryUpdateEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAEF1ZGl0SGlzdG9yeVVwZGF0ZUV2ZW50VHlwZUluc3RhbmNlAQA4" +
           "CAEAOAg4CAAA/////w4AAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAMwNAC4ARMwNAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAM0NAC4ARM0NAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQDODQAuAETODQAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAzw0ALgBEzw0AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "ANANAC4ARNANAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQDRDQAu" +
           "AETRDQAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQDTDQAuAETTDQAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBANQNAC4ARNQNAAAABf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAA8AAABBY3Rpb25UaW1lU3RhbXABANUNAC4ARNUNAAABACYB/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAABgAAAFN0YXR1cwEA1g0ALgBE1g0AAAAB/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CAAAAFNlcnZlcklkAQDXDQAuAETXDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAASAAAAQ2xpZW50" +
           "QXVkaXRFbnRyeUlkAQDYDQAuAETYDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAMAAAAQ2xpZW50" +
           "VXNlcklkAQDZDQAuAETZDQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAATAAAAUGFyYW1ldGVyRGF0" +
           "YVR5cGVJZAEAvwoALgBEvwoAAAAR/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<NodeId> ParameterDataTypeId
        {
            get
            {
                return m_parameterDataTypeId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_parameterDataTypeId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_parameterDataTypeId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_parameterDataTypeId != null)
            {
                children.Add(m_parameterDataTypeId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ParameterDataTypeId:
                {
                    if (createOrReplace)
                    {
                        if (ParameterDataTypeId == null)
                        {
                            if (replacement == null)
                            {
                                ParameterDataTypeId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                ParameterDataTypeId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = ParameterDataTypeId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<NodeId> m_parameterDataTypeId;
        #endregion
    }
    #endif
    #endregion

    #region AuditUpdateMethodEventState Class
    #if (!OPCUA_EXCLUDE_AuditUpdateMethodEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditUpdateMethodEventState : AuditEventState
    {
        #region Constructors
        /// <remarks />
        public AuditUpdateMethodEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditUpdateMethodEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIgAAAEF1ZGl0VXBkYXRlTWV0aG9kRXZlbnRUeXBlSW5zdGFuY2UBAE8I" +
           "AQBPCE8IAAD/////DwAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEANw4ALgBENw4AAAAP/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAOA4ALgBEOA4AAAAR/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACgAAAFNvdXJjZU5vZGUBADkOAC4ARDkOAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoA" +
           "AABTb3VyY2VOYW1lAQA6DgAuAEQ6DgAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEA" +
           "Ow4ALgBEOw4AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBADwOAC4A" +
           "RDwOAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAD4OAC4ARD4OAAAAFf//" +
           "//8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAPw4ALgBEPw4AAAAF/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAQA4ALgBEQA4AAAEAJgH/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAGAAAAU3RhdHVzAQBBDgAuAERBDgAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAI" +
           "AAAAU2VydmVySWQBAEIOAC4AREIOAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGllbnRB" +
           "dWRpdEVudHJ5SWQBAEMOAC4AREMOAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGllbnRV" +
           "c2VySWQBAEQOAC4AREQOAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABNZXRob2RJZAEAUAgA" +
           "LgBEUAgAAAAR/////wEB/////wAAAAAXYIkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBRCAAuAERR" +
           "CAAAABgBAAAAAQAAAAAAAAABAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<NodeId> MethodId
        {
            get
            {
                return m_methodId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_methodId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_methodId = value;
            }
        }

        /// <remarks />
        public PropertyState<object[]> InputArguments
        {
            get
            {
                return m_inputArguments;
            }

            set
            {
                if (!Object.ReferenceEquals(m_inputArguments, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_inputArguments = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_methodId != null)
            {
                children.Add(m_methodId);
            }

            if (m_inputArguments != null)
            {
                children.Add(m_inputArguments);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.MethodId:
                {
                    if (createOrReplace)
                    {
                        if (MethodId == null)
                        {
                            if (replacement == null)
                            {
                                MethodId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                MethodId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = MethodId;
                    break;
                }

                case Opc.Ua.BrowseNames.InputArguments:
                {
                    if (createOrReplace)
                    {
                        if (InputArguments == null)
                        {
                            if (replacement == null)
                            {
                                InputArguments = new PropertyState<object[]>(this);
                            }
                            else
                            {
                                InputArguments = (PropertyState<object[]>)replacement;
                            }
                        }
                    }

                    instance = InputArguments;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<NodeId> m_methodId;
        private PropertyState<object[]> m_inputArguments;
        #endregion
    }
    #endif
    #endregion

    #region SystemEventState Class
    #if (!OPCUA_EXCLUDE_SystemEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SystemEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public SystemEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.SystemEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFwAAAFN5c3RlbUV2ZW50VHlwZUluc3RhbmNlAQBSCAEAUghSCAAA////" +
           "/wgAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAEUOAC4AREUOAAAAD/////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAkAAABFdmVudFR5cGUBAEYOAC4AREYOAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABT" +
           "b3VyY2VOb2RlAQBHDgAuAERHDgAAABH/////AQH/////AAAAABVgiQoCAAAAAAAKAAAAU291cmNlTmFt" +
           "ZQEASA4ALgBESA4AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUBAEkOAC4AREkOAAAB" +
           "ACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQBKDgAuAERKDgAAAQAmAf//" +
           "//8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQBMDgAuAERMDgAAABX/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAE0OAC4ARE0OAAAABf////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region DeviceFailureEventState Class
    #if (!OPCUA_EXCLUDE_DeviceFailureEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class DeviceFailureEventState : SystemEventState
    {
        #region Constructors
        /// <remarks />
        public DeviceFailureEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.DeviceFailureEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHgAAAERldmljZUZhaWx1cmVFdmVudFR5cGVJbnN0YW5jZQEAUwgBAFMI" +
           "UwgAAP////8IAAAAFWCJCgIAAAAAAAcAAABFdmVudElkAQBODgAuAERODgAAAA//////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAJAAAARXZlbnRUeXBlAQBPDgAuAERPDgAAABH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAKAAAAU291cmNlTm9kZQEAUA4ALgBEUA4AAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAAAFNv" +
           "dXJjZU5hbWUBAFEOAC4ARFEOAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAQAAABUaW1lAQBSDgAu" +
           "AERSDgAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABSZWNlaXZlVGltZQEAUw4ALgBEUw4A" +
           "AAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAAHAAAATWVzc2FnZQEAVQ4ALgBEVQ4AAAAV/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACAAAAFNldmVyaXR5AQBWDgAuAERWDgAAAAX/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region SystemStatusChangeEventState Class
    #if (!OPCUA_EXCLUDE_SystemStatusChangeEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SystemStatusChangeEventState : SystemEventState
    {
        #region Constructors
        /// <remarks />
        public SystemStatusChangeEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.SystemStatusChangeEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAFN5c3RlbVN0YXR1c0NoYW5nZUV2ZW50VHlwZUluc3RhbmNlAQC2" +
           "LAEAtiy2LAAA/////wkAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBALcsAC4ARLcsAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBALgsAC4ARLgsAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQC5LAAuAES5LAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAuiwALgBEuiwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "ALssAC4ARLssAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQC8LAAu" +
           "AES8LAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQC+LAAuAES+LAAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAL8sAC4ARL8sAAAABf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAsAAABTeXN0ZW1TdGF0ZQEAsC0ALgBEsC0AAAEAVAP/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<ServerState> SystemState
        {
            get
            {
                return m_systemState;
            }

            set
            {
                if (!Object.ReferenceEquals(m_systemState, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_systemState = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_systemState != null)
            {
                children.Add(m_systemState);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SystemState:
                {
                    if (createOrReplace)
                    {
                        if (SystemState == null)
                        {
                            if (replacement == null)
                            {
                                SystemState = new PropertyState<ServerState>(this);
                            }
                            else
                            {
                                SystemState = (PropertyState<ServerState>)replacement;
                            }
                        }
                    }

                    instance = SystemState;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<ServerState> m_systemState;
        #endregion
    }
    #endif
    #endregion

    #region BaseModelChangeEventState Class
    #if (!OPCUA_EXCLUDE_BaseModelChangeEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class BaseModelChangeEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public BaseModelChangeEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.BaseModelChangeEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIAAAAEJhc2VNb2RlbENoYW5nZUV2ZW50VHlwZUluc3RhbmNlAQBUCAEA" +
           "VAhUCAAA/////wgAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAFcOAC4ARFcOAAAAD/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAFgOAC4ARFgOAAAAEf////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAAoAAABTb3VyY2VOb2RlAQBZDgAuAERZDgAAABH/////AQH/////AAAAABVgiQoCAAAAAAAKAAAA" +
           "U291cmNlTmFtZQEAWg4ALgBEWg4AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUBAFsO" +
           "AC4ARFsOAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQBcDgAuAERc" +
           "DgAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQBeDgAuAEReDgAAABX/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAF8OAC4ARF8OAAAABf////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region GeneralModelChangeEventState Class
    #if (!OPCUA_EXCLUDE_GeneralModelChangeEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class GeneralModelChangeEventState : BaseModelChangeEventState
    {
        #region Constructors
        /// <remarks />
        public GeneralModelChangeEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.GeneralModelChangeEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAEdlbmVyYWxNb2RlbENoYW5nZUV2ZW50VHlwZUluc3RhbmNlAQBV" +
           "CAEAVQhVCAAA/////wkAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAGAOAC4ARGAOAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAGEOAC4ARGEOAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQBiDgAuAERiDgAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAYw4ALgBEYw4AAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "AGQOAC4ARGQOAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQBlDgAu" +
           "AERlDgAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQBnDgAuAERnDgAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAGgOAC4ARGgOAAAABf////8BAf////8A" +
           "AAAAF2CJCgIAAAAAAAcAAABDaGFuZ2VzAQBWCAAuAERWCAAAAQBtAwEAAAABAAAAAAAAAAEB/////wAA" +
           "AAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<ModelChangeStructureDataType[]> Changes
        {
            get
            {
                return m_changes;
            }

            set
            {
                if (!Object.ReferenceEquals(m_changes, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_changes = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_changes != null)
            {
                children.Add(m_changes);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Changes:
                {
                    if (createOrReplace)
                    {
                        if (Changes == null)
                        {
                            if (replacement == null)
                            {
                                Changes = new PropertyState<ModelChangeStructureDataType[]>(this);
                            }
                            else
                            {
                                Changes = (PropertyState<ModelChangeStructureDataType[]>)replacement;
                            }
                        }
                    }

                    instance = Changes;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<ModelChangeStructureDataType[]> m_changes;
        #endregion
    }
    #endif
    #endregion

    #region SemanticChangeEventState Class
    #if (!OPCUA_EXCLUDE_SemanticChangeEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SemanticChangeEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public SemanticChangeEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.SemanticChangeEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHwAAAFNlbWFudGljQ2hhbmdlRXZlbnRUeXBlSW5zdGFuY2UBALIKAQCy" +
           "CrIKAAD/////CQAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAaQ4ALgBEaQ4AAAAP/////wEB/////wAA" +
           "AAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAag4ALgBEag4AAAAR/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAACgAAAFNvdXJjZU5vZGUBAGsOAC4ARGsOAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABT" +
           "b3VyY2VOYW1lAQBsDgAuAERsDgAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEAbQ4A" +
           "LgBEbQ4AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBAG4OAC4ARG4O" +
           "AAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAHAOAC4ARHAOAAAAFf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAcQ4ALgBEcQ4AAAAF/////wEB/////wAAAAAX" +
           "YIkKAgAAAAAABwAAAENoYW5nZXMBALMKAC4ARLMKAAABAIEDAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<SemanticChangeStructureDataType[]> Changes
        {
            get
            {
                return m_changes;
            }

            set
            {
                if (!Object.ReferenceEquals(m_changes, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_changes = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_changes != null)
            {
                children.Add(m_changes);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Changes:
                {
                    if (createOrReplace)
                    {
                        if (Changes == null)
                        {
                            if (replacement == null)
                            {
                                Changes = new PropertyState<SemanticChangeStructureDataType[]>(this);
                            }
                            else
                            {
                                Changes = (PropertyState<SemanticChangeStructureDataType[]>)replacement;
                            }
                        }
                    }

                    instance = Changes;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<SemanticChangeStructureDataType[]> m_changes;
        #endregion
    }
    #endif
    #endregion

    #region EventQueueOverflowEventState Class
    #if (!OPCUA_EXCLUDE_EventQueueOverflowEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class EventQueueOverflowEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public EventQueueOverflowEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.EventQueueOverflowEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAIwAAAEV2ZW50UXVldWVPdmVyZmxvd0V2ZW50VHlwZUluc3RhbmNlAQDb" +
           "CwEA2wvbCwAA/////wgAAAAVYIkKAgAAAAAABwAAAEV2ZW50SWQBAHcMAC4ARHcMAAAAD/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAkAAABFdmVudFR5cGUBAHgMAC4ARHgMAAAAEf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAAAoAAABTb3VyY2VOb2RlAQB5DAAuAER5DAAAABH/////AQH/////AAAAABVgiQoCAAAAAAAK" +
           "AAAAU291cmNlTmFtZQEAegwALgBEegwAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAABAAAAFRpbWUB" +
           "AHsMAC4ARHsMAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAFJlY2VpdmVUaW1lAQB8DAAu" +
           "AER8DAAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAAAcAAABNZXNzYWdlAQB+DAAuAER+DAAAABX/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2V2ZXJpdHkBAH8MAC4ARH8MAAAABf////8BAf////8A" +
           "AAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ProgressEventState Class
    #if (!OPCUA_EXCLUDE_ProgressEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ProgressEventState : BaseEventState
    {
        #region Constructors
        /// <remarks />
        public ProgressEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ProgressEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAGQAAAFByb2dyZXNzRXZlbnRUeXBlSW5zdGFuY2UBAKwsAQCsLKwsAAD/" +
           "////CgAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEArSwALgBErSwAAAAP/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAACQAAAEV2ZW50VHlwZQEAriwALgBEriwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAACgAA" +
           "AFNvdXJjZU5vZGUBAK8sAC4ARK8sAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABTb3VyY2VO" +
           "YW1lAQCwLAAuAESwLAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEAsSwALgBEsSwA" +
           "AAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBALIsAC4ARLIsAAABACYB" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBALQsAC4ARLQsAAAAFf////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAtSwALgBEtSwAAAAF/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAABwAAAENvbnRleHQBANYwAC4ARNYwAAAAGP////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABQcm9n" +
           "cmVzcwEA1zAALgBE1zAAAAAF/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState Context
        {
            get
            {
                return m_context;
            }

            set
            {
                if (!Object.ReferenceEquals(m_context, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_context = value;
            }
        }

        /// <remarks />
        public PropertyState<ushort> Progress
        {
            get
            {
                return m_progress;
            }

            set
            {
                if (!Object.ReferenceEquals(m_progress, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_progress = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_context != null)
            {
                children.Add(m_context);
            }

            if (m_progress != null)
            {
                children.Add(m_progress);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Context:
                {
                    if (createOrReplace)
                    {
                        if (Context == null)
                        {
                            if (replacement == null)
                            {
                                Context = new PropertyState(this);
                            }
                            else
                            {
                                Context = (PropertyState)replacement;
                            }
                        }
                    }

                    instance = Context;
                    break;
                }

                case Opc.Ua.BrowseNames.Progress:
                {
                    if (createOrReplace)
                    {
                        if (Progress == null)
                        {
                            if (replacement == null)
                            {
                                Progress = new PropertyState<ushort>(this);
                            }
                            else
                            {
                                Progress = (PropertyState<ushort>)replacement;
                            }
                        }
                    }

                    instance = Progress;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState m_context;
        private PropertyState<ushort> m_progress;
        #endregion
    }
    #endif
    #endregion

    #region AuditClientEventState Class
    #if (!OPCUA_EXCLUDE_AuditClientEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditClientEventState : AuditEventState
    {
        #region Constructors
        /// <remarks />
        public AuditClientEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditClientEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHAAAAEF1ZGl0Q2xpZW50RXZlbnRUeXBlSW5zdGFuY2UBADZcAQA2XDZc" +
           "AAD/////DgAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAN1wALgBEN1wAAAAP/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAOlwALgBEOlwAAAAR/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CgAAAFNvdXJjZU5vZGUBADtcAC4ARDtcAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAoAAABTb3Vy" +
           "Y2VOYW1lAQBTXQAuAERTXQAAAAz/////AQH/////AAAAABVgiQoCAAAAAAAEAAAAVGltZQEAVF0ALgBE" +
           "VF0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2ZVRpbWUBAFVdAC4ARFVdAAAB" +
           "ACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAFddAC4ARFddAAAAFf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAWF0ALgBEWF0AAAAF/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAX10ALgBEX10AAAEAJgH/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAGAAAAU3RhdHVzAQBgXQAuAERgXQAAAAH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAAU2Vy" +
           "dmVySWQBAGFdAC4ARGFdAAAADP////8BAf////8AAAAAFWCJCgIAAAAAABIAAABDbGllbnRBdWRpdEVu" +
           "dHJ5SWQBAGJdAC4ARGJdAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABDbGllbnRVc2VySWQB" +
           "AGNdAC4ARGNdAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABTZXJ2ZXJVcmkBAGRdAC4ARGRd" +
           "AAABAMdc/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> ServerUri
        {
            get
            {
                return m_serverUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverUri = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverUri != null)
            {
                children.Add(m_serverUri);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerUri:
                {
                    if (createOrReplace)
                    {
                        if (ServerUri == null)
                        {
                            if (replacement == null)
                            {
                                ServerUri = new PropertyState<string>(this);
                            }
                            else
                            {
                                ServerUri = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ServerUri;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_serverUri;
        #endregion
    }
    #endif
    #endregion

    #region AuditClientUpdateMethodResultEventState Class
    #if (!OPCUA_EXCLUDE_AuditClientUpdateMethodResultEventState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AuditClientUpdateMethodResultEventState : AuditClientEventState
    {
        #region Constructors
        /// <remarks />
        public AuditClientUpdateMethodResultEventState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AuditClientUpdateMethodResultEventType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAALgAAAEF1ZGl0Q2xpZW50VXBkYXRlTWV0aG9kUmVzdWx0RXZlbnRUeXBl" +
           "SW5zdGFuY2UBAHZdAQB2XXZdAAD/////EwAAABVgiQoCAAAAAAAHAAAARXZlbnRJZAEAd10ALgBEd10A" +
           "AAAP/////wEB/////wAAAAAVYIkKAgAAAAAACQAAAEV2ZW50VHlwZQEAel0ALgBEel0AAAAR/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAACgAAAFNvdXJjZU5vZGUBAHtdAC4ARHtdAAAAEf////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAAoAAABTb3VyY2VOYW1lAQCXXQAuAESXXQAAAAz/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAEAAAAVGltZQEAmF0ALgBEmF0AAAEAJgH/////AQH/////AAAAABVgiQoCAAAAAAALAAAAUmVjZWl2" +
           "ZVRpbWUBAJldAC4ARJldAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAABwAAAE1lc3NhZ2UBAJtd" +
           "AC4ARJtdAAAAFf////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABTZXZlcml0eQEAnF0ALgBEnF0AAAAF" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAADwAAAEFjdGlvblRpbWVTdGFtcAEAo10ALgBEo10AAAEAJgH/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAGAAAAU3RhdHVzAQCkXQAuAESkXQAAAAH/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAIAAAAU2VydmVySWQBAKVdAC4ARKVdAAAADP////8BAf////8AAAAAFWCJCgIAAAAA" +
           "ABIAAABDbGllbnRBdWRpdEVudHJ5SWQBAKZdAC4ARKZdAAAADP////8BAf////8AAAAAFWCJCgIAAAAA" +
           "AAwAAABDbGllbnRVc2VySWQBAKddAC4ARKddAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABT" +
           "ZXJ2ZXJVcmkBAKhdAC4ARKhdAAABAMdc/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAE9iamVjdElk" +
           "AQC6XQAuAES6XQAAABH/////AQH/////AAAAABVgiQoCAAAAAAAIAAAATWV0aG9kSWQBALtdAC4ARLtd" +
           "AAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABTdGF0dXNDb2RlSWQBAL5dAC4ARL5dAAAAE///" +
           "//8BAf////8AAAAAF2CJCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEAv10ALgBEv10AAAEAKAEBAAAA" +
           "AQAAAAAAAAABAf////8AAAAAF2CJCgIAAAAAAA8AAABPdXRwdXRBcmd1bWVudHMBAFRkAC4ARFRkAAAB" +
           "ACgBAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<NodeId> ObjectId
        {
            get
            {
                return m_objectId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_objectId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_objectId = value;
            }
        }

        /// <remarks />
        public PropertyState<NodeId> MethodId
        {
            get
            {
                return m_methodId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_methodId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_methodId = value;
            }
        }

        /// <remarks />
        public PropertyState<StatusCode> StatusCodeId
        {
            get
            {
                return m_statusCodeId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_statusCodeId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_statusCodeId = value;
            }
        }

        /// <remarks />
        public PropertyState<Argument[]> InputArguments
        {
            get
            {
                return m_inputArguments;
            }

            set
            {
                if (!Object.ReferenceEquals(m_inputArguments, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_inputArguments = value;
            }
        }

        /// <remarks />
        public PropertyState<Argument[]> OutputArguments
        {
            get
            {
                return m_outputArguments;
            }

            set
            {
                if (!Object.ReferenceEquals(m_outputArguments, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_outputArguments = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_objectId != null)
            {
                children.Add(m_objectId);
            }

            if (m_methodId != null)
            {
                children.Add(m_methodId);
            }

            if (m_statusCodeId != null)
            {
                children.Add(m_statusCodeId);
            }

            if (m_inputArguments != null)
            {
                children.Add(m_inputArguments);
            }

            if (m_outputArguments != null)
            {
                children.Add(m_outputArguments);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ObjectId:
                {
                    if (createOrReplace)
                    {
                        if (ObjectId == null)
                        {
                            if (replacement == null)
                            {
                                ObjectId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                ObjectId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = ObjectId;
                    break;
                }

                case Opc.Ua.BrowseNames.MethodId:
                {
                    if (createOrReplace)
                    {
                        if (MethodId == null)
                        {
                            if (replacement == null)
                            {
                                MethodId = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                MethodId = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = MethodId;
                    break;
                }

                case Opc.Ua.BrowseNames.StatusCodeId:
                {
                    if (createOrReplace)
                    {
                        if (StatusCodeId == null)
                        {
                            if (replacement == null)
                            {
                                StatusCodeId = new PropertyState<StatusCode>(this);
                            }
                            else
                            {
                                StatusCodeId = (PropertyState<StatusCode>)replacement;
                            }
                        }
                    }

                    instance = StatusCodeId;
                    break;
                }

                case Opc.Ua.BrowseNames.InputArguments:
                {
                    if (createOrReplace)
                    {
                        if (InputArguments == null)
                        {
                            if (replacement == null)
                            {
                                InputArguments = new PropertyState<Argument[]>(this);
                            }
                            else
                            {
                                InputArguments = (PropertyState<Argument[]>)replacement;
                            }
                        }
                    }

                    instance = InputArguments;
                    break;
                }

                case Opc.Ua.BrowseNames.OutputArguments:
                {
                    if (createOrReplace)
                    {
                        if (OutputArguments == null)
                        {
                            if (replacement == null)
                            {
                                OutputArguments = new PropertyState<Argument[]>(this);
                            }
                            else
                            {
                                OutputArguments = (PropertyState<Argument[]>)replacement;
                            }
                        }
                    }

                    instance = OutputArguments;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<NodeId> m_objectId;
        private PropertyState<NodeId> m_methodId;
        private PropertyState<StatusCode> m_statusCodeId;
        private PropertyState<Argument[]> m_inputArguments;
        private PropertyState<Argument[]> m_outputArguments;
        #endregion
    }
    #endif
    #endregion

    #region AggregateFunctionState Class
    #if (!OPCUA_EXCLUDE_AggregateFunctionState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AggregateFunctionState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public AggregateFunctionState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.AggregateFunctionType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHQAAAEFnZ3JlZ2F0ZUZ1bmN0aW9uVHlwZUluc3RhbmNlAQAkCQEAJAkk" +
           "CQAA/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ServerVendorCapabilityState Class
    #if (!OPCUA_EXCLUDE_ServerVendorCapabilityState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerVendorCapabilityState : BaseDataVariableState
    {
        #region Constructors
        /// <remarks />
        public ServerVendorCapabilityState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ServerVendorCapabilityType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.BaseDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAIgAAAFNlcnZlclZlbmRvckNhcGFiaWxpdHlUeXBlSW5zdGFuY2UBAFkI" +
           "AQBZCFkIAAAAGP////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }

    #region ServerVendorCapabilityState<T> Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ServerVendorCapabilityState<T> : ServerVendorCapabilityState
    {
        #region Constructors
        /// <remarks />
        public ServerVendorCapabilityState(NodeState parent) : base(parent)
        {
            Value = default(T);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);

            Value = default(T);
            DataType = TypeInfo.GetDataTypeId(typeof(T));
            ValueRank = TypeInfo.GetValueRank(typeof(T));
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public new T Value
        {
            get
            {
                return CheckTypeBeforeCast<T>(((BaseVariableState)this).Value, true);
            }

            set
            {
                ((BaseVariableState)this).Value = value;
            }
        }
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region ServerStatusState Class
    #if (!OPCUA_EXCLUDE_ServerStatusState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerStatusState : BaseDataVariableState<ServerStatusDataType>
    {
        #region Constructors
        /// <remarks />
        public ServerStatusState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ServerStatusType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.ServerStatusDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGAAAAFNlcnZlclN0YXR1c1R5cGVJbnN0YW5jZQEAWggBAFoIWggAAAEA" +
           "XgP/////AQH/////BgAAABVgiQoCAAAAAAAJAAAAU3RhcnRUaW1lAQBbCAAvAD9bCAAAAQAmAf////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAsAAABDdXJyZW50VGltZQEAXAgALwA/XAgAAAEAJgH/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAFAAAAU3RhdGUBAF0IAC8AP10IAAABAFQD/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAACQAAAEJ1aWxkSW5mbwEAXggALwEA6wteCAAAAQBSAf////8BAf////8GAAAAFXCJCgIAAAAAAAoA" +
           "AABQcm9kdWN0VXJpAQByDgAvAD9yDgAAAAz/////AQEAAAAAAECPQP////8AAAAAFXCJCgIAAAAAABAA" +
           "AABNYW51ZmFjdHVyZXJOYW1lAQBzDgAvAD9zDgAAAAz/////AQEAAAAAAECPQP////8AAAAAFXCJCgIA" +
           "AAAAAAsAAABQcm9kdWN0TmFtZQEAdA4ALwA/dA4AAAAM/////wEBAAAAAABAj0D/////AAAAABVwiQoC" +
           "AAAAAAAPAAAAU29mdHdhcmVWZXJzaW9uAQB1DgAvAD91DgAAAAz/////AQEAAAAAAECPQP////8AAAAA" +
           "FXCJCgIAAAAAAAsAAABCdWlsZE51bWJlcgEAdg4ALwA/dg4AAAAM/////wEBAAAAAABAj0D/////AAAA" +
           "ABVwiQoCAAAAAAAJAAAAQnVpbGREYXRlAQB3DgAvAD93DgAAAQAmAf////8BAQAAAAAAQI9A/////wAA" +
           "AAAVYIkKAgAAAAAAEwAAAFNlY29uZHNUaWxsU2h1dGRvd24BAMAKAC8AP8AKAAAAB/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAA4AAABTaHV0ZG93blJlYXNvbgEAwQoALwA/wQoAAAAV/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<DateTime> StartTime
        {
            get
            {
                return m_startTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_startTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_startTime = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<DateTime> CurrentTime
        {
            get
            {
                return m_currentTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentTime = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServerState> State
        {
            get
            {
                return m_state;
            }

            set
            {
                if (!Object.ReferenceEquals(m_state, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_state = value;
            }
        }

        /// <remarks />
        public BuildInfoVariableState BuildInfo
        {
            get
            {
                return m_buildInfo;
            }

            set
            {
                if (!Object.ReferenceEquals(m_buildInfo, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_buildInfo = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SecondsTillShutdown
        {
            get
            {
                return m_secondsTillShutdown;
            }

            set
            {
                if (!Object.ReferenceEquals(m_secondsTillShutdown, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_secondsTillShutdown = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<LocalizedText> ShutdownReason
        {
            get
            {
                return m_shutdownReason;
            }

            set
            {
                if (!Object.ReferenceEquals(m_shutdownReason, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_shutdownReason = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_startTime != null)
            {
                children.Add(m_startTime);
            }

            if (m_currentTime != null)
            {
                children.Add(m_currentTime);
            }

            if (m_state != null)
            {
                children.Add(m_state);
            }

            if (m_buildInfo != null)
            {
                children.Add(m_buildInfo);
            }

            if (m_secondsTillShutdown != null)
            {
                children.Add(m_secondsTillShutdown);
            }

            if (m_shutdownReason != null)
            {
                children.Add(m_shutdownReason);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.StartTime:
                {
                    if (createOrReplace)
                    {
                        if (StartTime == null)
                        {
                            if (replacement == null)
                            {
                                StartTime = new BaseDataVariableState<DateTime>(this);
                            }
                            else
                            {
                                StartTime = (BaseDataVariableState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = StartTime;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentTime:
                {
                    if (createOrReplace)
                    {
                        if (CurrentTime == null)
                        {
                            if (replacement == null)
                            {
                                CurrentTime = new BaseDataVariableState<DateTime>(this);
                            }
                            else
                            {
                                CurrentTime = (BaseDataVariableState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = CurrentTime;
                    break;
                }

                case Opc.Ua.BrowseNames.State:
                {
                    if (createOrReplace)
                    {
                        if (State == null)
                        {
                            if (replacement == null)
                            {
                                State = new BaseDataVariableState<ServerState>(this);
                            }
                            else
                            {
                                State = (BaseDataVariableState<ServerState>)replacement;
                            }
                        }
                    }

                    instance = State;
                    break;
                }

                case Opc.Ua.BrowseNames.BuildInfo:
                {
                    if (createOrReplace)
                    {
                        if (BuildInfo == null)
                        {
                            if (replacement == null)
                            {
                                BuildInfo = new BuildInfoVariableState(this);
                            }
                            else
                            {
                                BuildInfo = (BuildInfoVariableState)replacement;
                            }
                        }
                    }

                    instance = BuildInfo;
                    break;
                }

                case Opc.Ua.BrowseNames.SecondsTillShutdown:
                {
                    if (createOrReplace)
                    {
                        if (SecondsTillShutdown == null)
                        {
                            if (replacement == null)
                            {
                                SecondsTillShutdown = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SecondsTillShutdown = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SecondsTillShutdown;
                    break;
                }

                case Opc.Ua.BrowseNames.ShutdownReason:
                {
                    if (createOrReplace)
                    {
                        if (ShutdownReason == null)
                        {
                            if (replacement == null)
                            {
                                ShutdownReason = new BaseDataVariableState<LocalizedText>(this);
                            }
                            else
                            {
                                ShutdownReason = (BaseDataVariableState<LocalizedText>)replacement;
                            }
                        }
                    }

                    instance = ShutdownReason;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<DateTime> m_startTime;
        private BaseDataVariableState<DateTime> m_currentTime;
        private BaseDataVariableState<ServerState> m_state;
        private BuildInfoVariableState m_buildInfo;
        private BaseDataVariableState<uint> m_secondsTillShutdown;
        private BaseDataVariableState<LocalizedText> m_shutdownReason;
        #endregion
    }

    #region ServerStatusValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ServerStatusValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public ServerStatusValue(ServerStatusState variable, ServerStatusDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new ServerStatusDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public ServerStatusState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public ServerStatusDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(ServerStatusState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.StartTime;
                instance.OnReadValue = OnRead_StartTime;
                instance.OnSimpleWriteValue = OnWrite_StartTime;
                updateList.Add(instance);
                instance = m_variable.CurrentTime;
                instance.OnReadValue = OnRead_CurrentTime;
                instance.OnSimpleWriteValue = OnWrite_CurrentTime;
                updateList.Add(instance);
                instance = m_variable.State;
                instance.OnReadValue = OnRead_State;
                instance.OnSimpleWriteValue = OnWrite_State;
                updateList.Add(instance);
                instance = m_variable.BuildInfo;
                instance.OnReadValue = OnRead_BuildInfo;
                instance.OnSimpleWriteValue = OnWrite_BuildInfo;
                updateList.Add(instance);
                instance = m_variable.SecondsTillShutdown;
                instance.OnReadValue = OnRead_SecondsTillShutdown;
                instance.OnSimpleWriteValue = OnWrite_SecondsTillShutdown;
                updateList.Add(instance);
                instance = m_variable.ShutdownReason;
                instance.OnReadValue = OnRead_ShutdownReason;
                instance.OnSimpleWriteValue = OnWrite_ShutdownReason;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (ServerStatusDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region StartTime Access Methods
        /// <remarks />
        private ServiceResult OnRead_StartTime(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.StartTime;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_StartTime(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.StartTime = (DateTime)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentTime Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentTime(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentTime;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentTime(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentTime = (DateTime)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region State Access Methods
        /// <remarks />
        private ServiceResult OnRead_State(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.State;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_State(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.State = (ServerState)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region BuildInfo Access Methods
        /// <remarks />
        private ServiceResult OnRead_BuildInfo(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.BuildInfo;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_BuildInfo(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.BuildInfo = (BuildInfo)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SecondsTillShutdown Access Methods
        /// <remarks />
        private ServiceResult OnRead_SecondsTillShutdown(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SecondsTillShutdown;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SecondsTillShutdown(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SecondsTillShutdown = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ShutdownReason Access Methods
        /// <remarks />
        private ServiceResult OnRead_ShutdownReason(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ShutdownReason;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ShutdownReason(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ShutdownReason = (LocalizedText)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private ServerStatusDataType m_value;
        private ServerStatusState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region BuildInfoVariableState Class
    #if (!OPCUA_EXCLUDE_BuildInfoVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class BuildInfoVariableState : BaseDataVariableState<BuildInfo>
    {
        #region Constructors
        /// <remarks />
        public BuildInfoVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.BuildInfoType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.BuildInfo, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAFQAAAEJ1aWxkSW5mb1R5cGVJbnN0YW5jZQEA6wsBAOsL6wsAAAEAUgH/" +
           "////AQH/////BgAAABVwiQoCAAAAAAAKAAAAUHJvZHVjdFVyaQEA7AsALwA/7AsAAAAM/////wEBAAAA" +
           "AABAj0D/////AAAAABVwiQoCAAAAAAAQAAAATWFudWZhY3R1cmVyTmFtZQEA7QsALwA/7QsAAAAM////" +
           "/wEBAAAAAABAj0D/////AAAAABVwiQoCAAAAAAALAAAAUHJvZHVjdE5hbWUBAO4LAC8AP+4LAAAADP//" +
           "//8BAQAAAAAAQI9A/////wAAAAAVcIkKAgAAAAAADwAAAFNvZnR3YXJlVmVyc2lvbgEA7wsALwA/7wsA" +
           "AAAM/////wEBAAAAAABAj0D/////AAAAABVwiQoCAAAAAAALAAAAQnVpbGROdW1iZXIBAPALAC8AP/AL" +
           "AAAADP////8BAQAAAAAAQI9A/////wAAAAAVcIkKAgAAAAAACQAAAEJ1aWxkRGF0ZQEA8QsALwA/8QsA" +
           "AAEAJgH/////AQEAAAAAAECPQP////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<string> ProductUri
        {
            get
            {
                return m_productUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_productUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_productUri = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> ManufacturerName
        {
            get
            {
                return m_manufacturerName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_manufacturerName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_manufacturerName = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> ProductName
        {
            get
            {
                return m_productName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_productName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_productName = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> SoftwareVersion
        {
            get
            {
                return m_softwareVersion;
            }

            set
            {
                if (!Object.ReferenceEquals(m_softwareVersion, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_softwareVersion = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> BuildNumber
        {
            get
            {
                return m_buildNumber;
            }

            set
            {
                if (!Object.ReferenceEquals(m_buildNumber, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_buildNumber = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<DateTime> BuildDate
        {
            get
            {
                return m_buildDate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_buildDate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_buildDate = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_productUri != null)
            {
                children.Add(m_productUri);
            }

            if (m_manufacturerName != null)
            {
                children.Add(m_manufacturerName);
            }

            if (m_productName != null)
            {
                children.Add(m_productName);
            }

            if (m_softwareVersion != null)
            {
                children.Add(m_softwareVersion);
            }

            if (m_buildNumber != null)
            {
                children.Add(m_buildNumber);
            }

            if (m_buildDate != null)
            {
                children.Add(m_buildDate);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ProductUri:
                {
                    if (createOrReplace)
                    {
                        if (ProductUri == null)
                        {
                            if (replacement == null)
                            {
                                ProductUri = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                ProductUri = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = ProductUri;
                    break;
                }

                case Opc.Ua.BrowseNames.ManufacturerName:
                {
                    if (createOrReplace)
                    {
                        if (ManufacturerName == null)
                        {
                            if (replacement == null)
                            {
                                ManufacturerName = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                ManufacturerName = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = ManufacturerName;
                    break;
                }

                case Opc.Ua.BrowseNames.ProductName:
                {
                    if (createOrReplace)
                    {
                        if (ProductName == null)
                        {
                            if (replacement == null)
                            {
                                ProductName = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                ProductName = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = ProductName;
                    break;
                }

                case Opc.Ua.BrowseNames.SoftwareVersion:
                {
                    if (createOrReplace)
                    {
                        if (SoftwareVersion == null)
                        {
                            if (replacement == null)
                            {
                                SoftwareVersion = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                SoftwareVersion = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = SoftwareVersion;
                    break;
                }

                case Opc.Ua.BrowseNames.BuildNumber:
                {
                    if (createOrReplace)
                    {
                        if (BuildNumber == null)
                        {
                            if (replacement == null)
                            {
                                BuildNumber = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                BuildNumber = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = BuildNumber;
                    break;
                }

                case Opc.Ua.BrowseNames.BuildDate:
                {
                    if (createOrReplace)
                    {
                        if (BuildDate == null)
                        {
                            if (replacement == null)
                            {
                                BuildDate = new BaseDataVariableState<DateTime>(this);
                            }
                            else
                            {
                                BuildDate = (BaseDataVariableState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = BuildDate;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<string> m_productUri;
        private BaseDataVariableState<string> m_manufacturerName;
        private BaseDataVariableState<string> m_productName;
        private BaseDataVariableState<string> m_softwareVersion;
        private BaseDataVariableState<string> m_buildNumber;
        private BaseDataVariableState<DateTime> m_buildDate;
        #endregion
    }

    #region BuildInfoVariableValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class BuildInfoVariableValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public BuildInfoVariableValue(BuildInfoVariableState variable, BuildInfo value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new BuildInfo();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public BuildInfoVariableState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public BuildInfo Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(BuildInfoVariableState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.ProductUri;
                instance.OnReadValue = OnRead_ProductUri;
                instance.OnSimpleWriteValue = OnWrite_ProductUri;
                updateList.Add(instance);
                instance = m_variable.ManufacturerName;
                instance.OnReadValue = OnRead_ManufacturerName;
                instance.OnSimpleWriteValue = OnWrite_ManufacturerName;
                updateList.Add(instance);
                instance = m_variable.ProductName;
                instance.OnReadValue = OnRead_ProductName;
                instance.OnSimpleWriteValue = OnWrite_ProductName;
                updateList.Add(instance);
                instance = m_variable.SoftwareVersion;
                instance.OnReadValue = OnRead_SoftwareVersion;
                instance.OnSimpleWriteValue = OnWrite_SoftwareVersion;
                updateList.Add(instance);
                instance = m_variable.BuildNumber;
                instance.OnReadValue = OnRead_BuildNumber;
                instance.OnSimpleWriteValue = OnWrite_BuildNumber;
                updateList.Add(instance);
                instance = m_variable.BuildDate;
                instance.OnReadValue = OnRead_BuildDate;
                instance.OnSimpleWriteValue = OnWrite_BuildDate;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (BuildInfo)Write(value);
            }

            return ServiceResult.Good;
        }

        #region ProductUri Access Methods
        /// <remarks />
        private ServiceResult OnRead_ProductUri(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ProductUri;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ProductUri(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ProductUri = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ManufacturerName Access Methods
        /// <remarks />
        private ServiceResult OnRead_ManufacturerName(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ManufacturerName;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ManufacturerName(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ManufacturerName = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ProductName Access Methods
        /// <remarks />
        private ServiceResult OnRead_ProductName(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ProductName;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ProductName(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ProductName = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SoftwareVersion Access Methods
        /// <remarks />
        private ServiceResult OnRead_SoftwareVersion(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SoftwareVersion;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SoftwareVersion(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SoftwareVersion = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region BuildNumber Access Methods
        /// <remarks />
        private ServiceResult OnRead_BuildNumber(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.BuildNumber;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_BuildNumber(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.BuildNumber = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region BuildDate Access Methods
        /// <remarks />
        private ServiceResult OnRead_BuildDate(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.BuildDate;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_BuildDate(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.BuildDate = (DateTime)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private BuildInfo m_value;
        private BuildInfoVariableState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region ServerDiagnosticsSummaryState Class
    #if (!OPCUA_EXCLUDE_ServerDiagnosticsSummaryState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ServerDiagnosticsSummaryState : BaseDataVariableState<ServerDiagnosticsSummaryDataType>
    {
        #region Constructors
        /// <remarks />
        public ServerDiagnosticsSummaryState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ServerDiagnosticsSummaryType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.ServerDiagnosticsSummaryDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJAAAAFNlcnZlckRpYWdub3N0aWNzU3VtbWFyeVR5cGVJbnN0YW5jZQEA" +
           "ZggBAGYIZggAAAEAWwP/////AQH/////DAAAABVgiQoCAAAAAAAPAAAAU2VydmVyVmlld0NvdW50AQBn" +
           "CAAvAD9nCAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAATAAAAQ3VycmVudFNlc3Npb25Db3VudAEA" +
           "aAgALwA/aAgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFQAAAEN1bXVsYXRlZFNlc3Npb25Db3Vu" +
           "dAEAaQgALwA/aQgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAHAAAAFNlY3VyaXR5UmVqZWN0ZWRT" +
           "ZXNzaW9uQ291bnQBAGoIAC8AP2oIAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABQAAABSZWplY3Rl" +
           "ZFNlc3Npb25Db3VudAEAawgALwA/awgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEwAAAFNlc3Np" +
           "b25UaW1lb3V0Q291bnQBAGwIAC8AP2wIAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABEAAABTZXNz" +
           "aW9uQWJvcnRDb3VudAEAbQgALwA/bQgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFwAAAFB1Ymxp" +
           "c2hpbmdJbnRlcnZhbENvdW50AQBvCAAvAD9vCAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAYAAAA" +
           "Q3VycmVudFN1YnNjcmlwdGlvbkNvdW50AQBwCAAvAD9wCAAAAAf/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAaAAAAQ3VtdWxhdGVkU3Vic2NyaXB0aW9uQ291bnQBAHEIAC8AP3EIAAAAB/////8BAf////8AAAAA" +
           "FWCJCgIAAAAAAB0AAABTZWN1cml0eVJlamVjdGVkUmVxdWVzdHNDb3VudAEAcggALwA/cggAAAAH////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAFQAAAFJlamVjdGVkUmVxdWVzdHNDb3VudAEAcwgALwA/cwgAAAAH" +
           "/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<uint> ServerViewCount
        {
            get
            {
                return m_serverViewCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverViewCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverViewCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentSessionCount
        {
            get
            {
                return m_currentSessionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentSessionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentSessionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CumulatedSessionCount
        {
            get
            {
                return m_cumulatedSessionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_cumulatedSessionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_cumulatedSessionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SecurityRejectedSessionCount
        {
            get
            {
                return m_securityRejectedSessionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityRejectedSessionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityRejectedSessionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> RejectedSessionCount
        {
            get
            {
                return m_rejectedSessionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_rejectedSessionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_rejectedSessionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SessionTimeoutCount
        {
            get
            {
                return m_sessionTimeoutCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionTimeoutCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionTimeoutCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SessionAbortCount
        {
            get
            {
                return m_sessionAbortCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionAbortCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionAbortCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> PublishingIntervalCount
        {
            get
            {
                return m_publishingIntervalCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_publishingIntervalCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_publishingIntervalCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentSubscriptionCount
        {
            get
            {
                return m_currentSubscriptionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentSubscriptionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentSubscriptionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CumulatedSubscriptionCount
        {
            get
            {
                return m_cumulatedSubscriptionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_cumulatedSubscriptionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_cumulatedSubscriptionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SecurityRejectedRequestsCount
        {
            get
            {
                return m_securityRejectedRequestsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityRejectedRequestsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityRejectedRequestsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> RejectedRequestsCount
        {
            get
            {
                return m_rejectedRequestsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_rejectedRequestsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_rejectedRequestsCount = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_serverViewCount != null)
            {
                children.Add(m_serverViewCount);
            }

            if (m_currentSessionCount != null)
            {
                children.Add(m_currentSessionCount);
            }

            if (m_cumulatedSessionCount != null)
            {
                children.Add(m_cumulatedSessionCount);
            }

            if (m_securityRejectedSessionCount != null)
            {
                children.Add(m_securityRejectedSessionCount);
            }

            if (m_rejectedSessionCount != null)
            {
                children.Add(m_rejectedSessionCount);
            }

            if (m_sessionTimeoutCount != null)
            {
                children.Add(m_sessionTimeoutCount);
            }

            if (m_sessionAbortCount != null)
            {
                children.Add(m_sessionAbortCount);
            }

            if (m_publishingIntervalCount != null)
            {
                children.Add(m_publishingIntervalCount);
            }

            if (m_currentSubscriptionCount != null)
            {
                children.Add(m_currentSubscriptionCount);
            }

            if (m_cumulatedSubscriptionCount != null)
            {
                children.Add(m_cumulatedSubscriptionCount);
            }

            if (m_securityRejectedRequestsCount != null)
            {
                children.Add(m_securityRejectedRequestsCount);
            }

            if (m_rejectedRequestsCount != null)
            {
                children.Add(m_rejectedRequestsCount);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ServerViewCount:
                {
                    if (createOrReplace)
                    {
                        if (ServerViewCount == null)
                        {
                            if (replacement == null)
                            {
                                ServerViewCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                ServerViewCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = ServerViewCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentSessionCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentSessionCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentSessionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentSessionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentSessionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CumulatedSessionCount:
                {
                    if (createOrReplace)
                    {
                        if (CumulatedSessionCount == null)
                        {
                            if (replacement == null)
                            {
                                CumulatedSessionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CumulatedSessionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CumulatedSessionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityRejectedSessionCount:
                {
                    if (createOrReplace)
                    {
                        if (SecurityRejectedSessionCount == null)
                        {
                            if (replacement == null)
                            {
                                SecurityRejectedSessionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SecurityRejectedSessionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SecurityRejectedSessionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RejectedSessionCount:
                {
                    if (createOrReplace)
                    {
                        if (RejectedSessionCount == null)
                        {
                            if (replacement == null)
                            {
                                RejectedSessionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                RejectedSessionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = RejectedSessionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionTimeoutCount:
                {
                    if (createOrReplace)
                    {
                        if (SessionTimeoutCount == null)
                        {
                            if (replacement == null)
                            {
                                SessionTimeoutCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SessionTimeoutCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SessionTimeoutCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionAbortCount:
                {
                    if (createOrReplace)
                    {
                        if (SessionAbortCount == null)
                        {
                            if (replacement == null)
                            {
                                SessionAbortCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SessionAbortCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SessionAbortCount;
                    break;
                }

                case Opc.Ua.BrowseNames.PublishingIntervalCount:
                {
                    if (createOrReplace)
                    {
                        if (PublishingIntervalCount == null)
                        {
                            if (replacement == null)
                            {
                                PublishingIntervalCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                PublishingIntervalCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = PublishingIntervalCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentSubscriptionCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentSubscriptionCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentSubscriptionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentSubscriptionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentSubscriptionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CumulatedSubscriptionCount:
                {
                    if (createOrReplace)
                    {
                        if (CumulatedSubscriptionCount == null)
                        {
                            if (replacement == null)
                            {
                                CumulatedSubscriptionCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CumulatedSubscriptionCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CumulatedSubscriptionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityRejectedRequestsCount:
                {
                    if (createOrReplace)
                    {
                        if (SecurityRejectedRequestsCount == null)
                        {
                            if (replacement == null)
                            {
                                SecurityRejectedRequestsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SecurityRejectedRequestsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SecurityRejectedRequestsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RejectedRequestsCount:
                {
                    if (createOrReplace)
                    {
                        if (RejectedRequestsCount == null)
                        {
                            if (replacement == null)
                            {
                                RejectedRequestsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                RejectedRequestsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = RejectedRequestsCount;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<uint> m_serverViewCount;
        private BaseDataVariableState<uint> m_currentSessionCount;
        private BaseDataVariableState<uint> m_cumulatedSessionCount;
        private BaseDataVariableState<uint> m_securityRejectedSessionCount;
        private BaseDataVariableState<uint> m_rejectedSessionCount;
        private BaseDataVariableState<uint> m_sessionTimeoutCount;
        private BaseDataVariableState<uint> m_sessionAbortCount;
        private BaseDataVariableState<uint> m_publishingIntervalCount;
        private BaseDataVariableState<uint> m_currentSubscriptionCount;
        private BaseDataVariableState<uint> m_cumulatedSubscriptionCount;
        private BaseDataVariableState<uint> m_securityRejectedRequestsCount;
        private BaseDataVariableState<uint> m_rejectedRequestsCount;
        #endregion
    }

    #region ServerDiagnosticsSummaryValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ServerDiagnosticsSummaryValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public ServerDiagnosticsSummaryValue(ServerDiagnosticsSummaryState variable, ServerDiagnosticsSummaryDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new ServerDiagnosticsSummaryDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public ServerDiagnosticsSummaryState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public ServerDiagnosticsSummaryDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(ServerDiagnosticsSummaryState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.ServerViewCount;
                instance.OnReadValue = OnRead_ServerViewCount;
                instance.OnSimpleWriteValue = OnWrite_ServerViewCount;
                updateList.Add(instance);
                instance = m_variable.CurrentSessionCount;
                instance.OnReadValue = OnRead_CurrentSessionCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentSessionCount;
                updateList.Add(instance);
                instance = m_variable.CumulatedSessionCount;
                instance.OnReadValue = OnRead_CumulatedSessionCount;
                instance.OnSimpleWriteValue = OnWrite_CumulatedSessionCount;
                updateList.Add(instance);
                instance = m_variable.SecurityRejectedSessionCount;
                instance.OnReadValue = OnRead_SecurityRejectedSessionCount;
                instance.OnSimpleWriteValue = OnWrite_SecurityRejectedSessionCount;
                updateList.Add(instance);
                instance = m_variable.RejectedSessionCount;
                instance.OnReadValue = OnRead_RejectedSessionCount;
                instance.OnSimpleWriteValue = OnWrite_RejectedSessionCount;
                updateList.Add(instance);
                instance = m_variable.SessionTimeoutCount;
                instance.OnReadValue = OnRead_SessionTimeoutCount;
                instance.OnSimpleWriteValue = OnWrite_SessionTimeoutCount;
                updateList.Add(instance);
                instance = m_variable.SessionAbortCount;
                instance.OnReadValue = OnRead_SessionAbortCount;
                instance.OnSimpleWriteValue = OnWrite_SessionAbortCount;
                updateList.Add(instance);
                instance = m_variable.CurrentSubscriptionCount;
                instance.OnReadValue = OnRead_CurrentSubscriptionCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentSubscriptionCount;
                updateList.Add(instance);
                instance = m_variable.CumulatedSubscriptionCount;
                instance.OnReadValue = OnRead_CumulatedSubscriptionCount;
                instance.OnSimpleWriteValue = OnWrite_CumulatedSubscriptionCount;
                updateList.Add(instance);
                instance = m_variable.PublishingIntervalCount;
                instance.OnReadValue = OnRead_PublishingIntervalCount;
                instance.OnSimpleWriteValue = OnWrite_PublishingIntervalCount;
                updateList.Add(instance);
                instance = m_variable.SecurityRejectedRequestsCount;
                instance.OnReadValue = OnRead_SecurityRejectedRequestsCount;
                instance.OnSimpleWriteValue = OnWrite_SecurityRejectedRequestsCount;
                updateList.Add(instance);
                instance = m_variable.RejectedRequestsCount;
                instance.OnReadValue = OnRead_RejectedRequestsCount;
                instance.OnSimpleWriteValue = OnWrite_RejectedRequestsCount;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (ServerDiagnosticsSummaryDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region ServerViewCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_ServerViewCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ServerViewCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ServerViewCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ServerViewCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentSessionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentSessionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentSessionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentSessionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentSessionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CumulatedSessionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CumulatedSessionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CumulatedSessionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CumulatedSessionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CumulatedSessionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SecurityRejectedSessionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SecurityRejectedSessionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SecurityRejectedSessionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SecurityRejectedSessionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SecurityRejectedSessionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RejectedSessionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RejectedSessionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RejectedSessionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RejectedSessionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RejectedSessionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SessionTimeoutCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionTimeoutCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionTimeoutCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionTimeoutCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionTimeoutCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SessionAbortCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionAbortCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionAbortCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionAbortCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionAbortCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentSubscriptionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentSubscriptionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentSubscriptionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentSubscriptionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentSubscriptionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CumulatedSubscriptionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CumulatedSubscriptionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CumulatedSubscriptionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CumulatedSubscriptionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CumulatedSubscriptionCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region PublishingIntervalCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_PublishingIntervalCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.PublishingIntervalCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_PublishingIntervalCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.PublishingIntervalCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SecurityRejectedRequestsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SecurityRejectedRequestsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SecurityRejectedRequestsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SecurityRejectedRequestsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SecurityRejectedRequestsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RejectedRequestsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RejectedRequestsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RejectedRequestsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RejectedRequestsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RejectedRequestsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private ServerDiagnosticsSummaryDataType m_value;
        private ServerDiagnosticsSummaryState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region SamplingIntervalDiagnosticsArrayState Class
    #if (!OPCUA_EXCLUDE_SamplingIntervalDiagnosticsArrayState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SamplingIntervalDiagnosticsArrayState : BaseDataVariableState<SamplingIntervalDiagnosticsDataType[]>
    {
        #region Constructors
        /// <remarks />
        public SamplingIntervalDiagnosticsArrayState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SamplingIntervalDiagnosticsArrayType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SamplingIntervalDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.OneDimension;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8XYIkCAgAAAAAALAAAAFNhbXBsaW5nSW50ZXJ2YWxEaWFnbm9zdGljc0FycmF5VHlwZUlu" +
           "c3RhbmNlAQB0CAEAdAh0CAAAAQBYAwEAAAABAAAAAAAAAAEB/////wEAAAAVYIkKAgAAAAAAGwAAAFNh" +
           "bXBsaW5nSW50ZXJ2YWxEaWFnbm9zdGljcwEA6zEALwEAdQjrMQAAAQBYA/////8BAf////8EAAAAFWCJ" +
           "CgIAAAAAABAAAABTYW1wbGluZ0ludGVydmFsAQDsMQAvAD/sMQAAAQAiAf////8BAf////8AAAAAFWCJ" +
           "CgIAAAAAABoAAABTYW1wbGVkTW9uaXRvcmVkSXRlbXNDb3VudAEA7TEALwA/7TEAAAAH/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAHQAAAE1heFNhbXBsZWRNb25pdG9yZWRJdGVtc0NvdW50AQDuMQAvAD/uMQAA" +
           "AAf/////AQH/////AAAAABVgiQoCAAAAAAAjAAAARGlzYWJsZWRNb25pdG9yZWRJdGVtc1NhbXBsaW5n" +
           "Q291bnQBAO8xAC8AP+8xAAAAB/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region SamplingIntervalDiagnosticsState Class
    #if (!OPCUA_EXCLUDE_SamplingIntervalDiagnosticsState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SamplingIntervalDiagnosticsState : BaseDataVariableState<SamplingIntervalDiagnosticsDataType>
    {
        #region Constructors
        /// <remarks />
        public SamplingIntervalDiagnosticsState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SamplingIntervalDiagnosticsType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SamplingIntervalDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJwAAAFNhbXBsaW5nSW50ZXJ2YWxEaWFnbm9zdGljc1R5cGVJbnN0YW5j" +
           "ZQEAdQgBAHUIdQgAAAEAWAP/////AQH/////BAAAABVgiQoCAAAAAAAQAAAAU2FtcGxpbmdJbnRlcnZh" +
           "bAEAdggALwA/dggAAAEAIgH/////AQH/////AAAAABVgiQoCAAAAAAAaAAAAU2FtcGxlZE1vbml0b3Jl" +
           "ZEl0ZW1zQ291bnQBALEtAC8AP7EtAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAB0AAABNYXhTYW1w" +
           "bGVkTW9uaXRvcmVkSXRlbXNDb3VudAEAsi0ALwA/si0AAAAH/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "IwAAAERpc2FibGVkTW9uaXRvcmVkSXRlbXNTYW1wbGluZ0NvdW50AQCzLQAvAD+zLQAAAAf/////AQH/" +
           "////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<double> SamplingInterval
        {
            get
            {
                return m_samplingInterval;
            }

            set
            {
                if (!Object.ReferenceEquals(m_samplingInterval, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_samplingInterval = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SampledMonitoredItemsCount
        {
            get
            {
                return m_sampledMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sampledMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sampledMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MaxSampledMonitoredItemsCount
        {
            get
            {
                return m_maxSampledMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxSampledMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxSampledMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> DisabledMonitoredItemsSamplingCount
        {
            get
            {
                return m_disabledMonitoredItemsSamplingCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_disabledMonitoredItemsSamplingCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_disabledMonitoredItemsSamplingCount = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_samplingInterval != null)
            {
                children.Add(m_samplingInterval);
            }

            if (m_sampledMonitoredItemsCount != null)
            {
                children.Add(m_sampledMonitoredItemsCount);
            }

            if (m_maxSampledMonitoredItemsCount != null)
            {
                children.Add(m_maxSampledMonitoredItemsCount);
            }

            if (m_disabledMonitoredItemsSamplingCount != null)
            {
                children.Add(m_disabledMonitoredItemsSamplingCount);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SamplingInterval:
                {
                    if (createOrReplace)
                    {
                        if (SamplingInterval == null)
                        {
                            if (replacement == null)
                            {
                                SamplingInterval = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                SamplingInterval = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = SamplingInterval;
                    break;
                }

                case Opc.Ua.BrowseNames.SampledMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (SampledMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                SampledMonitoredItemsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SampledMonitoredItemsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SampledMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxSampledMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (MaxSampledMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                MaxSampledMonitoredItemsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MaxSampledMonitoredItemsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxSampledMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DisabledMonitoredItemsSamplingCount:
                {
                    if (createOrReplace)
                    {
                        if (DisabledMonitoredItemsSamplingCount == null)
                        {
                            if (replacement == null)
                            {
                                DisabledMonitoredItemsSamplingCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                DisabledMonitoredItemsSamplingCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = DisabledMonitoredItemsSamplingCount;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<double> m_samplingInterval;
        private BaseDataVariableState<uint> m_sampledMonitoredItemsCount;
        private BaseDataVariableState<uint> m_maxSampledMonitoredItemsCount;
        private BaseDataVariableState<uint> m_disabledMonitoredItemsSamplingCount;
        #endregion
    }

    #region SamplingIntervalDiagnosticsValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class SamplingIntervalDiagnosticsValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public SamplingIntervalDiagnosticsValue(SamplingIntervalDiagnosticsState variable, SamplingIntervalDiagnosticsDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new SamplingIntervalDiagnosticsDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public SamplingIntervalDiagnosticsState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public SamplingIntervalDiagnosticsDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(SamplingIntervalDiagnosticsState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.SamplingInterval;
                instance.OnReadValue = OnRead_SamplingInterval;
                instance.OnSimpleWriteValue = OnWrite_SamplingInterval;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (SamplingIntervalDiagnosticsDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region SamplingInterval Access Methods
        /// <remarks />
        private ServiceResult OnRead_SamplingInterval(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SamplingInterval;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SamplingInterval(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SamplingInterval = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private SamplingIntervalDiagnosticsDataType m_value;
        private SamplingIntervalDiagnosticsState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region SubscriptionDiagnosticsArrayState Class
    #if (!OPCUA_EXCLUDE_SubscriptionDiagnosticsArrayState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SubscriptionDiagnosticsArrayState : BaseDataVariableState<SubscriptionDiagnosticsDataType[]>
    {
        #region Constructors
        /// <remarks />
        public SubscriptionDiagnosticsArrayState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SubscriptionDiagnosticsArrayType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SubscriptionDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.OneDimension;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8XYIkCAgAAAAAAKAAAAFN1YnNjcmlwdGlvbkRpYWdub3N0aWNzQXJyYXlUeXBlSW5zdGFu" +
           "Y2UBAHsIAQB7CHsIAAABAGoDAQAAAAEAAAAAAAAAAQH/////AQAAABVgiQoCAAAAAAAXAAAAU3Vic2Ny" +
           "aXB0aW9uRGlhZ25vc3RpY3MBAPAxAC8BAHwI8DEAAAEAagP/////AQH/////HwAAABVgiQoCAAAAAAAJ" +
           "AAAAU2Vzc2lvbklkAQDxMQAvAD/xMQAAABH/////AQH/////AAAAABVgiQoCAAAAAAAOAAAAU3Vic2Ny" +
           "aXB0aW9uSWQBAPIxAC8AP/IxAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAAgAAABQcmlvcml0eQEA" +
           "8zEALwA/8zEAAAAD/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAFB1Ymxpc2hpbmdJbnRlcnZhbAEA" +
           "9DEALwA/9DEAAAEAIgH/////AQH/////AAAAABVgiQoCAAAAAAARAAAATWF4S2VlcEFsaXZlQ291bnQB" +
           "APUxAC8AP/UxAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABAAAABNYXhMaWZldGltZUNvdW50AQD2" +
           "MQAvAD/2MQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAaAAAATWF4Tm90aWZpY2F0aW9uc1BlclB1" +
           "Ymxpc2gBAPcxAC8AP/cxAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABEAAABQdWJsaXNoaW5nRW5h" +
           "YmxlZAEA+DEALwA/+DEAAAAB/////wEB/////wAAAAAVYIkKAgAAAAAACwAAAE1vZGlmeUNvdW50AQD5" +
           "MQAvAD/5MQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAALAAAARW5hYmxlQ291bnQBAPoxAC8AP/ox" +
           "AAAAB/////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABEaXNhYmxlQ291bnQBAPsxAC8AP/sxAAAAB///" +
           "//8BAf////8AAAAAFWCJCgIAAAAAABUAAABSZXB1Ymxpc2hSZXF1ZXN0Q291bnQBAPwxAC8AP/wxAAAA" +
           "B/////8BAf////8AAAAAFWCJCgIAAAAAABwAAABSZXB1Ymxpc2hNZXNzYWdlUmVxdWVzdENvdW50AQD9" +
           "MQAvAD/9MQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAVAAAAUmVwdWJsaXNoTWVzc2FnZUNvdW50" +
           "AQD+MQAvAD/+MQAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAUAAAAVHJhbnNmZXJSZXF1ZXN0Q291" +
           "bnQBAP8xAC8AP/8xAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABsAAABUcmFuc2ZlcnJlZFRvQWx0" +
           "Q2xpZW50Q291bnQBAAAyAC8APwAyAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABwAAABUcmFuc2Zl" +
           "cnJlZFRvU2FtZUNsaWVudENvdW50AQABMgAvAD8BMgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAT" +
           "AAAAUHVibGlzaFJlcXVlc3RDb3VudAEAAjIALwA/AjIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "HAAAAERhdGFDaGFuZ2VOb3RpZmljYXRpb25zQ291bnQBAAMyAC8APwMyAAAAB/////8BAf////8AAAAA" +
           "FWCJCgIAAAAAABcAAABFdmVudE5vdGlmaWNhdGlvbnNDb3VudAEABDIALwA/BDIAAAAH/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAEgAAAE5vdGlmaWNhdGlvbnNDb3VudAEABTIALwA/BTIAAAAH/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAFwAAAExhdGVQdWJsaXNoUmVxdWVzdENvdW50AQAGMgAvAD8GMgAAAAf/////" +
           "AQH/////AAAAABVgiQoCAAAAAAAVAAAAQ3VycmVudEtlZXBBbGl2ZUNvdW50AQAHMgAvAD8HMgAAAAf/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAUAAAAQ3VycmVudExpZmV0aW1lQ291bnQBAAgyAC8APwgyAAAA" +
           "B/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABVbmFja25vd2xlZGdlZE1lc3NhZ2VDb3VudAEACTIA" +
           "LwA/CTIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFQAAAERpc2NhcmRlZE1lc3NhZ2VDb3VudAEA" +
           "CjIALwA/CjIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAE1vbml0b3JlZEl0ZW1Db3VudAEA" +
           "CzIALwA/CzIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAGgAAAERpc2FibGVkTW9uaXRvcmVkSXRl" +
           "bUNvdW50AQAMMgAvAD8MMgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAcAAAATW9uaXRvcmluZ1F1" +
           "ZXVlT3ZlcmZsb3dDb3VudAEADTIALwA/DTIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAEgAAAE5l" +
           "eHRTZXF1ZW5jZU51bWJlcgEADjIALwA/DjIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFwAAAEV2" +
           "ZW50UXVldWVPdmVyZmxvd0NvdW50AQAPMgAvAD8PMgAAAAf/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region SubscriptionDiagnosticsState Class
    #if (!OPCUA_EXCLUDE_SubscriptionDiagnosticsState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SubscriptionDiagnosticsState : BaseDataVariableState<SubscriptionDiagnosticsDataType>
    {
        #region Constructors
        /// <remarks />
        public SubscriptionDiagnosticsState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SubscriptionDiagnosticsType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SubscriptionDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAIwAAAFN1YnNjcmlwdGlvbkRpYWdub3N0aWNzVHlwZUluc3RhbmNlAQB8" +
           "CAEAfAh8CAAAAQBqA/////8BAf////8fAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9uSWQBAH0IAC8AP30I" +
           "AAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAA4AAABTdWJzY3JpcHRpb25JZAEAfggALwA/fggAAAAH" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAFByaW9yaXR5AQB/CAAvAD9/CAAAAAP/////AQH/////" +
           "AAAAABVgiQoCAAAAAAASAAAAUHVibGlzaGluZ0ludGVydmFsAQCACAAvAD+ACAAAAQAiAf////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABEAAABNYXhLZWVwQWxpdmVDb3VudAEAgQgALwA/gQgAAAAH/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAEAAAAE1heExpZmV0aW1lQ291bnQBALgiAC8AP7giAAAAB/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAABoAAABNYXhOb3RpZmljYXRpb25zUGVyUHVibGlzaAEAgwgALwA/gwgAAAAH////" +
           "/wEB/////wAAAAAVYIkKAgAAAAAAEQAAAFB1Ymxpc2hpbmdFbmFibGVkAQCECAAvAD+ECAAAAAH/////" +
           "AQH/////AAAAABVgiQoCAAAAAAALAAAATW9kaWZ5Q291bnQBAIUIAC8AP4UIAAAAB/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAsAAABFbmFibGVDb3VudAEAhggALwA/hggAAAAH/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAADAAAAERpc2FibGVDb3VudAEAhwgALwA/hwgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "FQAAAFJlcHVibGlzaFJlcXVlc3RDb3VudAEAiAgALwA/iAgAAAAH/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAHAAAAFJlcHVibGlzaE1lc3NhZ2VSZXF1ZXN0Q291bnQBAIkIAC8AP4kIAAAAB/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAABUAAABSZXB1Ymxpc2hNZXNzYWdlQ291bnQBAIoIAC8AP4oIAAAAB/////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAABQAAABUcmFuc2ZlclJlcXVlc3RDb3VudAEAiwgALwA/iwgAAAAH/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAAGwAAAFRyYW5zZmVycmVkVG9BbHRDbGllbnRDb3VudAEAjAgALwA/jAgA" +
           "AAAH/////wEB/////wAAAAAVYIkKAgAAAAAAHAAAAFRyYW5zZmVycmVkVG9TYW1lQ2xpZW50Q291bnQB" +
           "AI0IAC8AP40IAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABMAAABQdWJsaXNoUmVxdWVzdENvdW50" +
           "AQCOCAAvAD+OCAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAcAAAARGF0YUNoYW5nZU5vdGlmaWNh" +
           "dGlvbnNDb3VudAEAjwgALwA/jwgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFwAAAEV2ZW50Tm90" +
           "aWZpY2F0aW9uc0NvdW50AQC2CwAvAD+2CwAAAAf/////AQH/////AAAAABVgiQoCAAAAAAASAAAATm90" +
           "aWZpY2F0aW9uc0NvdW50AQCRCAAvAD+RCAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAXAAAATGF0" +
           "ZVB1Ymxpc2hSZXF1ZXN0Q291bnQBALkiAC8AP7kiAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABUA" +
           "AABDdXJyZW50S2VlcEFsaXZlQ291bnQBALoiAC8AP7oiAAAAB/////8BAf////8AAAAAFWCJCgIAAAAA" +
           "ABQAAABDdXJyZW50TGlmZXRpbWVDb3VudAEAuyIALwA/uyIAAAAH/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAGgAAAFVuYWNrbm93bGVkZ2VkTWVzc2FnZUNvdW50AQC8IgAvAD+8IgAAAAf/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAAVAAAARGlzY2FyZGVkTWVzc2FnZUNvdW50AQC9IgAvAD+9IgAAAAf/////AQH/////" +
           "AAAAABVgiQoCAAAAAAASAAAATW9uaXRvcmVkSXRlbUNvdW50AQC+IgAvAD++IgAAAAf/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAaAAAARGlzYWJsZWRNb25pdG9yZWRJdGVtQ291bnQBAL8iAC8AP78iAAAAB///" +
           "//8BAf////8AAAAAFWCJCgIAAAAAABwAAABNb25pdG9yaW5nUXVldWVPdmVyZmxvd0NvdW50AQDAIgAv" +
           "AD/AIgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAASAAAATmV4dFNlcXVlbmNlTnVtYmVyAQDBIgAv" +
           "AD/BIgAAAAf/////AQH/////AAAAABVgiQoCAAAAAAAXAAAARXZlbnRRdWV1ZU92ZXJmbG93Q291bnQB" +
           "AMYiAC8AP8YiAAAAB/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<NodeId> SessionId
        {
            get
            {
                return m_sessionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionId = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> SubscriptionId
        {
            get
            {
                return m_subscriptionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_subscriptionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_subscriptionId = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<byte> Priority
        {
            get
            {
                return m_priority;
            }

            set
            {
                if (!Object.ReferenceEquals(m_priority, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_priority = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> PublishingInterval
        {
            get
            {
                return m_publishingInterval;
            }

            set
            {
                if (!Object.ReferenceEquals(m_publishingInterval, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_publishingInterval = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MaxKeepAliveCount
        {
            get
            {
                return m_maxKeepAliveCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxKeepAliveCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxKeepAliveCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MaxLifetimeCount
        {
            get
            {
                return m_maxLifetimeCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxLifetimeCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxLifetimeCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MaxNotificationsPerPublish
        {
            get
            {
                return m_maxNotificationsPerPublish;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxNotificationsPerPublish, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxNotificationsPerPublish = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<bool> PublishingEnabled
        {
            get
            {
                return m_publishingEnabled;
            }

            set
            {
                if (!Object.ReferenceEquals(m_publishingEnabled, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_publishingEnabled = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> ModifyCount
        {
            get
            {
                return m_modifyCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_modifyCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_modifyCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> EnableCount
        {
            get
            {
                return m_enableCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_enableCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_enableCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> DisableCount
        {
            get
            {
                return m_disableCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_disableCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_disableCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> RepublishRequestCount
        {
            get
            {
                return m_republishRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_republishRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_republishRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> RepublishMessageRequestCount
        {
            get
            {
                return m_republishMessageRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_republishMessageRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_republishMessageRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> RepublishMessageCount
        {
            get
            {
                return m_republishMessageCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_republishMessageCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_republishMessageCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> TransferRequestCount
        {
            get
            {
                return m_transferRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transferRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transferRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> TransferredToAltClientCount
        {
            get
            {
                return m_transferredToAltClientCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transferredToAltClientCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transferredToAltClientCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> TransferredToSameClientCount
        {
            get
            {
                return m_transferredToSameClientCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transferredToSameClientCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transferredToSameClientCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> PublishRequestCount
        {
            get
            {
                return m_publishRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_publishRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_publishRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> DataChangeNotificationsCount
        {
            get
            {
                return m_dataChangeNotificationsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_dataChangeNotificationsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_dataChangeNotificationsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> EventNotificationsCount
        {
            get
            {
                return m_eventNotificationsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_eventNotificationsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_eventNotificationsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> NotificationsCount
        {
            get
            {
                return m_notificationsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_notificationsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_notificationsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> LatePublishRequestCount
        {
            get
            {
                return m_latePublishRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_latePublishRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_latePublishRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentKeepAliveCount
        {
            get
            {
                return m_currentKeepAliveCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentKeepAliveCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentKeepAliveCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentLifetimeCount
        {
            get
            {
                return m_currentLifetimeCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentLifetimeCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentLifetimeCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> UnacknowledgedMessageCount
        {
            get
            {
                return m_unacknowledgedMessageCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_unacknowledgedMessageCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_unacknowledgedMessageCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> DiscardedMessageCount
        {
            get
            {
                return m_discardedMessageCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_discardedMessageCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_discardedMessageCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MonitoredItemCount
        {
            get
            {
                return m_monitoredItemCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_monitoredItemCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_monitoredItemCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> DisabledMonitoredItemCount
        {
            get
            {
                return m_disabledMonitoredItemCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_disabledMonitoredItemCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_disabledMonitoredItemCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MonitoringQueueOverflowCount
        {
            get
            {
                return m_monitoringQueueOverflowCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_monitoringQueueOverflowCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_monitoringQueueOverflowCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> NextSequenceNumber
        {
            get
            {
                return m_nextSequenceNumber;
            }

            set
            {
                if (!Object.ReferenceEquals(m_nextSequenceNumber, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_nextSequenceNumber = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> EventQueueOverflowCount
        {
            get
            {
                return m_eventQueueOverflowCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_eventQueueOverflowCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_eventQueueOverflowCount = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionId != null)
            {
                children.Add(m_sessionId);
            }

            if (m_subscriptionId != null)
            {
                children.Add(m_subscriptionId);
            }

            if (m_priority != null)
            {
                children.Add(m_priority);
            }

            if (m_publishingInterval != null)
            {
                children.Add(m_publishingInterval);
            }

            if (m_maxKeepAliveCount != null)
            {
                children.Add(m_maxKeepAliveCount);
            }

            if (m_maxLifetimeCount != null)
            {
                children.Add(m_maxLifetimeCount);
            }

            if (m_maxNotificationsPerPublish != null)
            {
                children.Add(m_maxNotificationsPerPublish);
            }

            if (m_publishingEnabled != null)
            {
                children.Add(m_publishingEnabled);
            }

            if (m_modifyCount != null)
            {
                children.Add(m_modifyCount);
            }

            if (m_enableCount != null)
            {
                children.Add(m_enableCount);
            }

            if (m_disableCount != null)
            {
                children.Add(m_disableCount);
            }

            if (m_republishRequestCount != null)
            {
                children.Add(m_republishRequestCount);
            }

            if (m_republishMessageRequestCount != null)
            {
                children.Add(m_republishMessageRequestCount);
            }

            if (m_republishMessageCount != null)
            {
                children.Add(m_republishMessageCount);
            }

            if (m_transferRequestCount != null)
            {
                children.Add(m_transferRequestCount);
            }

            if (m_transferredToAltClientCount != null)
            {
                children.Add(m_transferredToAltClientCount);
            }

            if (m_transferredToSameClientCount != null)
            {
                children.Add(m_transferredToSameClientCount);
            }

            if (m_publishRequestCount != null)
            {
                children.Add(m_publishRequestCount);
            }

            if (m_dataChangeNotificationsCount != null)
            {
                children.Add(m_dataChangeNotificationsCount);
            }

            if (m_eventNotificationsCount != null)
            {
                children.Add(m_eventNotificationsCount);
            }

            if (m_notificationsCount != null)
            {
                children.Add(m_notificationsCount);
            }

            if (m_latePublishRequestCount != null)
            {
                children.Add(m_latePublishRequestCount);
            }

            if (m_currentKeepAliveCount != null)
            {
                children.Add(m_currentKeepAliveCount);
            }

            if (m_currentLifetimeCount != null)
            {
                children.Add(m_currentLifetimeCount);
            }

            if (m_unacknowledgedMessageCount != null)
            {
                children.Add(m_unacknowledgedMessageCount);
            }

            if (m_discardedMessageCount != null)
            {
                children.Add(m_discardedMessageCount);
            }

            if (m_monitoredItemCount != null)
            {
                children.Add(m_monitoredItemCount);
            }

            if (m_disabledMonitoredItemCount != null)
            {
                children.Add(m_disabledMonitoredItemCount);
            }

            if (m_monitoringQueueOverflowCount != null)
            {
                children.Add(m_monitoringQueueOverflowCount);
            }

            if (m_nextSequenceNumber != null)
            {
                children.Add(m_nextSequenceNumber);
            }

            if (m_eventQueueOverflowCount != null)
            {
                children.Add(m_eventQueueOverflowCount);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionId:
                {
                    if (createOrReplace)
                    {
                        if (SessionId == null)
                        {
                            if (replacement == null)
                            {
                                SessionId = new BaseDataVariableState<NodeId>(this);
                            }
                            else
                            {
                                SessionId = (BaseDataVariableState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = SessionId;
                    break;
                }

                case Opc.Ua.BrowseNames.SubscriptionId:
                {
                    if (createOrReplace)
                    {
                        if (SubscriptionId == null)
                        {
                            if (replacement == null)
                            {
                                SubscriptionId = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                SubscriptionId = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = SubscriptionId;
                    break;
                }

                case Opc.Ua.BrowseNames.Priority:
                {
                    if (createOrReplace)
                    {
                        if (Priority == null)
                        {
                            if (replacement == null)
                            {
                                Priority = new BaseDataVariableState<byte>(this);
                            }
                            else
                            {
                                Priority = (BaseDataVariableState<byte>)replacement;
                            }
                        }
                    }

                    instance = Priority;
                    break;
                }

                case Opc.Ua.BrowseNames.PublishingInterval:
                {
                    if (createOrReplace)
                    {
                        if (PublishingInterval == null)
                        {
                            if (replacement == null)
                            {
                                PublishingInterval = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                PublishingInterval = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = PublishingInterval;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxKeepAliveCount:
                {
                    if (createOrReplace)
                    {
                        if (MaxKeepAliveCount == null)
                        {
                            if (replacement == null)
                            {
                                MaxKeepAliveCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MaxKeepAliveCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxKeepAliveCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxLifetimeCount:
                {
                    if (createOrReplace)
                    {
                        if (MaxLifetimeCount == null)
                        {
                            if (replacement == null)
                            {
                                MaxLifetimeCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MaxLifetimeCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxLifetimeCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxNotificationsPerPublish:
                {
                    if (createOrReplace)
                    {
                        if (MaxNotificationsPerPublish == null)
                        {
                            if (replacement == null)
                            {
                                MaxNotificationsPerPublish = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MaxNotificationsPerPublish = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxNotificationsPerPublish;
                    break;
                }

                case Opc.Ua.BrowseNames.PublishingEnabled:
                {
                    if (createOrReplace)
                    {
                        if (PublishingEnabled == null)
                        {
                            if (replacement == null)
                            {
                                PublishingEnabled = new BaseDataVariableState<bool>(this);
                            }
                            else
                            {
                                PublishingEnabled = (BaseDataVariableState<bool>)replacement;
                            }
                        }
                    }

                    instance = PublishingEnabled;
                    break;
                }

                case Opc.Ua.BrowseNames.ModifyCount:
                {
                    if (createOrReplace)
                    {
                        if (ModifyCount == null)
                        {
                            if (replacement == null)
                            {
                                ModifyCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                ModifyCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = ModifyCount;
                    break;
                }

                case Opc.Ua.BrowseNames.EnableCount:
                {
                    if (createOrReplace)
                    {
                        if (EnableCount == null)
                        {
                            if (replacement == null)
                            {
                                EnableCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                EnableCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = EnableCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DisableCount:
                {
                    if (createOrReplace)
                    {
                        if (DisableCount == null)
                        {
                            if (replacement == null)
                            {
                                DisableCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                DisableCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = DisableCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RepublishRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (RepublishRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                RepublishRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                RepublishRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = RepublishRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RepublishMessageRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (RepublishMessageRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                RepublishMessageRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                RepublishMessageRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = RepublishMessageRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RepublishMessageCount:
                {
                    if (createOrReplace)
                    {
                        if (RepublishMessageCount == null)
                        {
                            if (replacement == null)
                            {
                                RepublishMessageCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                RepublishMessageCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = RepublishMessageCount;
                    break;
                }

                case Opc.Ua.BrowseNames.TransferRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (TransferRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                TransferRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                TransferRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = TransferRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.TransferredToAltClientCount:
                {
                    if (createOrReplace)
                    {
                        if (TransferredToAltClientCount == null)
                        {
                            if (replacement == null)
                            {
                                TransferredToAltClientCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                TransferredToAltClientCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = TransferredToAltClientCount;
                    break;
                }

                case Opc.Ua.BrowseNames.TransferredToSameClientCount:
                {
                    if (createOrReplace)
                    {
                        if (TransferredToSameClientCount == null)
                        {
                            if (replacement == null)
                            {
                                TransferredToSameClientCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                TransferredToSameClientCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = TransferredToSameClientCount;
                    break;
                }

                case Opc.Ua.BrowseNames.PublishRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (PublishRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                PublishRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                PublishRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = PublishRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DataChangeNotificationsCount:
                {
                    if (createOrReplace)
                    {
                        if (DataChangeNotificationsCount == null)
                        {
                            if (replacement == null)
                            {
                                DataChangeNotificationsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                DataChangeNotificationsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = DataChangeNotificationsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.EventNotificationsCount:
                {
                    if (createOrReplace)
                    {
                        if (EventNotificationsCount == null)
                        {
                            if (replacement == null)
                            {
                                EventNotificationsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                EventNotificationsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = EventNotificationsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.NotificationsCount:
                {
                    if (createOrReplace)
                    {
                        if (NotificationsCount == null)
                        {
                            if (replacement == null)
                            {
                                NotificationsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                NotificationsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = NotificationsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.LatePublishRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (LatePublishRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                LatePublishRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                LatePublishRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = LatePublishRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentKeepAliveCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentKeepAliveCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentKeepAliveCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentKeepAliveCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentKeepAliveCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentLifetimeCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentLifetimeCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentLifetimeCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentLifetimeCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentLifetimeCount;
                    break;
                }

                case Opc.Ua.BrowseNames.UnacknowledgedMessageCount:
                {
                    if (createOrReplace)
                    {
                        if (UnacknowledgedMessageCount == null)
                        {
                            if (replacement == null)
                            {
                                UnacknowledgedMessageCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                UnacknowledgedMessageCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = UnacknowledgedMessageCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DiscardedMessageCount:
                {
                    if (createOrReplace)
                    {
                        if (DiscardedMessageCount == null)
                        {
                            if (replacement == null)
                            {
                                DiscardedMessageCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                DiscardedMessageCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = DiscardedMessageCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MonitoredItemCount:
                {
                    if (createOrReplace)
                    {
                        if (MonitoredItemCount == null)
                        {
                            if (replacement == null)
                            {
                                MonitoredItemCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MonitoredItemCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MonitoredItemCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DisabledMonitoredItemCount:
                {
                    if (createOrReplace)
                    {
                        if (DisabledMonitoredItemCount == null)
                        {
                            if (replacement == null)
                            {
                                DisabledMonitoredItemCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                DisabledMonitoredItemCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = DisabledMonitoredItemCount;
                    break;
                }

                case Opc.Ua.BrowseNames.MonitoringQueueOverflowCount:
                {
                    if (createOrReplace)
                    {
                        if (MonitoringQueueOverflowCount == null)
                        {
                            if (replacement == null)
                            {
                                MonitoringQueueOverflowCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MonitoringQueueOverflowCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MonitoringQueueOverflowCount;
                    break;
                }

                case Opc.Ua.BrowseNames.NextSequenceNumber:
                {
                    if (createOrReplace)
                    {
                        if (NextSequenceNumber == null)
                        {
                            if (replacement == null)
                            {
                                NextSequenceNumber = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                NextSequenceNumber = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = NextSequenceNumber;
                    break;
                }

                case Opc.Ua.BrowseNames.EventQueueOverflowCount:
                {
                    if (createOrReplace)
                    {
                        if (EventQueueOverflowCount == null)
                        {
                            if (replacement == null)
                            {
                                EventQueueOverflowCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                EventQueueOverflowCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = EventQueueOverflowCount;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<NodeId> m_sessionId;
        private BaseDataVariableState<uint> m_subscriptionId;
        private BaseDataVariableState<byte> m_priority;
        private BaseDataVariableState<double> m_publishingInterval;
        private BaseDataVariableState<uint> m_maxKeepAliveCount;
        private BaseDataVariableState<uint> m_maxLifetimeCount;
        private BaseDataVariableState<uint> m_maxNotificationsPerPublish;
        private BaseDataVariableState<bool> m_publishingEnabled;
        private BaseDataVariableState<uint> m_modifyCount;
        private BaseDataVariableState<uint> m_enableCount;
        private BaseDataVariableState<uint> m_disableCount;
        private BaseDataVariableState<uint> m_republishRequestCount;
        private BaseDataVariableState<uint> m_republishMessageRequestCount;
        private BaseDataVariableState<uint> m_republishMessageCount;
        private BaseDataVariableState<uint> m_transferRequestCount;
        private BaseDataVariableState<uint> m_transferredToAltClientCount;
        private BaseDataVariableState<uint> m_transferredToSameClientCount;
        private BaseDataVariableState<uint> m_publishRequestCount;
        private BaseDataVariableState<uint> m_dataChangeNotificationsCount;
        private BaseDataVariableState<uint> m_eventNotificationsCount;
        private BaseDataVariableState<uint> m_notificationsCount;
        private BaseDataVariableState<uint> m_latePublishRequestCount;
        private BaseDataVariableState<uint> m_currentKeepAliveCount;
        private BaseDataVariableState<uint> m_currentLifetimeCount;
        private BaseDataVariableState<uint> m_unacknowledgedMessageCount;
        private BaseDataVariableState<uint> m_discardedMessageCount;
        private BaseDataVariableState<uint> m_monitoredItemCount;
        private BaseDataVariableState<uint> m_disabledMonitoredItemCount;
        private BaseDataVariableState<uint> m_monitoringQueueOverflowCount;
        private BaseDataVariableState<uint> m_nextSequenceNumber;
        private BaseDataVariableState<uint> m_eventQueueOverflowCount;
        #endregion
    }

    #region SubscriptionDiagnosticsValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class SubscriptionDiagnosticsValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public SubscriptionDiagnosticsValue(SubscriptionDiagnosticsState variable, SubscriptionDiagnosticsDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new SubscriptionDiagnosticsDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public SubscriptionDiagnosticsState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public SubscriptionDiagnosticsDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(SubscriptionDiagnosticsState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.SessionId;
                instance.OnReadValue = OnRead_SessionId;
                instance.OnSimpleWriteValue = OnWrite_SessionId;
                updateList.Add(instance);
                instance = m_variable.SubscriptionId;
                instance.OnReadValue = OnRead_SubscriptionId;
                instance.OnSimpleWriteValue = OnWrite_SubscriptionId;
                updateList.Add(instance);
                instance = m_variable.Priority;
                instance.OnReadValue = OnRead_Priority;
                instance.OnSimpleWriteValue = OnWrite_Priority;
                updateList.Add(instance);
                instance = m_variable.PublishingInterval;
                instance.OnReadValue = OnRead_PublishingInterval;
                instance.OnSimpleWriteValue = OnWrite_PublishingInterval;
                updateList.Add(instance);
                instance = m_variable.MaxKeepAliveCount;
                instance.OnReadValue = OnRead_MaxKeepAliveCount;
                instance.OnSimpleWriteValue = OnWrite_MaxKeepAliveCount;
                updateList.Add(instance);
                instance = m_variable.MaxLifetimeCount;
                instance.OnReadValue = OnRead_MaxLifetimeCount;
                instance.OnSimpleWriteValue = OnWrite_MaxLifetimeCount;
                updateList.Add(instance);
                instance = m_variable.MaxNotificationsPerPublish;
                instance.OnReadValue = OnRead_MaxNotificationsPerPublish;
                instance.OnSimpleWriteValue = OnWrite_MaxNotificationsPerPublish;
                updateList.Add(instance);
                instance = m_variable.PublishingEnabled;
                instance.OnReadValue = OnRead_PublishingEnabled;
                instance.OnSimpleWriteValue = OnWrite_PublishingEnabled;
                updateList.Add(instance);
                instance = m_variable.ModifyCount;
                instance.OnReadValue = OnRead_ModifyCount;
                instance.OnSimpleWriteValue = OnWrite_ModifyCount;
                updateList.Add(instance);
                instance = m_variable.EnableCount;
                instance.OnReadValue = OnRead_EnableCount;
                instance.OnSimpleWriteValue = OnWrite_EnableCount;
                updateList.Add(instance);
                instance = m_variable.DisableCount;
                instance.OnReadValue = OnRead_DisableCount;
                instance.OnSimpleWriteValue = OnWrite_DisableCount;
                updateList.Add(instance);
                instance = m_variable.RepublishRequestCount;
                instance.OnReadValue = OnRead_RepublishRequestCount;
                instance.OnSimpleWriteValue = OnWrite_RepublishRequestCount;
                updateList.Add(instance);
                instance = m_variable.RepublishMessageRequestCount;
                instance.OnReadValue = OnRead_RepublishMessageRequestCount;
                instance.OnSimpleWriteValue = OnWrite_RepublishMessageRequestCount;
                updateList.Add(instance);
                instance = m_variable.RepublishMessageCount;
                instance.OnReadValue = OnRead_RepublishMessageCount;
                instance.OnSimpleWriteValue = OnWrite_RepublishMessageCount;
                updateList.Add(instance);
                instance = m_variable.TransferRequestCount;
                instance.OnReadValue = OnRead_TransferRequestCount;
                instance.OnSimpleWriteValue = OnWrite_TransferRequestCount;
                updateList.Add(instance);
                instance = m_variable.TransferredToAltClientCount;
                instance.OnReadValue = OnRead_TransferredToAltClientCount;
                instance.OnSimpleWriteValue = OnWrite_TransferredToAltClientCount;
                updateList.Add(instance);
                instance = m_variable.TransferredToSameClientCount;
                instance.OnReadValue = OnRead_TransferredToSameClientCount;
                instance.OnSimpleWriteValue = OnWrite_TransferredToSameClientCount;
                updateList.Add(instance);
                instance = m_variable.PublishRequestCount;
                instance.OnReadValue = OnRead_PublishRequestCount;
                instance.OnSimpleWriteValue = OnWrite_PublishRequestCount;
                updateList.Add(instance);
                instance = m_variable.DataChangeNotificationsCount;
                instance.OnReadValue = OnRead_DataChangeNotificationsCount;
                instance.OnSimpleWriteValue = OnWrite_DataChangeNotificationsCount;
                updateList.Add(instance);
                instance = m_variable.EventNotificationsCount;
                instance.OnReadValue = OnRead_EventNotificationsCount;
                instance.OnSimpleWriteValue = OnWrite_EventNotificationsCount;
                updateList.Add(instance);
                instance = m_variable.NotificationsCount;
                instance.OnReadValue = OnRead_NotificationsCount;
                instance.OnSimpleWriteValue = OnWrite_NotificationsCount;
                updateList.Add(instance);
                instance = m_variable.LatePublishRequestCount;
                instance.OnReadValue = OnRead_LatePublishRequestCount;
                instance.OnSimpleWriteValue = OnWrite_LatePublishRequestCount;
                updateList.Add(instance);
                instance = m_variable.CurrentKeepAliveCount;
                instance.OnReadValue = OnRead_CurrentKeepAliveCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentKeepAliveCount;
                updateList.Add(instance);
                instance = m_variable.CurrentLifetimeCount;
                instance.OnReadValue = OnRead_CurrentLifetimeCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentLifetimeCount;
                updateList.Add(instance);
                instance = m_variable.UnacknowledgedMessageCount;
                instance.OnReadValue = OnRead_UnacknowledgedMessageCount;
                instance.OnSimpleWriteValue = OnWrite_UnacknowledgedMessageCount;
                updateList.Add(instance);
                instance = m_variable.DiscardedMessageCount;
                instance.OnReadValue = OnRead_DiscardedMessageCount;
                instance.OnSimpleWriteValue = OnWrite_DiscardedMessageCount;
                updateList.Add(instance);
                instance = m_variable.MonitoredItemCount;
                instance.OnReadValue = OnRead_MonitoredItemCount;
                instance.OnSimpleWriteValue = OnWrite_MonitoredItemCount;
                updateList.Add(instance);
                instance = m_variable.DisabledMonitoredItemCount;
                instance.OnReadValue = OnRead_DisabledMonitoredItemCount;
                instance.OnSimpleWriteValue = OnWrite_DisabledMonitoredItemCount;
                updateList.Add(instance);
                instance = m_variable.MonitoringQueueOverflowCount;
                instance.OnReadValue = OnRead_MonitoringQueueOverflowCount;
                instance.OnSimpleWriteValue = OnWrite_MonitoringQueueOverflowCount;
                updateList.Add(instance);
                instance = m_variable.NextSequenceNumber;
                instance.OnReadValue = OnRead_NextSequenceNumber;
                instance.OnSimpleWriteValue = OnWrite_NextSequenceNumber;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (SubscriptionDiagnosticsDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region SessionId Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionId(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionId;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionId(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionId = (NodeId)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SubscriptionId Access Methods
        /// <remarks />
        private ServiceResult OnRead_SubscriptionId(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SubscriptionId;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SubscriptionId(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SubscriptionId = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Priority Access Methods
        /// <remarks />
        private ServiceResult OnRead_Priority(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Priority;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Priority(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Priority = (byte)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region PublishingInterval Access Methods
        /// <remarks />
        private ServiceResult OnRead_PublishingInterval(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.PublishingInterval;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_PublishingInterval(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.PublishingInterval = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MaxKeepAliveCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_MaxKeepAliveCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MaxKeepAliveCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MaxKeepAliveCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MaxKeepAliveCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MaxLifetimeCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_MaxLifetimeCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MaxLifetimeCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MaxLifetimeCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MaxLifetimeCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MaxNotificationsPerPublish Access Methods
        /// <remarks />
        private ServiceResult OnRead_MaxNotificationsPerPublish(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MaxNotificationsPerPublish;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MaxNotificationsPerPublish(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MaxNotificationsPerPublish = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region PublishingEnabled Access Methods
        /// <remarks />
        private ServiceResult OnRead_PublishingEnabled(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.PublishingEnabled;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_PublishingEnabled(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.PublishingEnabled = (bool)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ModifyCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_ModifyCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ModifyCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ModifyCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ModifyCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region EnableCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_EnableCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.EnableCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_EnableCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.EnableCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DisableCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DisableCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DisableCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DisableCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DisableCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RepublishRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RepublishRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RepublishRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RepublishRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RepublishRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RepublishMessageRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RepublishMessageRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RepublishMessageRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RepublishMessageRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RepublishMessageRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RepublishMessageCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RepublishMessageCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RepublishMessageCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RepublishMessageCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RepublishMessageCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TransferRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TransferRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TransferRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TransferRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TransferRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TransferredToAltClientCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TransferredToAltClientCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TransferredToAltClientCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TransferredToAltClientCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TransferredToAltClientCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TransferredToSameClientCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TransferredToSameClientCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TransferredToSameClientCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TransferredToSameClientCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TransferredToSameClientCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region PublishRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_PublishRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.PublishRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_PublishRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.PublishRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DataChangeNotificationsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DataChangeNotificationsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DataChangeNotificationsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DataChangeNotificationsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DataChangeNotificationsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region EventNotificationsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_EventNotificationsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.EventNotificationsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_EventNotificationsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.EventNotificationsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region NotificationsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_NotificationsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.NotificationsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_NotificationsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.NotificationsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region LatePublishRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_LatePublishRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.LatePublishRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_LatePublishRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.LatePublishRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentKeepAliveCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentKeepAliveCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentKeepAliveCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentKeepAliveCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentKeepAliveCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentLifetimeCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentLifetimeCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentLifetimeCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentLifetimeCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentLifetimeCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region UnacknowledgedMessageCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_UnacknowledgedMessageCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.UnacknowledgedMessageCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_UnacknowledgedMessageCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.UnacknowledgedMessageCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DiscardedMessageCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DiscardedMessageCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DiscardedMessageCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DiscardedMessageCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DiscardedMessageCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MonitoredItemCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_MonitoredItemCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MonitoredItemCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MonitoredItemCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MonitoredItemCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DisabledMonitoredItemCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DisabledMonitoredItemCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DisabledMonitoredItemCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DisabledMonitoredItemCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DisabledMonitoredItemCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MonitoringQueueOverflowCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_MonitoringQueueOverflowCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MonitoringQueueOverflowCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MonitoringQueueOverflowCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MonitoringQueueOverflowCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region NextSequenceNumber Access Methods
        /// <remarks />
        private ServiceResult OnRead_NextSequenceNumber(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.NextSequenceNumber;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_NextSequenceNumber(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.NextSequenceNumber = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private SubscriptionDiagnosticsDataType m_value;
        private SubscriptionDiagnosticsState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region SessionDiagnosticsArrayState Class
    #if (!OPCUA_EXCLUDE_SessionDiagnosticsArrayState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionDiagnosticsArrayState : BaseDataVariableState<SessionDiagnosticsDataType[]>
    {
        #region Constructors
        /// <remarks />
        public SessionDiagnosticsArrayState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SessionDiagnosticsArrayType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SessionDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.OneDimension;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8XYIkCAgAAAAAAIwAAAFNlc3Npb25EaWFnbm9zdGljc0FycmF5VHlwZUluc3RhbmNlAQCU" +
           "CAEAlAiUCAAAAQBhAwEAAAABAAAAAAAAAAEB/////wEAAAAVYIkKAgAAAAAAEgAAAFNlc3Npb25EaWFn" +
           "bm9zdGljcwEAEDIALwEAlQgQMgAAAQBhA/////8BAf////8rAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9u" +
           "SWQBABEyAC8APxEyAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABTZXNzaW9uTmFtZQEAEjIA" +
           "LwA/EjIAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAENsaWVudERlc2NyaXB0aW9uAQATMgAv" +
           "AD8TMgAAAQA0Af////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABTZXJ2ZXJVcmkBABQyAC8APxQyAAAA" +
           "DP////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABFbmRwb2ludFVybAEAFTIALwA/FTIAAAAM/////wEB" +
           "/////wAAAAAXYIkKAgAAAAAACQAAAExvY2FsZUlkcwEAFjIALwA/FjIAAAEAJwEBAAAAAQAAAAAAAAAB" +
           "Af////8AAAAAFWCJCgIAAAAAABQAAABBY3R1YWxTZXNzaW9uVGltZW91dAEAFzIALwA/FzIAAAEAIgH/" +
           "////AQH/////AAAAABVgiQoCAAAAAAAWAAAATWF4UmVzcG9uc2VNZXNzYWdlU2l6ZQEAGDIALwA/GDIA" +
           "AAAH/////wEB/////wAAAAAVYIkKAgAAAAAAFAAAAENsaWVudENvbm5lY3Rpb25UaW1lAQAZMgAvAD8Z" +
           "MgAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAABUAAABDbGllbnRMYXN0Q29udGFjdFRpbWUBABoy" +
           "AC8APxoyAAABACYB/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAEN1cnJlbnRTdWJzY3JpcHRpb25z" +
           "Q291bnQBABsyAC8APxsyAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABDdXJyZW50TW9uaXRv" +
           "cmVkSXRlbXNDb3VudAEAHDIALwA/HDIAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAHQAAAEN1cnJl" +
           "bnRQdWJsaXNoUmVxdWVzdHNJblF1ZXVlAQAdMgAvAD8dMgAAAAf/////AQH/////AAAAABVgiQoCAAAA" +
           "AAARAAAAVG90YWxSZXF1ZXN0Q291bnQBAB4yAC8APx4yAAABAGcD/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAGAAAAFVuYXV0aG9yaXplZFJlcXVlc3RDb3VudAEAHzIALwA/HzIAAAAH/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAACQAAAFJlYWRDb3VudAEAIDIALwA/IDIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAQAAAASGlzdG9yeVJlYWRDb3VudAEAITIALwA/ITIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAKAAAAV3JpdGVDb3VudAEAIjIALwA/IjIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAASAAAA" +
           "SGlzdG9yeVVwZGF0ZUNvdW50AQAjMgAvAD8jMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAkA" +
           "AABDYWxsQ291bnQBACQyAC8APyQyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAENyZWF0" +
           "ZU1vbml0b3JlZEl0ZW1zQ291bnQBACUyAC8APyUyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "GQAAAE1vZGlmeU1vbml0b3JlZEl0ZW1zQ291bnQBACYyAC8APyYyAAABAGcD/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAAFgAAAFNldE1vbml0b3JpbmdNb2RlQ291bnQBACcyAC8APycyAAABAGcD/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAAEgAAAFNldFRyaWdnZXJpbmdDb3VudAEAKDIALwA/KDIAAAEAZwP/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAZAAAARGVsZXRlTW9uaXRvcmVkSXRlbXNDb3VudAEAKTIALwA/KTIAAAEA" +
           "ZwP/////AQH/////AAAAABVgiQoCAAAAAAAXAAAAQ3JlYXRlU3Vic2NyaXB0aW9uQ291bnQBACoyAC8A" +
           "PyoyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAFwAAAE1vZGlmeVN1YnNjcmlwdGlvbkNvdW50" +
           "AQArMgAvAD8rMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABYAAABTZXRQdWJsaXNoaW5nTW9k" +
           "ZUNvdW50AQAsMgAvAD8sMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABQdWJsaXNoQ291" +
           "bnQBAC0yAC8APy0yAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADgAAAFJlcHVibGlzaENvdW50" +
           "AQAuMgAvAD8uMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABUcmFuc2ZlclN1YnNjcmlw" +
           "dGlvbnNDb3VudAEALzIALwA/LzIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAYAAAARGVsZXRl" +
           "U3Vic2NyaXB0aW9uc0NvdW50AQAwMgAvAD8wMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAA0A" +
           "AABBZGROb2Rlc0NvdW50AQAxMgAvAD8xMgAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABIAAABB" +
           "ZGRSZWZlcmVuY2VzQ291bnQBADIyAC8APzIyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEAAA" +
           "AERlbGV0ZU5vZGVzQ291bnQBADMyAC8APzMyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAFQAA" +
           "AERlbGV0ZVJlZmVyZW5jZXNDb3VudAEANDIALwA/NDIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAA" +
           "AAALAAAAQnJvd3NlQ291bnQBADUyAC8APzUyAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADwAA" +
           "AEJyb3dzZU5leHRDb3VudAEANjIALwA/NjIAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAiAAAA" +
           "VHJhbnNsYXRlQnJvd3NlUGF0aHNUb05vZGVJZHNDb3VudAEANzIALwA/NzIAAAEAZwP/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAPAAAAUXVlcnlGaXJzdENvdW50AQA4MgAvAD84MgAAAQBnA/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAA4AAABRdWVyeU5leHRDb3VudAEAOTIALwA/OTIAAAEAZwP/////AQH/////AAAA" +
           "ABVgiQoCAAAAAAASAAAAUmVnaXN0ZXJOb2Rlc0NvdW50AQA6MgAvAD86MgAAAQBnA/////8BAf////8A" +
           "AAAAFWCJCgIAAAAAABQAAABVbnJlZ2lzdGVyTm9kZXNDb3VudAEAOzIALwA/OzIAAAEAZwP/////AQH/" +
           "////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region SessionDiagnosticsVariableState Class
    #if (!OPCUA_EXCLUDE_SessionDiagnosticsVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionDiagnosticsVariableState : BaseDataVariableState<SessionDiagnosticsDataType>
    {
        #region Constructors
        /// <remarks />
        public SessionDiagnosticsVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SessionDiagnosticsVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SessionDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJgAAAFNlc3Npb25EaWFnbm9zdGljc1ZhcmlhYmxlVHlwZUluc3RhbmNl" +
           "AQCVCAEAlQiVCAAAAQBhA/////8BAf////8rAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9uSWQBAJYIAC8A" +
           "P5YIAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAAAsAAABTZXNzaW9uTmFtZQEAlwgALwA/lwgAAAAM" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAENsaWVudERlc2NyaXB0aW9uAQCYCAAvAD+YCAAAAQA0" +
           "Af////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABTZXJ2ZXJVcmkBAJkIAC8AP5kIAAAADP////8BAf//" +
           "//8AAAAAFWCJCgIAAAAAAAsAAABFbmRwb2ludFVybAEAmggALwA/mggAAAAM/////wEB/////wAAAAAX" +
           "YIkKAgAAAAAACQAAAExvY2FsZUlkcwEAmwgALwA/mwgAAAEAJwEBAAAAAQAAAAAAAAABAf////8AAAAA" +
           "FWCJCgIAAAAAABQAAABBY3R1YWxTZXNzaW9uVGltZW91dAEAnAgALwA/nAgAAAEAIgH/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAWAAAATWF4UmVzcG9uc2VNZXNzYWdlU2l6ZQEA6gsALwA/6gsAAAAH/////wEB" +
           "/////wAAAAAVYIkKAgAAAAAAFAAAAENsaWVudENvbm5lY3Rpb25UaW1lAQCdCAAvAD+dCAAAAQAmAf//" +
           "//8BAf////8AAAAAFWCJCgIAAAAAABUAAABDbGllbnRMYXN0Q29udGFjdFRpbWUBAJ4IAC8AP54IAAAB" +
           "ACYB/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAEN1cnJlbnRTdWJzY3JpcHRpb25zQ291bnQBAJ8I" +
           "AC8AP58IAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABDdXJyZW50TW9uaXRvcmVkSXRlbXND" +
           "b3VudAEAoAgALwA/oAgAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAAHQAAAEN1cnJlbnRQdWJsaXNo" +
           "UmVxdWVzdHNJblF1ZXVlAQChCAAvAD+hCAAAAAf/////AQH/////AAAAABVgiQoCAAAAAAARAAAAVG90" +
           "YWxSZXF1ZXN0Q291bnQBAMQiAC8AP8QiAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAGAAAAFVu" +
           "YXV0aG9yaXplZFJlcXVlc3RDb3VudAEAdC4ALwA/dC4AAAAH/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "CQAAAFJlYWRDb3VudAEAqQgALwA/qQgAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAQAAAASGlz" +
           "dG9yeVJlYWRDb3VudAEAqggALwA/qggAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAKAAAAV3Jp" +
           "dGVDb3VudAEAqwgALwA/qwgAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAASAAAASGlzdG9yeVVw" +
           "ZGF0ZUNvdW50AQCsCAAvAD+sCAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAkAAABDYWxsQ291" +
           "bnQBAK0IAC8AP60IAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAENyZWF0ZU1vbml0b3Jl" +
           "ZEl0ZW1zQ291bnQBAK4IAC8AP64IAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAGQAAAE1vZGlm" +
           "eU1vbml0b3JlZEl0ZW1zQ291bnQBAK8IAC8AP68IAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "FgAAAFNldE1vbml0b3JpbmdNb2RlQ291bnQBALAIAC8AP7AIAAABAGcD/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAAEgAAAFNldFRyaWdnZXJpbmdDb3VudAEAsQgALwA/sQgAAAEAZwP/////AQH/////AAAAABVg" +
           "iQoCAAAAAAAZAAAARGVsZXRlTW9uaXRvcmVkSXRlbXNDb3VudAEAsggALwA/sggAAAEAZwP/////AQH/" +
           "////AAAAABVgiQoCAAAAAAAXAAAAQ3JlYXRlU3Vic2NyaXB0aW9uQ291bnQBALMIAC8AP7MIAAABAGcD" +
           "/////wEB/////wAAAAAVYIkKAgAAAAAAFwAAAE1vZGlmeVN1YnNjcmlwdGlvbkNvdW50AQC0CAAvAD+0" +
           "CAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABYAAABTZXRQdWJsaXNoaW5nTW9kZUNvdW50AQC1" +
           "CAAvAD+1CAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABQdWJsaXNoQ291bnQBALYIAC8A" +
           "P7YIAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADgAAAFJlcHVibGlzaENvdW50AQC3CAAvAD+3" +
           "CAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABoAAABUcmFuc2ZlclN1YnNjcmlwdGlvbnNDb3Vu" +
           "dAEAuAgALwA/uAgAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAYAAAARGVsZXRlU3Vic2NyaXB0" +
           "aW9uc0NvdW50AQC5CAAvAD+5CAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAAA0AAABBZGROb2Rl" +
           "c0NvdW50AQC6CAAvAD+6CAAAAQBnA/////8BAf////8AAAAAFWCJCgIAAAAAABIAAABBZGRSZWZlcmVu" +
           "Y2VzQ291bnQBALsIAC8AP7sIAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAEAAAAERlbGV0ZU5v" +
           "ZGVzQ291bnQBALwIAC8AP7wIAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAAFQAAAERlbGV0ZVJl" +
           "ZmVyZW5jZXNDb3VudAEAvQgALwA/vQgAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAALAAAAQnJv" +
           "d3NlQ291bnQBAL4IAC8AP74IAAABAGcD/////wEB/////wAAAAAVYIkKAgAAAAAADwAAAEJyb3dzZU5l" +
           "eHRDb3VudAEAvwgALwA/vwgAAAEAZwP/////AQH/////AAAAABVgiQoCAAAAAAAiAAAAVHJhbnNsYXRl" +
           "QnJvd3NlUGF0aHNUb05vZGVJZHNDb3VudAEAwAgALwA/wAgAAAEAZwP/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAPAAAAUXVlcnlGaXJzdENvdW50AQDBCAAvAD/BCAAAAQBnA/////8BAf////8AAAAAFWCJCgIA" +
           "AAAAAA4AAABRdWVyeU5leHRDb3VudAEAwggALwA/wggAAAEAZwP/////AQH/////AAAAABVgiQoCAAAA" +
           "AAASAAAAUmVnaXN0ZXJOb2Rlc0NvdW50AQCqCgAvAD+qCgAAAQBnA/////8BAf////8AAAAAFWCJCgIA" +
           "AAAAABQAAABVbnJlZ2lzdGVyTm9kZXNDb3VudAEAqwoALwA/qwoAAAEAZwP/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<NodeId> SessionId
        {
            get
            {
                return m_sessionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionId = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> SessionName
        {
            get
            {
                return m_sessionName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionName = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ApplicationDescription> ClientDescription
        {
            get
            {
                return m_clientDescription;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientDescription, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientDescription = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> ServerUri
        {
            get
            {
                return m_serverUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_serverUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_serverUri = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> EndpointUrl
        {
            get
            {
                return m_endpointUrl;
            }

            set
            {
                if (!Object.ReferenceEquals(m_endpointUrl, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_endpointUrl = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string[]> LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                if (!Object.ReferenceEquals(m_localeIds, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_localeIds = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> ActualSessionTimeout
        {
            get
            {
                return m_actualSessionTimeout;
            }

            set
            {
                if (!Object.ReferenceEquals(m_actualSessionTimeout, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_actualSessionTimeout = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> MaxResponseMessageSize
        {
            get
            {
                return m_maxResponseMessageSize;
            }

            set
            {
                if (!Object.ReferenceEquals(m_maxResponseMessageSize, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_maxResponseMessageSize = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<DateTime> ClientConnectionTime
        {
            get
            {
                return m_clientConnectionTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientConnectionTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientConnectionTime = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<DateTime> ClientLastContactTime
        {
            get
            {
                return m_clientLastContactTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientLastContactTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientLastContactTime = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentSubscriptionsCount
        {
            get
            {
                return m_currentSubscriptionsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentSubscriptionsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentSubscriptionsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentMonitoredItemsCount
        {
            get
            {
                return m_currentMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> CurrentPublishRequestsInQueue
        {
            get
            {
                return m_currentPublishRequestsInQueue;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentPublishRequestsInQueue, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentPublishRequestsInQueue = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> TotalRequestCount
        {
            get
            {
                return m_totalRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_totalRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_totalRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> UnauthorizedRequestCount
        {
            get
            {
                return m_unauthorizedRequestCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_unauthorizedRequestCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_unauthorizedRequestCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> ReadCount
        {
            get
            {
                return m_readCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_readCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_readCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> HistoryReadCount
        {
            get
            {
                return m_historyReadCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_historyReadCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_historyReadCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> WriteCount
        {
            get
            {
                return m_writeCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_writeCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_writeCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> HistoryUpdateCount
        {
            get
            {
                return m_historyUpdateCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_historyUpdateCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_historyUpdateCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> CallCount
        {
            get
            {
                return m_callCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_callCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_callCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> CreateMonitoredItemsCount
        {
            get
            {
                return m_createMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_createMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_createMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> ModifyMonitoredItemsCount
        {
            get
            {
                return m_modifyMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_modifyMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_modifyMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> SetMonitoringModeCount
        {
            get
            {
                return m_setMonitoringModeCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_setMonitoringModeCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_setMonitoringModeCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> SetTriggeringCount
        {
            get
            {
                return m_setTriggeringCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_setTriggeringCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_setTriggeringCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> DeleteMonitoredItemsCount
        {
            get
            {
                return m_deleteMonitoredItemsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_deleteMonitoredItemsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_deleteMonitoredItemsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> CreateSubscriptionCount
        {
            get
            {
                return m_createSubscriptionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_createSubscriptionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_createSubscriptionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> ModifySubscriptionCount
        {
            get
            {
                return m_modifySubscriptionCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_modifySubscriptionCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_modifySubscriptionCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> SetPublishingModeCount
        {
            get
            {
                return m_setPublishingModeCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_setPublishingModeCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_setPublishingModeCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> PublishCount
        {
            get
            {
                return m_publishCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_publishCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_publishCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> RepublishCount
        {
            get
            {
                return m_republishCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_republishCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_republishCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> TransferSubscriptionsCount
        {
            get
            {
                return m_transferSubscriptionsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transferSubscriptionsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transferSubscriptionsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> DeleteSubscriptionsCount
        {
            get
            {
                return m_deleteSubscriptionsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_deleteSubscriptionsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_deleteSubscriptionsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> AddNodesCount
        {
            get
            {
                return m_addNodesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_addNodesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_addNodesCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> AddReferencesCount
        {
            get
            {
                return m_addReferencesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_addReferencesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_addReferencesCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> DeleteNodesCount
        {
            get
            {
                return m_deleteNodesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_deleteNodesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_deleteNodesCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> DeleteReferencesCount
        {
            get
            {
                return m_deleteReferencesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_deleteReferencesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_deleteReferencesCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> BrowseCount
        {
            get
            {
                return m_browseCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_browseCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_browseCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> BrowseNextCount
        {
            get
            {
                return m_browseNextCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_browseNextCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_browseNextCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> TranslateBrowsePathsToNodeIdsCount
        {
            get
            {
                return m_translateBrowsePathsToNodeIdsCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_translateBrowsePathsToNodeIdsCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_translateBrowsePathsToNodeIdsCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> QueryFirstCount
        {
            get
            {
                return m_queryFirstCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_queryFirstCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_queryFirstCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> QueryNextCount
        {
            get
            {
                return m_queryNextCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_queryNextCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_queryNextCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> RegisterNodesCount
        {
            get
            {
                return m_registerNodesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_registerNodesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_registerNodesCount = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<ServiceCounterDataType> UnregisterNodesCount
        {
            get
            {
                return m_unregisterNodesCount;
            }

            set
            {
                if (!Object.ReferenceEquals(m_unregisterNodesCount, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_unregisterNodesCount = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionId != null)
            {
                children.Add(m_sessionId);
            }

            if (m_sessionName != null)
            {
                children.Add(m_sessionName);
            }

            if (m_clientDescription != null)
            {
                children.Add(m_clientDescription);
            }

            if (m_serverUri != null)
            {
                children.Add(m_serverUri);
            }

            if (m_endpointUrl != null)
            {
                children.Add(m_endpointUrl);
            }

            if (m_localeIds != null)
            {
                children.Add(m_localeIds);
            }

            if (m_actualSessionTimeout != null)
            {
                children.Add(m_actualSessionTimeout);
            }

            if (m_maxResponseMessageSize != null)
            {
                children.Add(m_maxResponseMessageSize);
            }

            if (m_clientConnectionTime != null)
            {
                children.Add(m_clientConnectionTime);
            }

            if (m_clientLastContactTime != null)
            {
                children.Add(m_clientLastContactTime);
            }

            if (m_currentSubscriptionsCount != null)
            {
                children.Add(m_currentSubscriptionsCount);
            }

            if (m_currentMonitoredItemsCount != null)
            {
                children.Add(m_currentMonitoredItemsCount);
            }

            if (m_currentPublishRequestsInQueue != null)
            {
                children.Add(m_currentPublishRequestsInQueue);
            }

            if (m_totalRequestCount != null)
            {
                children.Add(m_totalRequestCount);
            }

            if (m_unauthorizedRequestCount != null)
            {
                children.Add(m_unauthorizedRequestCount);
            }

            if (m_readCount != null)
            {
                children.Add(m_readCount);
            }

            if (m_historyReadCount != null)
            {
                children.Add(m_historyReadCount);
            }

            if (m_writeCount != null)
            {
                children.Add(m_writeCount);
            }

            if (m_historyUpdateCount != null)
            {
                children.Add(m_historyUpdateCount);
            }

            if (m_callCount != null)
            {
                children.Add(m_callCount);
            }

            if (m_createMonitoredItemsCount != null)
            {
                children.Add(m_createMonitoredItemsCount);
            }

            if (m_modifyMonitoredItemsCount != null)
            {
                children.Add(m_modifyMonitoredItemsCount);
            }

            if (m_setMonitoringModeCount != null)
            {
                children.Add(m_setMonitoringModeCount);
            }

            if (m_setTriggeringCount != null)
            {
                children.Add(m_setTriggeringCount);
            }

            if (m_deleteMonitoredItemsCount != null)
            {
                children.Add(m_deleteMonitoredItemsCount);
            }

            if (m_createSubscriptionCount != null)
            {
                children.Add(m_createSubscriptionCount);
            }

            if (m_modifySubscriptionCount != null)
            {
                children.Add(m_modifySubscriptionCount);
            }

            if (m_setPublishingModeCount != null)
            {
                children.Add(m_setPublishingModeCount);
            }

            if (m_publishCount != null)
            {
                children.Add(m_publishCount);
            }

            if (m_republishCount != null)
            {
                children.Add(m_republishCount);
            }

            if (m_transferSubscriptionsCount != null)
            {
                children.Add(m_transferSubscriptionsCount);
            }

            if (m_deleteSubscriptionsCount != null)
            {
                children.Add(m_deleteSubscriptionsCount);
            }

            if (m_addNodesCount != null)
            {
                children.Add(m_addNodesCount);
            }

            if (m_addReferencesCount != null)
            {
                children.Add(m_addReferencesCount);
            }

            if (m_deleteNodesCount != null)
            {
                children.Add(m_deleteNodesCount);
            }

            if (m_deleteReferencesCount != null)
            {
                children.Add(m_deleteReferencesCount);
            }

            if (m_browseCount != null)
            {
                children.Add(m_browseCount);
            }

            if (m_browseNextCount != null)
            {
                children.Add(m_browseNextCount);
            }

            if (m_translateBrowsePathsToNodeIdsCount != null)
            {
                children.Add(m_translateBrowsePathsToNodeIdsCount);
            }

            if (m_queryFirstCount != null)
            {
                children.Add(m_queryFirstCount);
            }

            if (m_queryNextCount != null)
            {
                children.Add(m_queryNextCount);
            }

            if (m_registerNodesCount != null)
            {
                children.Add(m_registerNodesCount);
            }

            if (m_unregisterNodesCount != null)
            {
                children.Add(m_unregisterNodesCount);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionId:
                {
                    if (createOrReplace)
                    {
                        if (SessionId == null)
                        {
                            if (replacement == null)
                            {
                                SessionId = new BaseDataVariableState<NodeId>(this);
                            }
                            else
                            {
                                SessionId = (BaseDataVariableState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = SessionId;
                    break;
                }

                case Opc.Ua.BrowseNames.SessionName:
                {
                    if (createOrReplace)
                    {
                        if (SessionName == null)
                        {
                            if (replacement == null)
                            {
                                SessionName = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                SessionName = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = SessionName;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientDescription:
                {
                    if (createOrReplace)
                    {
                        if (ClientDescription == null)
                        {
                            if (replacement == null)
                            {
                                ClientDescription = new BaseDataVariableState<ApplicationDescription>(this);
                            }
                            else
                            {
                                ClientDescription = (BaseDataVariableState<ApplicationDescription>)replacement;
                            }
                        }
                    }

                    instance = ClientDescription;
                    break;
                }

                case Opc.Ua.BrowseNames.ServerUri:
                {
                    if (createOrReplace)
                    {
                        if (ServerUri == null)
                        {
                            if (replacement == null)
                            {
                                ServerUri = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                ServerUri = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = ServerUri;
                    break;
                }

                case Opc.Ua.BrowseNames.EndpointUrl:
                {
                    if (createOrReplace)
                    {
                        if (EndpointUrl == null)
                        {
                            if (replacement == null)
                            {
                                EndpointUrl = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                EndpointUrl = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = EndpointUrl;
                    break;
                }

                case Opc.Ua.BrowseNames.LocaleIds:
                {
                    if (createOrReplace)
                    {
                        if (LocaleIds == null)
                        {
                            if (replacement == null)
                            {
                                LocaleIds = new BaseDataVariableState<string[]>(this);
                            }
                            else
                            {
                                LocaleIds = (BaseDataVariableState<string[]>)replacement;
                            }
                        }
                    }

                    instance = LocaleIds;
                    break;
                }

                case Opc.Ua.BrowseNames.ActualSessionTimeout:
                {
                    if (createOrReplace)
                    {
                        if (ActualSessionTimeout == null)
                        {
                            if (replacement == null)
                            {
                                ActualSessionTimeout = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                ActualSessionTimeout = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = ActualSessionTimeout;
                    break;
                }

                case Opc.Ua.BrowseNames.MaxResponseMessageSize:
                {
                    if (createOrReplace)
                    {
                        if (MaxResponseMessageSize == null)
                        {
                            if (replacement == null)
                            {
                                MaxResponseMessageSize = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                MaxResponseMessageSize = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = MaxResponseMessageSize;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientConnectionTime:
                {
                    if (createOrReplace)
                    {
                        if (ClientConnectionTime == null)
                        {
                            if (replacement == null)
                            {
                                ClientConnectionTime = new BaseDataVariableState<DateTime>(this);
                            }
                            else
                            {
                                ClientConnectionTime = (BaseDataVariableState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = ClientConnectionTime;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientLastContactTime:
                {
                    if (createOrReplace)
                    {
                        if (ClientLastContactTime == null)
                        {
                            if (replacement == null)
                            {
                                ClientLastContactTime = new BaseDataVariableState<DateTime>(this);
                            }
                            else
                            {
                                ClientLastContactTime = (BaseDataVariableState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = ClientLastContactTime;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentSubscriptionsCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentSubscriptionsCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentSubscriptionsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentSubscriptionsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentSubscriptionsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (CurrentMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                CurrentMonitoredItemsCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentMonitoredItemsCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CurrentPublishRequestsInQueue:
                {
                    if (createOrReplace)
                    {
                        if (CurrentPublishRequestsInQueue == null)
                        {
                            if (replacement == null)
                            {
                                CurrentPublishRequestsInQueue = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                CurrentPublishRequestsInQueue = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = CurrentPublishRequestsInQueue;
                    break;
                }

                case Opc.Ua.BrowseNames.TotalRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (TotalRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                TotalRequestCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                TotalRequestCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = TotalRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.UnauthorizedRequestCount:
                {
                    if (createOrReplace)
                    {
                        if (UnauthorizedRequestCount == null)
                        {
                            if (replacement == null)
                            {
                                UnauthorizedRequestCount = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                UnauthorizedRequestCount = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = UnauthorizedRequestCount;
                    break;
                }

                case Opc.Ua.BrowseNames.ReadCount:
                {
                    if (createOrReplace)
                    {
                        if (ReadCount == null)
                        {
                            if (replacement == null)
                            {
                                ReadCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                ReadCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = ReadCount;
                    break;
                }

                case Opc.Ua.BrowseNames.HistoryReadCount:
                {
                    if (createOrReplace)
                    {
                        if (HistoryReadCount == null)
                        {
                            if (replacement == null)
                            {
                                HistoryReadCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                HistoryReadCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = HistoryReadCount;
                    break;
                }

                case Opc.Ua.BrowseNames.WriteCount:
                {
                    if (createOrReplace)
                    {
                        if (WriteCount == null)
                        {
                            if (replacement == null)
                            {
                                WriteCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                WriteCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = WriteCount;
                    break;
                }

                case Opc.Ua.BrowseNames.HistoryUpdateCount:
                {
                    if (createOrReplace)
                    {
                        if (HistoryUpdateCount == null)
                        {
                            if (replacement == null)
                            {
                                HistoryUpdateCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                HistoryUpdateCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = HistoryUpdateCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CallCount:
                {
                    if (createOrReplace)
                    {
                        if (CallCount == null)
                        {
                            if (replacement == null)
                            {
                                CallCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                CallCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = CallCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CreateMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (CreateMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                CreateMonitoredItemsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                CreateMonitoredItemsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = CreateMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.ModifyMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (ModifyMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                ModifyMonitoredItemsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                ModifyMonitoredItemsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = ModifyMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SetMonitoringModeCount:
                {
                    if (createOrReplace)
                    {
                        if (SetMonitoringModeCount == null)
                        {
                            if (replacement == null)
                            {
                                SetMonitoringModeCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                SetMonitoringModeCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = SetMonitoringModeCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SetTriggeringCount:
                {
                    if (createOrReplace)
                    {
                        if (SetTriggeringCount == null)
                        {
                            if (replacement == null)
                            {
                                SetTriggeringCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                SetTriggeringCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = SetTriggeringCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DeleteMonitoredItemsCount:
                {
                    if (createOrReplace)
                    {
                        if (DeleteMonitoredItemsCount == null)
                        {
                            if (replacement == null)
                            {
                                DeleteMonitoredItemsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                DeleteMonitoredItemsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = DeleteMonitoredItemsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.CreateSubscriptionCount:
                {
                    if (createOrReplace)
                    {
                        if (CreateSubscriptionCount == null)
                        {
                            if (replacement == null)
                            {
                                CreateSubscriptionCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                CreateSubscriptionCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = CreateSubscriptionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.ModifySubscriptionCount:
                {
                    if (createOrReplace)
                    {
                        if (ModifySubscriptionCount == null)
                        {
                            if (replacement == null)
                            {
                                ModifySubscriptionCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                ModifySubscriptionCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = ModifySubscriptionCount;
                    break;
                }

                case Opc.Ua.BrowseNames.SetPublishingModeCount:
                {
                    if (createOrReplace)
                    {
                        if (SetPublishingModeCount == null)
                        {
                            if (replacement == null)
                            {
                                SetPublishingModeCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                SetPublishingModeCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = SetPublishingModeCount;
                    break;
                }

                case Opc.Ua.BrowseNames.PublishCount:
                {
                    if (createOrReplace)
                    {
                        if (PublishCount == null)
                        {
                            if (replacement == null)
                            {
                                PublishCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                PublishCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = PublishCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RepublishCount:
                {
                    if (createOrReplace)
                    {
                        if (RepublishCount == null)
                        {
                            if (replacement == null)
                            {
                                RepublishCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                RepublishCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = RepublishCount;
                    break;
                }

                case Opc.Ua.BrowseNames.TransferSubscriptionsCount:
                {
                    if (createOrReplace)
                    {
                        if (TransferSubscriptionsCount == null)
                        {
                            if (replacement == null)
                            {
                                TransferSubscriptionsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                TransferSubscriptionsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = TransferSubscriptionsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DeleteSubscriptionsCount:
                {
                    if (createOrReplace)
                    {
                        if (DeleteSubscriptionsCount == null)
                        {
                            if (replacement == null)
                            {
                                DeleteSubscriptionsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                DeleteSubscriptionsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = DeleteSubscriptionsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.AddNodesCount:
                {
                    if (createOrReplace)
                    {
                        if (AddNodesCount == null)
                        {
                            if (replacement == null)
                            {
                                AddNodesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                AddNodesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = AddNodesCount;
                    break;
                }

                case Opc.Ua.BrowseNames.AddReferencesCount:
                {
                    if (createOrReplace)
                    {
                        if (AddReferencesCount == null)
                        {
                            if (replacement == null)
                            {
                                AddReferencesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                AddReferencesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = AddReferencesCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DeleteNodesCount:
                {
                    if (createOrReplace)
                    {
                        if (DeleteNodesCount == null)
                        {
                            if (replacement == null)
                            {
                                DeleteNodesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                DeleteNodesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = DeleteNodesCount;
                    break;
                }

                case Opc.Ua.BrowseNames.DeleteReferencesCount:
                {
                    if (createOrReplace)
                    {
                        if (DeleteReferencesCount == null)
                        {
                            if (replacement == null)
                            {
                                DeleteReferencesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                DeleteReferencesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = DeleteReferencesCount;
                    break;
                }

                case Opc.Ua.BrowseNames.BrowseCount:
                {
                    if (createOrReplace)
                    {
                        if (BrowseCount == null)
                        {
                            if (replacement == null)
                            {
                                BrowseCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                BrowseCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = BrowseCount;
                    break;
                }

                case Opc.Ua.BrowseNames.BrowseNextCount:
                {
                    if (createOrReplace)
                    {
                        if (BrowseNextCount == null)
                        {
                            if (replacement == null)
                            {
                                BrowseNextCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                BrowseNextCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = BrowseNextCount;
                    break;
                }

                case Opc.Ua.BrowseNames.TranslateBrowsePathsToNodeIdsCount:
                {
                    if (createOrReplace)
                    {
                        if (TranslateBrowsePathsToNodeIdsCount == null)
                        {
                            if (replacement == null)
                            {
                                TranslateBrowsePathsToNodeIdsCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                TranslateBrowsePathsToNodeIdsCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = TranslateBrowsePathsToNodeIdsCount;
                    break;
                }

                case Opc.Ua.BrowseNames.QueryFirstCount:
                {
                    if (createOrReplace)
                    {
                        if (QueryFirstCount == null)
                        {
                            if (replacement == null)
                            {
                                QueryFirstCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                QueryFirstCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = QueryFirstCount;
                    break;
                }

                case Opc.Ua.BrowseNames.QueryNextCount:
                {
                    if (createOrReplace)
                    {
                        if (QueryNextCount == null)
                        {
                            if (replacement == null)
                            {
                                QueryNextCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                QueryNextCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = QueryNextCount;
                    break;
                }

                case Opc.Ua.BrowseNames.RegisterNodesCount:
                {
                    if (createOrReplace)
                    {
                        if (RegisterNodesCount == null)
                        {
                            if (replacement == null)
                            {
                                RegisterNodesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                RegisterNodesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = RegisterNodesCount;
                    break;
                }

                case Opc.Ua.BrowseNames.UnregisterNodesCount:
                {
                    if (createOrReplace)
                    {
                        if (UnregisterNodesCount == null)
                        {
                            if (replacement == null)
                            {
                                UnregisterNodesCount = new BaseDataVariableState<ServiceCounterDataType>(this);
                            }
                            else
                            {
                                UnregisterNodesCount = (BaseDataVariableState<ServiceCounterDataType>)replacement;
                            }
                        }
                    }

                    instance = UnregisterNodesCount;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<NodeId> m_sessionId;
        private BaseDataVariableState<string> m_sessionName;
        private BaseDataVariableState<ApplicationDescription> m_clientDescription;
        private BaseDataVariableState<string> m_serverUri;
        private BaseDataVariableState<string> m_endpointUrl;
        private BaseDataVariableState<string[]> m_localeIds;
        private BaseDataVariableState<double> m_actualSessionTimeout;
        private BaseDataVariableState<uint> m_maxResponseMessageSize;
        private BaseDataVariableState<DateTime> m_clientConnectionTime;
        private BaseDataVariableState<DateTime> m_clientLastContactTime;
        private BaseDataVariableState<uint> m_currentSubscriptionsCount;
        private BaseDataVariableState<uint> m_currentMonitoredItemsCount;
        private BaseDataVariableState<uint> m_currentPublishRequestsInQueue;
        private BaseDataVariableState<ServiceCounterDataType> m_totalRequestCount;
        private BaseDataVariableState<uint> m_unauthorizedRequestCount;
        private BaseDataVariableState<ServiceCounterDataType> m_readCount;
        private BaseDataVariableState<ServiceCounterDataType> m_historyReadCount;
        private BaseDataVariableState<ServiceCounterDataType> m_writeCount;
        private BaseDataVariableState<ServiceCounterDataType> m_historyUpdateCount;
        private BaseDataVariableState<ServiceCounterDataType> m_callCount;
        private BaseDataVariableState<ServiceCounterDataType> m_createMonitoredItemsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_modifyMonitoredItemsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_setMonitoringModeCount;
        private BaseDataVariableState<ServiceCounterDataType> m_setTriggeringCount;
        private BaseDataVariableState<ServiceCounterDataType> m_deleteMonitoredItemsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_createSubscriptionCount;
        private BaseDataVariableState<ServiceCounterDataType> m_modifySubscriptionCount;
        private BaseDataVariableState<ServiceCounterDataType> m_setPublishingModeCount;
        private BaseDataVariableState<ServiceCounterDataType> m_publishCount;
        private BaseDataVariableState<ServiceCounterDataType> m_republishCount;
        private BaseDataVariableState<ServiceCounterDataType> m_transferSubscriptionsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_deleteSubscriptionsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_addNodesCount;
        private BaseDataVariableState<ServiceCounterDataType> m_addReferencesCount;
        private BaseDataVariableState<ServiceCounterDataType> m_deleteNodesCount;
        private BaseDataVariableState<ServiceCounterDataType> m_deleteReferencesCount;
        private BaseDataVariableState<ServiceCounterDataType> m_browseCount;
        private BaseDataVariableState<ServiceCounterDataType> m_browseNextCount;
        private BaseDataVariableState<ServiceCounterDataType> m_translateBrowsePathsToNodeIdsCount;
        private BaseDataVariableState<ServiceCounterDataType> m_queryFirstCount;
        private BaseDataVariableState<ServiceCounterDataType> m_queryNextCount;
        private BaseDataVariableState<ServiceCounterDataType> m_registerNodesCount;
        private BaseDataVariableState<ServiceCounterDataType> m_unregisterNodesCount;
        #endregion
    }

    #region SessionDiagnosticsVariableValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class SessionDiagnosticsVariableValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public SessionDiagnosticsVariableValue(SessionDiagnosticsVariableState variable, SessionDiagnosticsDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new SessionDiagnosticsDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public SessionDiagnosticsVariableState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public SessionDiagnosticsDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(SessionDiagnosticsVariableState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.SessionId;
                instance.OnReadValue = OnRead_SessionId;
                instance.OnSimpleWriteValue = OnWrite_SessionId;
                updateList.Add(instance);
                instance = m_variable.SessionName;
                instance.OnReadValue = OnRead_SessionName;
                instance.OnSimpleWriteValue = OnWrite_SessionName;
                updateList.Add(instance);
                instance = m_variable.ClientDescription;
                instance.OnReadValue = OnRead_ClientDescription;
                instance.OnSimpleWriteValue = OnWrite_ClientDescription;
                updateList.Add(instance);
                instance = m_variable.ServerUri;
                instance.OnReadValue = OnRead_ServerUri;
                instance.OnSimpleWriteValue = OnWrite_ServerUri;
                updateList.Add(instance);
                instance = m_variable.EndpointUrl;
                instance.OnReadValue = OnRead_EndpointUrl;
                instance.OnSimpleWriteValue = OnWrite_EndpointUrl;
                updateList.Add(instance);
                instance = m_variable.LocaleIds;
                instance.OnReadValue = OnRead_LocaleIds;
                instance.OnSimpleWriteValue = OnWrite_LocaleIds;
                updateList.Add(instance);
                instance = m_variable.ActualSessionTimeout;
                instance.OnReadValue = OnRead_ActualSessionTimeout;
                instance.OnSimpleWriteValue = OnWrite_ActualSessionTimeout;
                updateList.Add(instance);
                instance = m_variable.MaxResponseMessageSize;
                instance.OnReadValue = OnRead_MaxResponseMessageSize;
                instance.OnSimpleWriteValue = OnWrite_MaxResponseMessageSize;
                updateList.Add(instance);
                instance = m_variable.ClientConnectionTime;
                instance.OnReadValue = OnRead_ClientConnectionTime;
                instance.OnSimpleWriteValue = OnWrite_ClientConnectionTime;
                updateList.Add(instance);
                instance = m_variable.ClientLastContactTime;
                instance.OnReadValue = OnRead_ClientLastContactTime;
                instance.OnSimpleWriteValue = OnWrite_ClientLastContactTime;
                updateList.Add(instance);
                instance = m_variable.CurrentSubscriptionsCount;
                instance.OnReadValue = OnRead_CurrentSubscriptionsCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentSubscriptionsCount;
                updateList.Add(instance);
                instance = m_variable.CurrentMonitoredItemsCount;
                instance.OnReadValue = OnRead_CurrentMonitoredItemsCount;
                instance.OnSimpleWriteValue = OnWrite_CurrentMonitoredItemsCount;
                updateList.Add(instance);
                instance = m_variable.CurrentPublishRequestsInQueue;
                instance.OnReadValue = OnRead_CurrentPublishRequestsInQueue;
                instance.OnSimpleWriteValue = OnWrite_CurrentPublishRequestsInQueue;
                updateList.Add(instance);
                instance = m_variable.TotalRequestCount;
                instance.OnReadValue = OnRead_TotalRequestCount;
                instance.OnSimpleWriteValue = OnWrite_TotalRequestCount;
                updateList.Add(instance);
                instance = m_variable.UnauthorizedRequestCount;
                instance.OnReadValue = OnRead_UnauthorizedRequestCount;
                instance.OnSimpleWriteValue = OnWrite_UnauthorizedRequestCount;
                updateList.Add(instance);
                instance = m_variable.ReadCount;
                instance.OnReadValue = OnRead_ReadCount;
                instance.OnSimpleWriteValue = OnWrite_ReadCount;
                updateList.Add(instance);
                instance = m_variable.HistoryReadCount;
                instance.OnReadValue = OnRead_HistoryReadCount;
                instance.OnSimpleWriteValue = OnWrite_HistoryReadCount;
                updateList.Add(instance);
                instance = m_variable.WriteCount;
                instance.OnReadValue = OnRead_WriteCount;
                instance.OnSimpleWriteValue = OnWrite_WriteCount;
                updateList.Add(instance);
                instance = m_variable.HistoryUpdateCount;
                instance.OnReadValue = OnRead_HistoryUpdateCount;
                instance.OnSimpleWriteValue = OnWrite_HistoryUpdateCount;
                updateList.Add(instance);
                instance = m_variable.CallCount;
                instance.OnReadValue = OnRead_CallCount;
                instance.OnSimpleWriteValue = OnWrite_CallCount;
                updateList.Add(instance);
                instance = m_variable.CreateMonitoredItemsCount;
                instance.OnReadValue = OnRead_CreateMonitoredItemsCount;
                instance.OnSimpleWriteValue = OnWrite_CreateMonitoredItemsCount;
                updateList.Add(instance);
                instance = m_variable.ModifyMonitoredItemsCount;
                instance.OnReadValue = OnRead_ModifyMonitoredItemsCount;
                instance.OnSimpleWriteValue = OnWrite_ModifyMonitoredItemsCount;
                updateList.Add(instance);
                instance = m_variable.SetMonitoringModeCount;
                instance.OnReadValue = OnRead_SetMonitoringModeCount;
                instance.OnSimpleWriteValue = OnWrite_SetMonitoringModeCount;
                updateList.Add(instance);
                instance = m_variable.SetTriggeringCount;
                instance.OnReadValue = OnRead_SetTriggeringCount;
                instance.OnSimpleWriteValue = OnWrite_SetTriggeringCount;
                updateList.Add(instance);
                instance = m_variable.DeleteMonitoredItemsCount;
                instance.OnReadValue = OnRead_DeleteMonitoredItemsCount;
                instance.OnSimpleWriteValue = OnWrite_DeleteMonitoredItemsCount;
                updateList.Add(instance);
                instance = m_variable.CreateSubscriptionCount;
                instance.OnReadValue = OnRead_CreateSubscriptionCount;
                instance.OnSimpleWriteValue = OnWrite_CreateSubscriptionCount;
                updateList.Add(instance);
                instance = m_variable.ModifySubscriptionCount;
                instance.OnReadValue = OnRead_ModifySubscriptionCount;
                instance.OnSimpleWriteValue = OnWrite_ModifySubscriptionCount;
                updateList.Add(instance);
                instance = m_variable.SetPublishingModeCount;
                instance.OnReadValue = OnRead_SetPublishingModeCount;
                instance.OnSimpleWriteValue = OnWrite_SetPublishingModeCount;
                updateList.Add(instance);
                instance = m_variable.PublishCount;
                instance.OnReadValue = OnRead_PublishCount;
                instance.OnSimpleWriteValue = OnWrite_PublishCount;
                updateList.Add(instance);
                instance = m_variable.RepublishCount;
                instance.OnReadValue = OnRead_RepublishCount;
                instance.OnSimpleWriteValue = OnWrite_RepublishCount;
                updateList.Add(instance);
                instance = m_variable.TransferSubscriptionsCount;
                instance.OnReadValue = OnRead_TransferSubscriptionsCount;
                instance.OnSimpleWriteValue = OnWrite_TransferSubscriptionsCount;
                updateList.Add(instance);
                instance = m_variable.DeleteSubscriptionsCount;
                instance.OnReadValue = OnRead_DeleteSubscriptionsCount;
                instance.OnSimpleWriteValue = OnWrite_DeleteSubscriptionsCount;
                updateList.Add(instance);
                instance = m_variable.AddNodesCount;
                instance.OnReadValue = OnRead_AddNodesCount;
                instance.OnSimpleWriteValue = OnWrite_AddNodesCount;
                updateList.Add(instance);
                instance = m_variable.AddReferencesCount;
                instance.OnReadValue = OnRead_AddReferencesCount;
                instance.OnSimpleWriteValue = OnWrite_AddReferencesCount;
                updateList.Add(instance);
                instance = m_variable.DeleteNodesCount;
                instance.OnReadValue = OnRead_DeleteNodesCount;
                instance.OnSimpleWriteValue = OnWrite_DeleteNodesCount;
                updateList.Add(instance);
                instance = m_variable.DeleteReferencesCount;
                instance.OnReadValue = OnRead_DeleteReferencesCount;
                instance.OnSimpleWriteValue = OnWrite_DeleteReferencesCount;
                updateList.Add(instance);
                instance = m_variable.BrowseCount;
                instance.OnReadValue = OnRead_BrowseCount;
                instance.OnSimpleWriteValue = OnWrite_BrowseCount;
                updateList.Add(instance);
                instance = m_variable.BrowseNextCount;
                instance.OnReadValue = OnRead_BrowseNextCount;
                instance.OnSimpleWriteValue = OnWrite_BrowseNextCount;
                updateList.Add(instance);
                instance = m_variable.TranslateBrowsePathsToNodeIdsCount;
                instance.OnReadValue = OnRead_TranslateBrowsePathsToNodeIdsCount;
                instance.OnSimpleWriteValue = OnWrite_TranslateBrowsePathsToNodeIdsCount;
                updateList.Add(instance);
                instance = m_variable.QueryFirstCount;
                instance.OnReadValue = OnRead_QueryFirstCount;
                instance.OnSimpleWriteValue = OnWrite_QueryFirstCount;
                updateList.Add(instance);
                instance = m_variable.QueryNextCount;
                instance.OnReadValue = OnRead_QueryNextCount;
                instance.OnSimpleWriteValue = OnWrite_QueryNextCount;
                updateList.Add(instance);
                instance = m_variable.RegisterNodesCount;
                instance.OnReadValue = OnRead_RegisterNodesCount;
                instance.OnSimpleWriteValue = OnWrite_RegisterNodesCount;
                updateList.Add(instance);
                instance = m_variable.UnregisterNodesCount;
                instance.OnReadValue = OnRead_UnregisterNodesCount;
                instance.OnSimpleWriteValue = OnWrite_UnregisterNodesCount;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (SessionDiagnosticsDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region SessionId Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionId(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionId;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionId(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionId = (NodeId)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SessionName Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionName(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionName;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionName(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionName = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientDescription Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientDescription(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientDescription;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientDescription(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientDescription = (ApplicationDescription)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ServerUri Access Methods
        /// <remarks />
        private ServiceResult OnRead_ServerUri(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ServerUri;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ServerUri(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ServerUri = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region EndpointUrl Access Methods
        /// <remarks />
        private ServiceResult OnRead_EndpointUrl(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.EndpointUrl;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_EndpointUrl(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.EndpointUrl = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region LocaleIds Access Methods
        /// <remarks />
        private ServiceResult OnRead_LocaleIds(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.LocaleIds;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_LocaleIds(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.LocaleIds = (StringCollection)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ActualSessionTimeout Access Methods
        /// <remarks />
        private ServiceResult OnRead_ActualSessionTimeout(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ActualSessionTimeout;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ActualSessionTimeout(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ActualSessionTimeout = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region MaxResponseMessageSize Access Methods
        /// <remarks />
        private ServiceResult OnRead_MaxResponseMessageSize(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.MaxResponseMessageSize;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_MaxResponseMessageSize(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.MaxResponseMessageSize = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientConnectionTime Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientConnectionTime(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientConnectionTime;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientConnectionTime(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientConnectionTime = (DateTime)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientLastContactTime Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientLastContactTime(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientLastContactTime;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientLastContactTime(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientLastContactTime = (DateTime)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentSubscriptionsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentSubscriptionsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentSubscriptionsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentSubscriptionsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentSubscriptionsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentMonitoredItemsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentMonitoredItemsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentMonitoredItemsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentMonitoredItemsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentMonitoredItemsCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CurrentPublishRequestsInQueue Access Methods
        /// <remarks />
        private ServiceResult OnRead_CurrentPublishRequestsInQueue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CurrentPublishRequestsInQueue;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CurrentPublishRequestsInQueue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CurrentPublishRequestsInQueue = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TotalRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TotalRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TotalRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TotalRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TotalRequestCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region UnauthorizedRequestCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_UnauthorizedRequestCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.UnauthorizedRequestCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_UnauthorizedRequestCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.UnauthorizedRequestCount = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ReadCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_ReadCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ReadCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ReadCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ReadCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region HistoryReadCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_HistoryReadCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.HistoryReadCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_HistoryReadCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.HistoryReadCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region WriteCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_WriteCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.WriteCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_WriteCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.WriteCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region HistoryUpdateCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_HistoryUpdateCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.HistoryUpdateCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_HistoryUpdateCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.HistoryUpdateCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CallCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CallCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CallCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CallCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CallCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CreateMonitoredItemsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CreateMonitoredItemsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CreateMonitoredItemsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CreateMonitoredItemsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CreateMonitoredItemsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ModifyMonitoredItemsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_ModifyMonitoredItemsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ModifyMonitoredItemsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ModifyMonitoredItemsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ModifyMonitoredItemsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SetMonitoringModeCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SetMonitoringModeCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SetMonitoringModeCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SetMonitoringModeCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SetMonitoringModeCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SetTriggeringCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SetTriggeringCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SetTriggeringCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SetTriggeringCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SetTriggeringCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DeleteMonitoredItemsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DeleteMonitoredItemsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DeleteMonitoredItemsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DeleteMonitoredItemsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DeleteMonitoredItemsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region CreateSubscriptionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_CreateSubscriptionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.CreateSubscriptionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_CreateSubscriptionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.CreateSubscriptionCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ModifySubscriptionCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_ModifySubscriptionCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ModifySubscriptionCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ModifySubscriptionCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ModifySubscriptionCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SetPublishingModeCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_SetPublishingModeCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SetPublishingModeCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SetPublishingModeCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SetPublishingModeCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region PublishCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_PublishCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.PublishCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_PublishCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.PublishCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RepublishCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RepublishCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RepublishCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RepublishCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RepublishCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TransferSubscriptionsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TransferSubscriptionsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TransferSubscriptionsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TransferSubscriptionsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TransferSubscriptionsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DeleteSubscriptionsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DeleteSubscriptionsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DeleteSubscriptionsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DeleteSubscriptionsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DeleteSubscriptionsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region AddNodesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_AddNodesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.AddNodesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_AddNodesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.AddNodesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region AddReferencesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_AddReferencesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.AddReferencesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_AddReferencesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.AddReferencesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DeleteNodesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DeleteNodesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DeleteNodesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DeleteNodesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DeleteNodesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region DeleteReferencesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_DeleteReferencesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.DeleteReferencesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_DeleteReferencesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.DeleteReferencesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region BrowseCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_BrowseCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.BrowseCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_BrowseCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.BrowseCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region BrowseNextCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_BrowseNextCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.BrowseNextCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_BrowseNextCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.BrowseNextCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TranslateBrowsePathsToNodeIdsCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_TranslateBrowsePathsToNodeIdsCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TranslateBrowsePathsToNodeIdsCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TranslateBrowsePathsToNodeIdsCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TranslateBrowsePathsToNodeIdsCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region QueryFirstCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_QueryFirstCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.QueryFirstCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_QueryFirstCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.QueryFirstCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region QueryNextCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_QueryNextCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.QueryNextCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_QueryNextCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.QueryNextCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region RegisterNodesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_RegisterNodesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.RegisterNodesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_RegisterNodesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.RegisterNodesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region UnregisterNodesCount Access Methods
        /// <remarks />
        private ServiceResult OnRead_UnregisterNodesCount(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.UnregisterNodesCount;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_UnregisterNodesCount(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.UnregisterNodesCount = (ServiceCounterDataType)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private SessionDiagnosticsDataType m_value;
        private SessionDiagnosticsVariableState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region SessionSecurityDiagnosticsArrayState Class
    #if (!OPCUA_EXCLUDE_SessionSecurityDiagnosticsArrayState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionSecurityDiagnosticsArrayState : BaseDataVariableState<SessionSecurityDiagnosticsDataType[]>
    {
        #region Constructors
        /// <remarks />
        public SessionSecurityDiagnosticsArrayState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SessionSecurityDiagnosticsArrayType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SessionSecurityDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.OneDimension;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8XYIkCAgAAAAAAKwAAAFNlc3Npb25TZWN1cml0eURpYWdub3N0aWNzQXJyYXlUeXBlSW5z" +
           "dGFuY2UBAMMIAQDDCMMIAAABAGQDAQAAAAEAAAAAAAAAAQH/////AQAAABVgiQoCAAAAAAAaAAAAU2Vz" +
           "c2lvblNlY3VyaXR5RGlhZ25vc3RpY3MBADwyAC8BAMQIPDIAAAEAZAP/////AQH/////CQAAABVgiQoC" +
           "AAAAAAAJAAAAU2Vzc2lvbklkAQA9MgAvAD89MgAAABH/////AQH/////AAAAABVgiQoCAAAAAAAVAAAA" +
           "Q2xpZW50VXNlcklkT2ZTZXNzaW9uAQA+MgAvAD8+MgAAAAz/////AQH/////AAAAABdgiQoCAAAAAAAT" +
           "AAAAQ2xpZW50VXNlcklkSGlzdG9yeQEAPzIALwA/PzIAAAAMAQAAAAEAAAAAAAAAAQH/////AAAAABVg" +
           "iQoCAAAAAAAXAAAAQXV0aGVudGljYXRpb25NZWNoYW5pc20BAEAyAC8AP0AyAAAADP////8BAf////8A" +
           "AAAAFWCJCgIAAAAAAAgAAABFbmNvZGluZwEAQTIALwA/QTIAAAAM/////wEB/////wAAAAAVYIkKAgAA" +
           "AAAAEQAAAFRyYW5zcG9ydFByb3RvY29sAQBCMgAvAD9CMgAAAAz/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAMAAAAU2VjdXJpdHlNb2RlAQBDMgAvAD9DMgAAAQAuAf////8BAf////8AAAAAFWCJCgIAAAAAABEA" +
           "AABTZWN1cml0eVBvbGljeVVyaQEARDIALwA/RDIAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAAEQAA" +
           "AENsaWVudENlcnRpZmljYXRlAQBFMgAvAD9FMgAAAA//////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region SessionSecurityDiagnosticsState Class
    #if (!OPCUA_EXCLUDE_SessionSecurityDiagnosticsState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SessionSecurityDiagnosticsState : BaseDataVariableState<SessionSecurityDiagnosticsDataType>
    {
        #region Constructors
        /// <remarks />
        public SessionSecurityDiagnosticsState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SessionSecurityDiagnosticsType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.SessionSecurityDiagnosticsDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJgAAAFNlc3Npb25TZWN1cml0eURpYWdub3N0aWNzVHlwZUluc3RhbmNl" +
           "AQDECAEAxAjECAAAAQBkA/////8BAf////8JAAAAFWCJCgIAAAAAAAkAAABTZXNzaW9uSWQBAMUIAC8A" +
           "P8UIAAAAEf////8BAf////8AAAAAFWCJCgIAAAAAABUAAABDbGllbnRVc2VySWRPZlNlc3Npb24BAMYI" +
           "AC8AP8YIAAAADP////8BAf////8AAAAAF2CJCgIAAAAAABMAAABDbGllbnRVc2VySWRIaXN0b3J5AQDH" +
           "CAAvAD/HCAAAAAwBAAAAAQAAAAAAAAABAf////8AAAAAFWCJCgIAAAAAABcAAABBdXRoZW50aWNhdGlv" +
           "bk1lY2hhbmlzbQEAyAgALwA/yAgAAAAM/////wEB/////wAAAAAVYIkKAgAAAAAACAAAAEVuY29kaW5n" +
           "AQDJCAAvAD/JCAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAARAAAAVHJhbnNwb3J0UHJvdG9jb2wB" +
           "AMoIAC8AP8oIAAAADP////8BAf////8AAAAAFWCJCgIAAAAAAAwAAABTZWN1cml0eU1vZGUBAMsIAC8A" +
           "P8sIAAABAC4B/////wEB/////wAAAAAVYIkKAgAAAAAAEQAAAFNlY3VyaXR5UG9saWN5VXJpAQDMCAAv" +
           "AD/MCAAAAAz/////AQH/////AAAAABVgiQoCAAAAAAARAAAAQ2xpZW50Q2VydGlmaWNhdGUBAPILAC8A" +
           "P/ILAAAAD/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<NodeId> SessionId
        {
            get
            {
                return m_sessionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_sessionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_sessionId = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> ClientUserIdOfSession
        {
            get
            {
                return m_clientUserIdOfSession;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientUserIdOfSession, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientUserIdOfSession = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string[]> ClientUserIdHistory
        {
            get
            {
                return m_clientUserIdHistory;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientUserIdHistory, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientUserIdHistory = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> AuthenticationMechanism
        {
            get
            {
                return m_authenticationMechanism;
            }

            set
            {
                if (!Object.ReferenceEquals(m_authenticationMechanism, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_authenticationMechanism = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> Encoding
        {
            get
            {
                return m_encoding;
            }

            set
            {
                if (!Object.ReferenceEquals(m_encoding, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_encoding = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> TransportProtocol
        {
            get
            {
                return m_transportProtocol;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transportProtocol, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transportProtocol = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<MessageSecurityMode> SecurityMode
        {
            get
            {
                return m_securityMode;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityMode, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityMode = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<string> SecurityPolicyUri
        {
            get
            {
                return m_securityPolicyUri;
            }

            set
            {
                if (!Object.ReferenceEquals(m_securityPolicyUri, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_securityPolicyUri = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<byte[]> ClientCertificate
        {
            get
            {
                return m_clientCertificate;
            }

            set
            {
                if (!Object.ReferenceEquals(m_clientCertificate, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_clientCertificate = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_sessionId != null)
            {
                children.Add(m_sessionId);
            }

            if (m_clientUserIdOfSession != null)
            {
                children.Add(m_clientUserIdOfSession);
            }

            if (m_clientUserIdHistory != null)
            {
                children.Add(m_clientUserIdHistory);
            }

            if (m_authenticationMechanism != null)
            {
                children.Add(m_authenticationMechanism);
            }

            if (m_encoding != null)
            {
                children.Add(m_encoding);
            }

            if (m_transportProtocol != null)
            {
                children.Add(m_transportProtocol);
            }

            if (m_securityMode != null)
            {
                children.Add(m_securityMode);
            }

            if (m_securityPolicyUri != null)
            {
                children.Add(m_securityPolicyUri);
            }

            if (m_clientCertificate != null)
            {
                children.Add(m_clientCertificate);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.SessionId:
                {
                    if (createOrReplace)
                    {
                        if (SessionId == null)
                        {
                            if (replacement == null)
                            {
                                SessionId = new BaseDataVariableState<NodeId>(this);
                            }
                            else
                            {
                                SessionId = (BaseDataVariableState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = SessionId;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientUserIdOfSession:
                {
                    if (createOrReplace)
                    {
                        if (ClientUserIdOfSession == null)
                        {
                            if (replacement == null)
                            {
                                ClientUserIdOfSession = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                ClientUserIdOfSession = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = ClientUserIdOfSession;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientUserIdHistory:
                {
                    if (createOrReplace)
                    {
                        if (ClientUserIdHistory == null)
                        {
                            if (replacement == null)
                            {
                                ClientUserIdHistory = new BaseDataVariableState<string[]>(this);
                            }
                            else
                            {
                                ClientUserIdHistory = (BaseDataVariableState<string[]>)replacement;
                            }
                        }
                    }

                    instance = ClientUserIdHistory;
                    break;
                }

                case Opc.Ua.BrowseNames.AuthenticationMechanism:
                {
                    if (createOrReplace)
                    {
                        if (AuthenticationMechanism == null)
                        {
                            if (replacement == null)
                            {
                                AuthenticationMechanism = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                AuthenticationMechanism = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = AuthenticationMechanism;
                    break;
                }

                case Opc.Ua.BrowseNames.Encoding:
                {
                    if (createOrReplace)
                    {
                        if (Encoding == null)
                        {
                            if (replacement == null)
                            {
                                Encoding = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                Encoding = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = Encoding;
                    break;
                }

                case Opc.Ua.BrowseNames.TransportProtocol:
                {
                    if (createOrReplace)
                    {
                        if (TransportProtocol == null)
                        {
                            if (replacement == null)
                            {
                                TransportProtocol = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                TransportProtocol = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = TransportProtocol;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityMode:
                {
                    if (createOrReplace)
                    {
                        if (SecurityMode == null)
                        {
                            if (replacement == null)
                            {
                                SecurityMode = new BaseDataVariableState<MessageSecurityMode>(this);
                            }
                            else
                            {
                                SecurityMode = (BaseDataVariableState<MessageSecurityMode>)replacement;
                            }
                        }
                    }

                    instance = SecurityMode;
                    break;
                }

                case Opc.Ua.BrowseNames.SecurityPolicyUri:
                {
                    if (createOrReplace)
                    {
                        if (SecurityPolicyUri == null)
                        {
                            if (replacement == null)
                            {
                                SecurityPolicyUri = new BaseDataVariableState<string>(this);
                            }
                            else
                            {
                                SecurityPolicyUri = (BaseDataVariableState<string>)replacement;
                            }
                        }
                    }

                    instance = SecurityPolicyUri;
                    break;
                }

                case Opc.Ua.BrowseNames.ClientCertificate:
                {
                    if (createOrReplace)
                    {
                        if (ClientCertificate == null)
                        {
                            if (replacement == null)
                            {
                                ClientCertificate = new BaseDataVariableState<byte[]>(this);
                            }
                            else
                            {
                                ClientCertificate = (BaseDataVariableState<byte[]>)replacement;
                            }
                        }
                    }

                    instance = ClientCertificate;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<NodeId> m_sessionId;
        private BaseDataVariableState<string> m_clientUserIdOfSession;
        private BaseDataVariableState<string[]> m_clientUserIdHistory;
        private BaseDataVariableState<string> m_authenticationMechanism;
        private BaseDataVariableState<string> m_encoding;
        private BaseDataVariableState<string> m_transportProtocol;
        private BaseDataVariableState<MessageSecurityMode> m_securityMode;
        private BaseDataVariableState<string> m_securityPolicyUri;
        private BaseDataVariableState<byte[]> m_clientCertificate;
        #endregion
    }

    #region SessionSecurityDiagnosticsValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class SessionSecurityDiagnosticsValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public SessionSecurityDiagnosticsValue(SessionSecurityDiagnosticsState variable, SessionSecurityDiagnosticsDataType value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new SessionSecurityDiagnosticsDataType();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public SessionSecurityDiagnosticsState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public SessionSecurityDiagnosticsDataType Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(SessionSecurityDiagnosticsState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.SessionId;
                instance.OnReadValue = OnRead_SessionId;
                instance.OnSimpleWriteValue = OnWrite_SessionId;
                updateList.Add(instance);
                instance = m_variable.ClientUserIdOfSession;
                instance.OnReadValue = OnRead_ClientUserIdOfSession;
                instance.OnSimpleWriteValue = OnWrite_ClientUserIdOfSession;
                updateList.Add(instance);
                instance = m_variable.ClientUserIdHistory;
                instance.OnReadValue = OnRead_ClientUserIdHistory;
                instance.OnSimpleWriteValue = OnWrite_ClientUserIdHistory;
                updateList.Add(instance);
                instance = m_variable.AuthenticationMechanism;
                instance.OnReadValue = OnRead_AuthenticationMechanism;
                instance.OnSimpleWriteValue = OnWrite_AuthenticationMechanism;
                updateList.Add(instance);
                instance = m_variable.Encoding;
                instance.OnReadValue = OnRead_Encoding;
                instance.OnSimpleWriteValue = OnWrite_Encoding;
                updateList.Add(instance);
                instance = m_variable.TransportProtocol;
                instance.OnReadValue = OnRead_TransportProtocol;
                instance.OnSimpleWriteValue = OnWrite_TransportProtocol;
                updateList.Add(instance);
                instance = m_variable.SecurityMode;
                instance.OnReadValue = OnRead_SecurityMode;
                instance.OnSimpleWriteValue = OnWrite_SecurityMode;
                updateList.Add(instance);
                instance = m_variable.SecurityPolicyUri;
                instance.OnReadValue = OnRead_SecurityPolicyUri;
                instance.OnSimpleWriteValue = OnWrite_SecurityPolicyUri;
                updateList.Add(instance);
                instance = m_variable.ClientCertificate;
                instance.OnReadValue = OnRead_ClientCertificate;
                instance.OnSimpleWriteValue = OnWrite_ClientCertificate;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (SessionSecurityDiagnosticsDataType)Write(value);
            }

            return ServiceResult.Good;
        }

        #region SessionId Access Methods
        /// <remarks />
        private ServiceResult OnRead_SessionId(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SessionId;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SessionId(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SessionId = (NodeId)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientUserIdOfSession Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientUserIdOfSession(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientUserIdOfSession;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientUserIdOfSession(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientUserIdOfSession = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientUserIdHistory Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientUserIdHistory(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientUserIdHistory;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientUserIdHistory(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientUserIdHistory = (StringCollection)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region AuthenticationMechanism Access Methods
        /// <remarks />
        private ServiceResult OnRead_AuthenticationMechanism(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.AuthenticationMechanism;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_AuthenticationMechanism(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.AuthenticationMechanism = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Encoding Access Methods
        /// <remarks />
        private ServiceResult OnRead_Encoding(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Encoding;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Encoding(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Encoding = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region TransportProtocol Access Methods
        /// <remarks />
        private ServiceResult OnRead_TransportProtocol(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.TransportProtocol;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_TransportProtocol(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.TransportProtocol = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SecurityMode Access Methods
        /// <remarks />
        private ServiceResult OnRead_SecurityMode(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SecurityMode;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SecurityMode(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SecurityMode = (MessageSecurityMode)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region SecurityPolicyUri Access Methods
        /// <remarks />
        private ServiceResult OnRead_SecurityPolicyUri(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.SecurityPolicyUri;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_SecurityPolicyUri(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.SecurityPolicyUri = (string)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region ClientCertificate Access Methods
        /// <remarks />
        private ServiceResult OnRead_ClientCertificate(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.ClientCertificate;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_ClientCertificate(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.ClientCertificate = (byte[])Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private SessionSecurityDiagnosticsDataType m_value;
        private SessionSecurityDiagnosticsState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region OptionSetState Class
    #if (!OPCUA_EXCLUDE_OptionSetState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class OptionSetState : BaseDataVariableState
    {
        #region Constructors
        /// <remarks />
        public OptionSetState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.OptionSetType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.BaseDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (BitMask != null)
            {
                BitMask.Initialize(context, BitMask_InitializationString);
            }
        }

        #region Initialization String
        private const string BitMask_InitializationString =
           "//////////8XYIkKAgAAAAAABwAAAEJpdE1hc2sBALUtAC4ARLUtAAAAAQEAAAABAAAAAAAAAAEB////" +
           "/wAAAAA=";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAFQAAAE9wdGlvblNldFR5cGVJbnN0YW5jZQEA3ywBAN8s3ywAAAAY////" +
           "/wEB/////wIAAAAXYIkKAgAAAAAADwAAAE9wdGlvblNldFZhbHVlcwEA4CwALgBE4CwAAAAVAQAAAAEA" +
           "AAAAAAAAAQH/////AAAAABdgiQoCAAAAAAAHAAAAQml0TWFzawEAtS0ALgBEtS0AAAABAQAAAAEAAAAA" +
           "AAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<LocalizedText[]> OptionSetValues
        {
            get
            {
                return m_optionSetValues;
            }

            set
            {
                if (!Object.ReferenceEquals(m_optionSetValues, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_optionSetValues = value;
            }
        }

        /// <remarks />
        public PropertyState<bool[]> BitMask
        {
            get
            {
                return m_bitMask;
            }

            set
            {
                if (!Object.ReferenceEquals(m_bitMask, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_bitMask = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_optionSetValues != null)
            {
                children.Add(m_optionSetValues);
            }

            if (m_bitMask != null)
            {
                children.Add(m_bitMask);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.OptionSetValues:
                {
                    if (createOrReplace)
                    {
                        if (OptionSetValues == null)
                        {
                            if (replacement == null)
                            {
                                OptionSetValues = new PropertyState<LocalizedText[]>(this);
                            }
                            else
                            {
                                OptionSetValues = (PropertyState<LocalizedText[]>)replacement;
                            }
                        }
                    }

                    instance = OptionSetValues;
                    break;
                }

                case Opc.Ua.BrowseNames.BitMask:
                {
                    if (createOrReplace)
                    {
                        if (BitMask == null)
                        {
                            if (replacement == null)
                            {
                                BitMask = new PropertyState<bool[]>(this);
                            }
                            else
                            {
                                BitMask = (PropertyState<bool[]>)replacement;
                            }
                        }
                    }

                    instance = BitMask;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<LocalizedText[]> m_optionSetValues;
        private PropertyState<bool[]> m_bitMask;
        #endregion
    }

    #region OptionSetState<T> Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class OptionSetState<T> : OptionSetState
    {
        #region Constructors
        /// <remarks />
        public OptionSetState(NodeState parent) : base(parent)
        {
            Value = default(T);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);

            Value = default(T);
            DataType = TypeInfo.GetDataTypeId(typeof(T));
            ValueRank = TypeInfo.GetValueRank(typeof(T));
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public new T Value
        {
            get
            {
                return CheckTypeBeforeCast<T>(((BaseVariableState)this).Value, true);
            }

            set
            {
                ((BaseVariableState)this).Value = value;
            }
        }
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region SelectionListState Class
    #if (!OPCUA_EXCLUDE_SelectionListState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SelectionListState : BaseDataVariableState
    {
        #region Constructors
        /// <remarks />
        public SelectionListState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.SelectionListType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.BaseDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (SelectionDescriptions != null)
            {
                SelectionDescriptions.Initialize(context, SelectionDescriptions_InitializationString);
            }

            if (RestrictToList != null)
            {
                RestrictToList.Initialize(context, RestrictToList_InitializationString);
            }
        }

        #region Initialization String
        private const string SelectionDescriptions_InitializationString =
           "//////////8XYIkKAgAAAAAAFQAAAFNlbGVjdGlvbkRlc2NyaXB0aW9ucwEA4UQALgBE4UQAAAAVAQAA" +
           "AAEAAAAAAAAAAQH/////AAAAAA==";

        private const string RestrictToList_InitializationString =
           "//////////8VYIkKAgAAAAAADgAAAFJlc3RyaWN0VG9MaXN0AQC4PwAuAES4PwAAAAH/////AQH/////" +
           "AAAAAA==";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGQAAAFNlbGVjdGlvbkxpc3RUeXBlSW5zdGFuY2UBALU/AQC1P7U/AAAA" +
           "GP////8BAf////8DAAAAF2CJCgIAAAAAAAoAAABTZWxlY3Rpb25zAQDgRAAuAETgRAAAABgBAAAAAQAA" +
           "AAAAAAABAf////8AAAAAF2CJCgIAAAAAABUAAABTZWxlY3Rpb25EZXNjcmlwdGlvbnMBAOFEAC4AROFE" +
           "AAAAFQEAAAABAAAAAAAAAAEB/////wAAAAAVYIkKAgAAAAAADgAAAFJlc3RyaWN0VG9MaXN0AQC4PwAu" +
           "AES4PwAAAAH/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<object[]> Selections
        {
            get
            {
                return m_selections;
            }

            set
            {
                if (!Object.ReferenceEquals(m_selections, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_selections = value;
            }
        }

        /// <remarks />
        public PropertyState<LocalizedText[]> SelectionDescriptions
        {
            get
            {
                return m_selectionDescriptions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_selectionDescriptions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_selectionDescriptions = value;
            }
        }

        /// <remarks />
        public PropertyState<bool> RestrictToList
        {
            get
            {
                return m_restrictToList;
            }

            set
            {
                if (!Object.ReferenceEquals(m_restrictToList, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_restrictToList = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_selections != null)
            {
                children.Add(m_selections);
            }

            if (m_selectionDescriptions != null)
            {
                children.Add(m_selectionDescriptions);
            }

            if (m_restrictToList != null)
            {
                children.Add(m_restrictToList);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Selections:
                {
                    if (createOrReplace)
                    {
                        if (Selections == null)
                        {
                            if (replacement == null)
                            {
                                Selections = new PropertyState<object[]>(this);
                            }
                            else
                            {
                                Selections = (PropertyState<object[]>)replacement;
                            }
                        }
                    }

                    instance = Selections;
                    break;
                }

                case Opc.Ua.BrowseNames.SelectionDescriptions:
                {
                    if (createOrReplace)
                    {
                        if (SelectionDescriptions == null)
                        {
                            if (replacement == null)
                            {
                                SelectionDescriptions = new PropertyState<LocalizedText[]>(this);
                            }
                            else
                            {
                                SelectionDescriptions = (PropertyState<LocalizedText[]>)replacement;
                            }
                        }
                    }

                    instance = SelectionDescriptions;
                    break;
                }

                case Opc.Ua.BrowseNames.RestrictToList:
                {
                    if (createOrReplace)
                    {
                        if (RestrictToList == null)
                        {
                            if (replacement == null)
                            {
                                RestrictToList = new PropertyState<bool>(this);
                            }
                            else
                            {
                                RestrictToList = (PropertyState<bool>)replacement;
                            }
                        }
                    }

                    instance = RestrictToList;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<object[]> m_selections;
        private PropertyState<LocalizedText[]> m_selectionDescriptions;
        private PropertyState<bool> m_restrictToList;
        #endregion
    }

    #region SelectionListState<T> Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class SelectionListState<T> : SelectionListState
    {
        #region Constructors
        /// <remarks />
        public SelectionListState(NodeState parent) : base(parent)
        {
            Value = default(T);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);

            Value = default(T);
            DataType = TypeInfo.GetDataTypeId(typeof(T));
            ValueRank = TypeInfo.GetValueRank(typeof(T));
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public new T Value
        {
            get
            {
                return CheckTypeBeforeCast<T>(((BaseVariableState)this).Value, true);
            }

            set
            {
                ((BaseVariableState)this).Value = value;
            }
        }
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region AudioVariableState Class
    #if (!OPCUA_EXCLUDE_AudioVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class AudioVariableState : BaseDataVariableState<byte[]>
    {
        #region Constructors
        /// <remarks />
        public AudioVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.AudioVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.AudioDataType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (ListId != null)
            {
                ListId.Initialize(context, ListId_InitializationString);
            }

            if (AgencyId != null)
            {
                AgencyId.Initialize(context, AgencyId_InitializationString);
            }

            if (VersionId != null)
            {
                VersionId.Initialize(context, VersionId_InitializationString);
            }
        }

        #region Initialization String
        private const string ListId_InitializationString =
           "//////////8VYIkKAgAAAAAABgAAAExpc3RJZAEAREYALgBEREYAAAAM/////wEB/////wAAAAA=";

        private const string AgencyId_InitializationString =
           "//////////8VYIkKAgAAAAAACAAAAEFnZW5jeUlkAQBFRgAuAERFRgAAAAz/////AQH/////AAAAAA==";

        private const string VersionId_InitializationString =
           "//////////8VYIkKAgAAAAAACQAAAFZlcnNpb25JZAEARkYALgBERkYAAAAM/////wEB/////wAAAAA=";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGQAAAEF1ZGlvVmFyaWFibGVUeXBlSW5zdGFuY2UBAEJGAQBCRkJGAAAB" +
           "ALM//////wEB/////wMAAAAVYIkKAgAAAAAABgAAAExpc3RJZAEAREYALgBEREYAAAAM/////wEB////" +
           "/wAAAAAVYIkKAgAAAAAACAAAAEFnZW5jeUlkAQBFRgAuAERFRgAAAAz/////AQH/////AAAAABVgiQoC" +
           "AAAAAAAJAAAAVmVyc2lvbklkAQBGRgAuAERGRgAAAAz/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<string> ListId
        {
            get
            {
                return m_listId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_listId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_listId = value;
            }
        }

        /// <remarks />
        public PropertyState<string> AgencyId
        {
            get
            {
                return m_agencyId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_agencyId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_agencyId = value;
            }
        }

        /// <remarks />
        public PropertyState<string> VersionId
        {
            get
            {
                return m_versionId;
            }

            set
            {
                if (!Object.ReferenceEquals(m_versionId, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_versionId = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_listId != null)
            {
                children.Add(m_listId);
            }

            if (m_agencyId != null)
            {
                children.Add(m_agencyId);
            }

            if (m_versionId != null)
            {
                children.Add(m_versionId);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.ListId:
                {
                    if (createOrReplace)
                    {
                        if (ListId == null)
                        {
                            if (replacement == null)
                            {
                                ListId = new PropertyState<string>(this);
                            }
                            else
                            {
                                ListId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = ListId;
                    break;
                }

                case Opc.Ua.BrowseNames.AgencyId:
                {
                    if (createOrReplace)
                    {
                        if (AgencyId == null)
                        {
                            if (replacement == null)
                            {
                                AgencyId = new PropertyState<string>(this);
                            }
                            else
                            {
                                AgencyId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = AgencyId;
                    break;
                }

                case Opc.Ua.BrowseNames.VersionId:
                {
                    if (createOrReplace)
                    {
                        if (VersionId == null)
                        {
                            if (replacement == null)
                            {
                                VersionId = new PropertyState<string>(this);
                            }
                            else
                            {
                                VersionId = (PropertyState<string>)replacement;
                            }
                        }
                    }

                    instance = VersionId;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<string> m_listId;
        private PropertyState<string> m_agencyId;
        private PropertyState<string> m_versionId;
        #endregion
    }
    #endif
    #endregion

    #region GetMonitoredItemsMethodState Class
    #if (!OPCUA_EXCLUDE_GetMonitoredItemsMethodState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class GetMonitoredItemsMethodState : MethodState
    {
        #region Constructors
        /// <remarks />
        public GetMonitoredItemsMethodState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        public new static NodeState Construct(NodeState parent)
        {
            return new GetMonitoredItemsMethodState(parent);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYYIKBAAAAAAAGwAAAEdldE1vbml0b3JlZEl0ZW1zTWV0aG9kVHlwZQEA5ywALwEA5yzn" +
           "LAAAAQH/////AgAAABdgqQoCAAAAAAAOAAAASW5wdXRBcmd1bWVudHMBAOgsAC4AROgsAACWAQAAAAEA" +
           "KgEBHQAAAA4AAABTdWJzY3JpcHRpb25JZAAH/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAA" +
           "AAAXYKkKAgAAAAAADwAAAE91dHB1dEFyZ3VtZW50cwEA6SwALgBE6SwAAJYCAAAAAQAqAQEgAAAADQAA" +
           "AFNlcnZlckhhbmRsZXMABwEAAAABAAAAAAAAAAABACoBASAAAAANAAAAQ2xpZW50SGFuZGxlcwAHAQAA" +
           "AAEAAAAAAAAAAAEAKAEBAAAAAQAAAAIAAAABAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Event Callbacks
        /// <remarks />
        public GetMonitoredItemsMethodStateMethodCallHandler OnCall;
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        /// <remarks />
        protected override ServiceResult Call(
            ISystemContext _context,
            NodeId _objectId,
            IList<object> _inputArguments,
            IList<object> _outputArguments)
        {
            if (OnCall == null)
            {
                return base.Call(_context, _objectId, _inputArguments, _outputArguments);
            }

            ServiceResult _result = null;

            uint subscriptionId = (uint)_inputArguments[0];

            uint[] serverHandles = (uint[])_outputArguments[0];
            uint[] clientHandles = (uint[])_outputArguments[1];

            if (OnCall != null)
            {
                _result = OnCall(
                    _context,
                    this,
                    _objectId,
                    subscriptionId,
                    ref serverHandles,
                    ref clientHandles);
            }

            _outputArguments[0] = serverHandles;
            _outputArguments[1] = clientHandles;

            return _result;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    /// <remarks />
    /// <exclude />
    public delegate ServiceResult GetMonitoredItemsMethodStateMethodCallHandler(
        ISystemContext _context,
        MethodState _method,
        NodeId _objectId,
        uint subscriptionId,
        ref uint[] serverHandles,
        ref uint[] clientHandles);
    #endif
    #endregion

    #region ResendDataMethodState Class
    #if (!OPCUA_EXCLUDE_ResendDataMethodState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ResendDataMethodState : MethodState
    {
        #region Constructors
        /// <remarks />
        public ResendDataMethodState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        public new static NodeState Construct(NodeState parent)
        {
            return new ResendDataMethodState(parent);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYYIKBAAAAAAAFAAAAFJlc2VuZERhdGFNZXRob2RUeXBlAQBLMgAvAQBLMksyAAABAf//" +
           "//8BAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEATDIALgBETDIAAJYBAAAAAQAqAQEdAAAA" +
           "DgAAAFN1YnNjcmlwdGlvbklkAAf/////AAAAAAABACgBAQAAAAEAAAABAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Event Callbacks
        /// <remarks />
        public ResendDataMethodStateMethodCallHandler OnCall;
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        /// <remarks />
        protected override ServiceResult Call(
            ISystemContext _context,
            NodeId _objectId,
            IList<object> _inputArguments,
            IList<object> _outputArguments)
        {
            if (OnCall == null)
            {
                return base.Call(_context, _objectId, _inputArguments, _outputArguments);
            }

            ServiceResult _result = null;

            uint subscriptionId = (uint)_inputArguments[0];

            if (OnCall != null)
            {
                _result = OnCall(
                    _context,
                    this,
                    _objectId,
                    subscriptionId);
            }

            return _result;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    /// <remarks />
    /// <exclude />
    public delegate ServiceResult ResendDataMethodStateMethodCallHandler(
        ISystemContext _context,
        MethodState _method,
        NodeId _objectId,
        uint subscriptionId);
    #endif
    #endregion

    #region SetSubscriptionDurableMethodState Class
    #if (!OPCUA_EXCLUDE_SetSubscriptionDurableMethodState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class SetSubscriptionDurableMethodState : MethodState
    {
        #region Constructors
        /// <remarks />
        public SetSubscriptionDurableMethodState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        public new static NodeState Construct(NodeState parent)
        {
            return new SetSubscriptionDurableMethodState(parent);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYYIKBAAAAAAAIAAAAFNldFN1YnNjcmlwdGlvbkR1cmFibGVNZXRob2RUeXBlAQDQMQAv" +
           "AQDQMdAxAAABAf////8CAAAAF2CpCgIAAAAAAA4AAABJbnB1dEFyZ3VtZW50cwEA0TEALgBE0TEAAJYC" +
           "AAAAAQAqAQEdAAAADgAAAFN1YnNjcmlwdGlvbklkAAf/////AAAAAAABACoBAR4AAAAPAAAATGlmZXRp" +
           "bWVJbkhvdXJzAAf/////AAAAAAABACgBAQAAAAEAAAACAAAAAQH/////AAAAABdgqQoCAAAAAAAPAAAA" +
           "T3V0cHV0QXJndW1lbnRzAQDSMQAuAETSMQAAlgEAAAABACoBASUAAAAWAAAAUmV2aXNlZExpZmV0aW1l" +
           "SW5Ib3VycwAH/////wAAAAAAAQAoAQEAAAABAAAAAQAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Event Callbacks
        /// <remarks />
        public SetSubscriptionDurableMethodStateMethodCallHandler OnCall;
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        /// <remarks />
        protected override ServiceResult Call(
            ISystemContext _context,
            NodeId _objectId,
            IList<object> _inputArguments,
            IList<object> _outputArguments)
        {
            if (OnCall == null)
            {
                return base.Call(_context, _objectId, _inputArguments, _outputArguments);
            }

            ServiceResult _result = null;

            uint subscriptionId = (uint)_inputArguments[0];
            uint lifetimeInHours = (uint)_inputArguments[1];

            uint revisedLifetimeInHours = (uint)_outputArguments[0];

            if (OnCall != null)
            {
                _result = OnCall(
                    _context,
                    this,
                    _objectId,
                    subscriptionId,
                    lifetimeInHours,
                    ref revisedLifetimeInHours);
            }

            _outputArguments[0] = revisedLifetimeInHours;

            return _result;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    /// <remarks />
    /// <exclude />
    public delegate ServiceResult SetSubscriptionDurableMethodStateMethodCallHandler(
        ISystemContext _context,
        MethodState _method,
        NodeId _objectId,
        uint subscriptionId,
        uint lifetimeInHours,
        ref uint revisedLifetimeInHours);
    #endif
    #endregion

    #region RequestServerStateChangeMethodState Class
    #if (!OPCUA_EXCLUDE_RequestServerStateChangeMethodState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class RequestServerStateChangeMethodState : MethodState
    {
        #region Constructors
        /// <remarks />
        public RequestServerStateChangeMethodState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        public new static NodeState Construct(NodeState parent)
        {
            return new RequestServerStateChangeMethodState(parent);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYYIKBAAAAAAAIgAAAFJlcXVlc3RTZXJ2ZXJTdGF0ZUNoYW5nZU1ldGhvZFR5cGUBAFgy" +
           "AC8BAFgyWDIAAAEB/////wEAAAAXYKkKAgAAAAAADgAAAElucHV0QXJndW1lbnRzAQBZMgAuAERZMgAA" +
           "lgUAAAABACoBARYAAAAFAAAAU3RhdGUBAFQD/////wAAAAAAAQAqAQEiAAAAEwAAAEVzdGltYXRlZFJl" +
           "dHVyblRpbWUADf////8AAAAAAAEAKgEBIgAAABMAAABTZWNvbmRzVGlsbFNodXRkb3duAAf/////AAAA" +
           "AAABACoBARUAAAAGAAAAUmVhc29uABX/////AAAAAAABACoBARYAAAAHAAAAUmVzdGFydAAB/////wAA" +
           "AAAAAQAoAQEAAAABAAAABQAAAAEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Event Callbacks
        /// <remarks />
        public RequestServerStateChangeMethodStateMethodCallHandler OnCall;
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        /// <remarks />
        protected override ServiceResult Call(
            ISystemContext _context,
            NodeId _objectId,
            IList<object> _inputArguments,
            IList<object> _outputArguments)
        {
            if (OnCall == null)
            {
                return base.Call(_context, _objectId, _inputArguments, _outputArguments);
            }

            ServiceResult _result = null;

            ServerState state = (ServerState)_inputArguments[0];
            DateTime estimatedReturnTime = (DateTime)_inputArguments[1];
            uint secondsTillShutdown = (uint)_inputArguments[2];
            LocalizedText reason = (LocalizedText)_inputArguments[3];
            bool restart = (bool)_inputArguments[4];

            if (OnCall != null)
            {
                _result = OnCall(
                    _context,
                    this,
                    _objectId,
                    state,
                    estimatedReturnTime,
                    secondsTillShutdown,
                    reason,
                    restart);
            }

            return _result;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    /// <remarks />
    /// <exclude />
    public delegate ServiceResult RequestServerStateChangeMethodStateMethodCallHandler(
        ISystemContext _context,
        MethodState _method,
        NodeId _objectId,
        ServerState state,
        DateTime estimatedReturnTime,
        uint secondsTillShutdown,
        LocalizedText reason,
        bool restart);
    #endif
    #endregion

    #region StateMachineState Class
    #if (!OPCUA_EXCLUDE_StateMachineState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateMachineState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public StateMachineState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.StateMachineType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (LastTransition != null)
            {
                LastTransition.Initialize(context, LastTransition_InitializationString);
            }
        }

        #region Initialization String
        private const string LastTransition_InitializationString =
           "//////////8VYIkKAgAAAAAADgAAAExhc3RUcmFuc2l0aW9uAQDSCgAvAQDKCtIKAAAAFf////8BAf//" +
           "//8BAAAAFWCJCgIAAAAAAAIAAABJZAEAjA4ALgBEjA4AAAAY/////wEB/////wAAAAA=";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAGAAAAFN0YXRlTWFjaGluZVR5cGVJbnN0YW5jZQEA+wgBAPsI+wgAAP//" +
           "//8CAAAAFWCJCgIAAAAAAAwAAABDdXJyZW50U3RhdGUBANEKAC8BAMMK0QoAAAAV/////wEB/////wEA" +
           "AAAVYIkKAgAAAAAAAgAAAElkAQCIDgAuAESIDgAAABj/////AQH/////AAAAABVgiQoCAAAAAAAOAAAA" +
           "TGFzdFRyYW5zaXRpb24BANIKAC8BAMoK0goAAAAV/////wEB/////wEAAAAVYIkKAgAAAAAAAgAAAElk" +
           "AQCMDgAuAESMDgAAABj/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public StateVariableState CurrentState
        {
            get
            {
                return m_currentState;
            }

            set
            {
                if (!Object.ReferenceEquals(m_currentState, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_currentState = value;
            }
        }

        /// <remarks />
        public TransitionVariableState LastTransition
        {
            get
            {
                return m_lastTransition;
            }

            set
            {
                if (!Object.ReferenceEquals(m_lastTransition, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_lastTransition = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_currentState != null)
            {
                children.Add(m_currentState);
            }

            if (m_lastTransition != null)
            {
                children.Add(m_lastTransition);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.CurrentState:
                {
                    if (createOrReplace)
                    {
                        if (CurrentState == null)
                        {
                            if (replacement == null)
                            {
                                CurrentState = new StateVariableState(this);
                            }
                            else
                            {
                                CurrentState = (StateVariableState)replacement;
                            }
                        }
                    }

                    instance = CurrentState;
                    break;
                }

                case Opc.Ua.BrowseNames.LastTransition:
                {
                    if (createOrReplace)
                    {
                        if (LastTransition == null)
                        {
                            if (replacement == null)
                            {
                                LastTransition = new TransitionVariableState(this);
                            }
                            else
                            {
                                LastTransition = (TransitionVariableState)replacement;
                            }
                        }
                    }

                    instance = LastTransition;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private StateVariableState m_currentState;
        private TransitionVariableState m_lastTransition;
        #endregion
    }
    #endif
    #endregion

    #region StateVariableState Class
    #if (!OPCUA_EXCLUDE_StateVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateVariableState : BaseDataVariableState<LocalizedText>
    {
        #region Constructors
        /// <remarks />
        public StateVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.StateVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (Name != null)
            {
                Name.Initialize(context, Name_InitializationString);
            }

            if (Number != null)
            {
                Number.Initialize(context, Number_InitializationString);
            }

            if (EffectiveDisplayName != null)
            {
                EffectiveDisplayName.Initialize(context, EffectiveDisplayName_InitializationString);
            }
        }

        #region Initialization String
        private const string Name_InitializationString =
           "//////////8VYIkKAgAAAAAABAAAAE5hbWUBAMUKAC4ARMUKAAAAFP////8BAf////8AAAAA";

        private const string Number_InitializationString =
           "//////////8VYIkKAgAAAAAABgAAAE51bWJlcgEAxgoALgBExgoAAAAH/////wEB/////wAAAAA=";

        private const string EffectiveDisplayName_InitializationString =
           "//////////8VYIkKAgAAAAAAFAAAAEVmZmVjdGl2ZURpc3BsYXlOYW1lAQDHCgAuAETHCgAAABX/////" +
           "AQH/////AAAAAA==";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGQAAAFN0YXRlVmFyaWFibGVUeXBlSW5zdGFuY2UBAMMKAQDDCsMKAAAA" +
           "Ff////8BAf////8EAAAAFWCJCgIAAAAAAAIAAABJZAEAxAoALgBExAoAAAAY/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAABAAAAE5hbWUBAMUKAC4ARMUKAAAAFP////8BAf////8AAAAAFWCJCgIAAAAAAAYAAABO" +
           "dW1iZXIBAMYKAC4ARMYKAAAAB/////8BAf////8AAAAAFWCJCgIAAAAAABQAAABFZmZlY3RpdmVEaXNw" +
           "bGF5TmFtZQEAxwoALgBExwoAAAAV/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState Id
        {
            get
            {
                return m_id;
            }

            set
            {
                if (!Object.ReferenceEquals(m_id, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_id = value;
            }
        }

        /// <remarks />
        public PropertyState<QualifiedName> Name
        {
            get
            {
                return m_name;
            }

            set
            {
                if (!Object.ReferenceEquals(m_name, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_name = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> Number
        {
            get
            {
                return m_number;
            }

            set
            {
                if (!Object.ReferenceEquals(m_number, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_number = value;
            }
        }

        /// <remarks />
        public PropertyState<LocalizedText> EffectiveDisplayName
        {
            get
            {
                return m_effectiveDisplayName;
            }

            set
            {
                if (!Object.ReferenceEquals(m_effectiveDisplayName, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_effectiveDisplayName = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_id != null)
            {
                children.Add(m_id);
            }

            if (m_name != null)
            {
                children.Add(m_name);
            }

            if (m_number != null)
            {
                children.Add(m_number);
            }

            if (m_effectiveDisplayName != null)
            {
                children.Add(m_effectiveDisplayName);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Id:
                {
                    if (createOrReplace)
                    {
                        if (Id == null)
                        {
                            if (replacement == null)
                            {
                                Id = new PropertyState(this);
                            }
                            else
                            {
                                Id = (PropertyState)replacement;
                            }
                        }
                    }

                    instance = Id;
                    break;
                }

                case Opc.Ua.BrowseNames.Name:
                {
                    if (createOrReplace)
                    {
                        if (Name == null)
                        {
                            if (replacement == null)
                            {
                                Name = new PropertyState<QualifiedName>(this);
                            }
                            else
                            {
                                Name = (PropertyState<QualifiedName>)replacement;
                            }
                        }
                    }

                    instance = Name;
                    break;
                }

                case Opc.Ua.BrowseNames.Number:
                {
                    if (createOrReplace)
                    {
                        if (Number == null)
                        {
                            if (replacement == null)
                            {
                                Number = new PropertyState<uint>(this);
                            }
                            else
                            {
                                Number = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = Number;
                    break;
                }

                case Opc.Ua.BrowseNames.EffectiveDisplayName:
                {
                    if (createOrReplace)
                    {
                        if (EffectiveDisplayName == null)
                        {
                            if (replacement == null)
                            {
                                EffectiveDisplayName = new PropertyState<LocalizedText>(this);
                            }
                            else
                            {
                                EffectiveDisplayName = (PropertyState<LocalizedText>)replacement;
                            }
                        }
                    }

                    instance = EffectiveDisplayName;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState m_id;
        private PropertyState<QualifiedName> m_name;
        private PropertyState<uint> m_number;
        private PropertyState<LocalizedText> m_effectiveDisplayName;
        #endregion
    }
    #endif
    #endregion

    #region TransitionVariableState Class
    #if (!OPCUA_EXCLUDE_TransitionVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class TransitionVariableState : BaseDataVariableState<LocalizedText>
    {
        #region Constructors
        /// <remarks />
        public TransitionVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.TransitionVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (Name != null)
            {
                Name.Initialize(context, Name_InitializationString);
            }

            if (Number != null)
            {
                Number.Initialize(context, Number_InitializationString);
            }

            if (TransitionTime != null)
            {
                TransitionTime.Initialize(context, TransitionTime_InitializationString);
            }

            if (EffectiveTransitionTime != null)
            {
                EffectiveTransitionTime.Initialize(context, EffectiveTransitionTime_InitializationString);
            }
        }

        #region Initialization String
        private const string Name_InitializationString =
           "//////////8VYIkKAgAAAAAABAAAAE5hbWUBAMwKAC4ARMwKAAAAFP////8BAf////8AAAAA";

        private const string Number_InitializationString =
           "//////////8VYIkKAgAAAAAABgAAAE51bWJlcgEAzQoALgBEzQoAAAAH/////wEB/////wAAAAA=";

        private const string TransitionTime_InitializationString =
           "//////////8VYIkKAgAAAAAADgAAAFRyYW5zaXRpb25UaW1lAQDOCgAuAETOCgAAAQAmAf////8BAf//" +
           "//8AAAAA";

        private const string EffectiveTransitionTime_InitializationString =
           "//////////8VYIkKAgAAAAAAFwAAAEVmZmVjdGl2ZVRyYW5zaXRpb25UaW1lAQDALAAuAETALAAAAQAm" +
           "Af////8BAf////8AAAAA";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAHgAAAFRyYW5zaXRpb25WYXJpYWJsZVR5cGVJbnN0YW5jZQEAygoBAMoK" +
           "ygoAAAAV/////wEB/////wUAAAAVYIkKAgAAAAAAAgAAAElkAQDLCgAuAETLCgAAABj/////AQH/////" +
           "AAAAABVgiQoCAAAAAAAEAAAATmFtZQEAzAoALgBEzAoAAAAU/////wEB/////wAAAAAVYIkKAgAAAAAA" +
           "BgAAAE51bWJlcgEAzQoALgBEzQoAAAAH/////wEB/////wAAAAAVYIkKAgAAAAAADgAAAFRyYW5zaXRp" +
           "b25UaW1lAQDOCgAuAETOCgAAAQAmAf////8BAf////8AAAAAFWCJCgIAAAAAABcAAABFZmZlY3RpdmVU" +
           "cmFuc2l0aW9uVGltZQEAwCwALgBEwCwAAAEAJgH/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState Id
        {
            get
            {
                return m_id;
            }

            set
            {
                if (!Object.ReferenceEquals(m_id, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_id = value;
            }
        }

        /// <remarks />
        public PropertyState<QualifiedName> Name
        {
            get
            {
                return m_name;
            }

            set
            {
                if (!Object.ReferenceEquals(m_name, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_name = value;
            }
        }

        /// <remarks />
        public PropertyState<uint> Number
        {
            get
            {
                return m_number;
            }

            set
            {
                if (!Object.ReferenceEquals(m_number, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_number = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> TransitionTime
        {
            get
            {
                return m_transitionTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transitionTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transitionTime = value;
            }
        }

        /// <remarks />
        public PropertyState<DateTime> EffectiveTransitionTime
        {
            get
            {
                return m_effectiveTransitionTime;
            }

            set
            {
                if (!Object.ReferenceEquals(m_effectiveTransitionTime, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_effectiveTransitionTime = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_id != null)
            {
                children.Add(m_id);
            }

            if (m_name != null)
            {
                children.Add(m_name);
            }

            if (m_number != null)
            {
                children.Add(m_number);
            }

            if (m_transitionTime != null)
            {
                children.Add(m_transitionTime);
            }

            if (m_effectiveTransitionTime != null)
            {
                children.Add(m_effectiveTransitionTime);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Id:
                {
                    if (createOrReplace)
                    {
                        if (Id == null)
                        {
                            if (replacement == null)
                            {
                                Id = new PropertyState(this);
                            }
                            else
                            {
                                Id = (PropertyState)replacement;
                            }
                        }
                    }

                    instance = Id;
                    break;
                }

                case Opc.Ua.BrowseNames.Name:
                {
                    if (createOrReplace)
                    {
                        if (Name == null)
                        {
                            if (replacement == null)
                            {
                                Name = new PropertyState<QualifiedName>(this);
                            }
                            else
                            {
                                Name = (PropertyState<QualifiedName>)replacement;
                            }
                        }
                    }

                    instance = Name;
                    break;
                }

                case Opc.Ua.BrowseNames.Number:
                {
                    if (createOrReplace)
                    {
                        if (Number == null)
                        {
                            if (replacement == null)
                            {
                                Number = new PropertyState<uint>(this);
                            }
                            else
                            {
                                Number = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = Number;
                    break;
                }

                case Opc.Ua.BrowseNames.TransitionTime:
                {
                    if (createOrReplace)
                    {
                        if (TransitionTime == null)
                        {
                            if (replacement == null)
                            {
                                TransitionTime = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                TransitionTime = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = TransitionTime;
                    break;
                }

                case Opc.Ua.BrowseNames.EffectiveTransitionTime:
                {
                    if (createOrReplace)
                    {
                        if (EffectiveTransitionTime == null)
                        {
                            if (replacement == null)
                            {
                                EffectiveTransitionTime = new PropertyState<DateTime>(this);
                            }
                            else
                            {
                                EffectiveTransitionTime = (PropertyState<DateTime>)replacement;
                            }
                        }
                    }

                    instance = EffectiveTransitionTime;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState m_id;
        private PropertyState<QualifiedName> m_name;
        private PropertyState<uint> m_number;
        private PropertyState<DateTime> m_transitionTime;
        private PropertyState<DateTime> m_effectiveTransitionTime;
        #endregion
    }
    #endif
    #endregion

    #region FiniteStateMachineState Class
    #if (!OPCUA_EXCLUDE_FiniteStateMachineState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class FiniteStateMachineState : StateMachineState
    {
        #region Constructors
        /// <remarks />
        public FiniteStateMachineState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.FiniteStateMachineType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (LastTransition != null)
            {
                LastTransition.Initialize(context, LastTransition_InitializationString);
            }

            if (AvailableStates != null)
            {
                AvailableStates.Initialize(context, AvailableStates_InitializationString);
            }

            if (AvailableTransitions != null)
            {
                AvailableTransitions.Initialize(context, AvailableTransitions_InitializationString);
            }
        }

        #region Initialization String
        private const string LastTransition_InitializationString =
           "//////////8VYIkKAgAAAAAADgAAAExhc3RUcmFuc2l0aW9uAQDVCgAvAQDPCtUKAAAAFf////8BAf//" +
           "//8BAAAAFWCJCgIAAAAAAAIAAABJZAEAlA4ALgBElA4AAAAR/////wEB/////wAAAAA=";

        private const string AvailableStates_InitializationString =
           "//////////8XYIkKAgAAAAAADwAAAEF2YWlsYWJsZVN0YXRlcwEA40QALwA/40QAAAARAQAAAAEAAAAA" +
           "AAAAAQH/////AAAAAA==";

        private const string AvailableTransitions_InitializationString =
           "//////////8XYIkKAgAAAAAAFAAAAEF2YWlsYWJsZVRyYW5zaXRpb25zAQDkRAAvAD/kRAAAABEBAAAA" +
           "AQAAAAAAAAABAf////8AAAAA";

        private const string InitializationString =
           "//////////8EYIACAQAAAAAAHgAAAEZpbml0ZVN0YXRlTWFjaGluZVR5cGVJbnN0YW5jZQEA0woBANMK" +
           "0woAAP////8EAAAAFWCJCgIAAAAAAAwAAABDdXJyZW50U3RhdGUBANQKAC8BAMgK1AoAAAAV/////wEB" +
           "/////wEAAAAVYIkKAgAAAAAAAgAAAElkAQCQDgAuAESQDgAAABH/////AQH/////AAAAABVgiQoCAAAA" +
           "AAAOAAAATGFzdFRyYW5zaXRpb24BANUKAC8BAM8K1QoAAAAV/////wEB/////wEAAAAVYIkKAgAAAAAA" +
           "AgAAAElkAQCUDgAuAESUDgAAABH/////AQH/////AAAAABdgiQoCAAAAAAAPAAAAQXZhaWxhYmxlU3Rh" +
           "dGVzAQDjRAAvAD/jRAAAABEBAAAAAQAAAAAAAAABAf////8AAAAAF2CJCgIAAAAAABQAAABBdmFpbGFi" +
           "bGVUcmFuc2l0aW9ucwEA5EQALwA/5EQAAAARAQAAAAEAAAAAAAAAAQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public new FiniteStateVariableState CurrentState
        {
            get { return (FiniteStateVariableState)base.CurrentState; }
            set { base.CurrentState = value; }
        }

        /// <remarks />
        public new FiniteTransitionVariableState LastTransition
        {
            get { return (FiniteTransitionVariableState)base.LastTransition; }
            set { base.LastTransition = value; }
        }

        /// <remarks />
        public BaseDataVariableState<NodeId[]> AvailableStates
        {
            get
            {
                return m_availableStates;
            }

            set
            {
                if (!Object.ReferenceEquals(m_availableStates, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_availableStates = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<NodeId[]> AvailableTransitions
        {
            get
            {
                return m_availableTransitions;
            }

            set
            {
                if (!Object.ReferenceEquals(m_availableTransitions, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_availableTransitions = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_availableStates != null)
            {
                children.Add(m_availableStates);
            }

            if (m_availableTransitions != null)
            {
                children.Add(m_availableTransitions);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.CurrentState:
                {
                    if (createOrReplace)
                    {
                        if (CurrentState == null)
                        {
                            if (replacement == null)
                            {
                                CurrentState = new FiniteStateVariableState(this);
                            }
                            else
                            {
                                CurrentState = (FiniteStateVariableState)replacement;
                            }
                        }
                    }

                    instance = CurrentState;
                    break;
                }

                case Opc.Ua.BrowseNames.LastTransition:
                {
                    if (createOrReplace)
                    {
                        if (LastTransition == null)
                        {
                            if (replacement == null)
                            {
                                LastTransition = new FiniteTransitionVariableState(this);
                            }
                            else
                            {
                                LastTransition = (FiniteTransitionVariableState)replacement;
                            }
                        }
                    }

                    instance = LastTransition;
                    break;
                }

                case Opc.Ua.BrowseNames.AvailableStates:
                {
                    if (createOrReplace)
                    {
                        if (AvailableStates == null)
                        {
                            if (replacement == null)
                            {
                                AvailableStates = new BaseDataVariableState<NodeId[]>(this);
                            }
                            else
                            {
                                AvailableStates = (BaseDataVariableState<NodeId[]>)replacement;
                            }
                        }
                    }

                    instance = AvailableStates;
                    break;
                }

                case Opc.Ua.BrowseNames.AvailableTransitions:
                {
                    if (createOrReplace)
                    {
                        if (AvailableTransitions == null)
                        {
                            if (replacement == null)
                            {
                                AvailableTransitions = new BaseDataVariableState<NodeId[]>(this);
                            }
                            else
                            {
                                AvailableTransitions = (BaseDataVariableState<NodeId[]>)replacement;
                            }
                        }
                    }

                    instance = AvailableTransitions;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<NodeId[]> m_availableStates;
        private BaseDataVariableState<NodeId[]> m_availableTransitions;
        #endregion
    }
    #endif
    #endregion

    #region FiniteStateVariableState Class
    #if (!OPCUA_EXCLUDE_FiniteStateVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class FiniteStateVariableState : StateVariableState
    {
        #region Constructors
        /// <remarks />
        public FiniteStateVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.FiniteStateVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAHwAAAEZpbml0ZVN0YXRlVmFyaWFibGVUeXBlSW5zdGFuY2UBAMgKAQDI" +
           "CsgKAAAAFf////8BAf////8BAAAAFWCJCgIAAAAAAAIAAABJZAEAyQoALgBEyQoAAAAR/////wEB////" +
           "/wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public new PropertyState<NodeId> Id
        {
            get { return (PropertyState<NodeId>)base.Id; }
            set { base.Id = value; }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Id:
                {
                    if (createOrReplace)
                    {
                        if (Id == null)
                        {
                            if (replacement == null)
                            {
                                Id = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                Id = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = Id;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region FiniteTransitionVariableState Class
    #if (!OPCUA_EXCLUDE_FiniteTransitionVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class FiniteTransitionVariableState : TransitionVariableState
    {
        #region Constructors
        /// <remarks />
        public FiniteTransitionVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.FiniteTransitionVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJAAAAEZpbml0ZVRyYW5zaXRpb25WYXJpYWJsZVR5cGVJbnN0YW5jZQEA" +
           "zwoBAM8KzwoAAAAV/////wEB/////wEAAAAVYIkKAgAAAAAAAgAAAElkAQDQCgAuAETQCgAAABH/////" +
           "AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public new PropertyState<NodeId> Id
        {
            get { return (PropertyState<NodeId>)base.Id; }
            set { base.Id = value; }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Id:
                {
                    if (createOrReplace)
                    {
                        if (Id == null)
                        {
                            if (replacement == null)
                            {
                                Id = new PropertyState<NodeId>(this);
                            }
                            else
                            {
                                Id = (PropertyState<NodeId>)replacement;
                            }
                        }
                    }

                    instance = Id;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region StateMachineStateState Class
    #if (!OPCUA_EXCLUDE_StateMachineStateState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateMachineStateState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public StateMachineStateState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.StateType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAEQAAAFN0YXRlVHlwZUluc3RhbmNlAQADCQEAAwkDCQAA/////wEAAAAV" +
           "YIkKAgAAAAAACwAAAFN0YXRlTnVtYmVyAQAECQAuAEQECQAAAAf/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<uint> StateNumber
        {
            get
            {
                return m_stateNumber;
            }

            set
            {
                if (!Object.ReferenceEquals(m_stateNumber, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_stateNumber = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_stateNumber != null)
            {
                children.Add(m_stateNumber);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.StateNumber:
                {
                    if (createOrReplace)
                    {
                        if (StateNumber == null)
                        {
                            if (replacement == null)
                            {
                                StateNumber = new PropertyState<uint>(this);
                            }
                            else
                            {
                                StateNumber = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = StateNumber;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<uint> m_stateNumber;
        #endregion
    }
    #endif
    #endregion

    #region StateMachineInitialStateState Class
    #if (!OPCUA_EXCLUDE_StateMachineInitialStateState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateMachineInitialStateState : StateMachineStateState
    {
        #region Constructors
        /// <remarks />
        public StateMachineInitialStateState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.InitialStateType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAGAAAAEluaXRpYWxTdGF0ZVR5cGVJbnN0YW5jZQEABQkBAAUJBQkAAP//" +
           "//8BAAAAFWCJCgIAAAAAAAsAAABTdGF0ZU51bWJlcgEAmA4ALgBEmA4AAAAH/////wEB/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region StateMachineTransitionState Class
    #if (!OPCUA_EXCLUDE_StateMachineTransitionState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateMachineTransitionState : BaseObjectState
    {
        #region Constructors
        /// <remarks />
        public StateMachineTransitionState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.TransitionType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFgAAAFRyYW5zaXRpb25UeXBlSW5zdGFuY2UBAAYJAQAGCQYJAAD/////" +
           "AQAAABVgiQoCAAAAAAAQAAAAVHJhbnNpdGlvbk51bWJlcgEACAkALgBECAkAAAAH/////wEB/////wAA" +
           "AAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<uint> TransitionNumber
        {
            get
            {
                return m_transitionNumber;
            }

            set
            {
                if (!Object.ReferenceEquals(m_transitionNumber, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_transitionNumber = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_transitionNumber != null)
            {
                children.Add(m_transitionNumber);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.TransitionNumber:
                {
                    if (createOrReplace)
                    {
                        if (TransitionNumber == null)
                        {
                            if (replacement == null)
                            {
                                TransitionNumber = new PropertyState<uint>(this);
                            }
                            else
                            {
                                TransitionNumber = (PropertyState<uint>)replacement;
                            }
                        }
                    }

                    instance = TransitionNumber;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<uint> m_transitionNumber;
        #endregion
    }
    #endif
    #endregion

    #region StateMachineChoiceStateTypeState Class
    #if (!OPCUA_EXCLUDE_StateMachineChoiceStateTypeState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class StateMachineChoiceStateTypeState : StateMachineStateState
    {
        #region Constructors
        /// <remarks />
        public StateMachineChoiceStateTypeState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.ObjectTypes.ChoiceStateType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8EYIACAQAAAAAAFwAAAENob2ljZVN0YXRlVHlwZUluc3RhbmNlAQAFOwEABTsFOwAA////" +
           "/wEAAAAVYIkKAgAAAAAACwAAAFN0YXRlTnVtYmVyAQAGOwAuAEQGOwAAAAf/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region GuardVariableState Class
    #if (!OPCUA_EXCLUDE_GuardVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class GuardVariableState : BaseDataVariableState<LocalizedText>
    {
        #region Constructors
        /// <remarks />
        public GuardVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.GuardVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGQAAAEd1YXJkVmFyaWFibGVUeXBlSW5zdGFuY2UBAAk7AQAJOwk7AAAA" +
           "Ff////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ExpressionGuardVariableState Class
    #if (!OPCUA_EXCLUDE_ExpressionGuardVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ExpressionGuardVariableState : GuardVariableState
    {
        #region Constructors
        /// <remarks />
        public ExpressionGuardVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ExpressionGuardVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAIwAAAEV4cHJlc3Npb25HdWFyZFZhcmlhYmxlVHlwZUluc3RhbmNlAQAY" +
           "OwEAGDsYOwAAABX/////AQH/////AQAAABVgiQoCAAAAAAAKAAAARXhwcmVzc2lvbgEAGTsALgBEGTsA" +
           "AAEASgL/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<ContentFilter> Expression
        {
            get
            {
                return m_expression;
            }

            set
            {
                if (!Object.ReferenceEquals(m_expression, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_expression = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_expression != null)
            {
                children.Add(m_expression);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Expression:
                {
                    if (createOrReplace)
                    {
                        if (Expression == null)
                        {
                            if (replacement == null)
                            {
                                Expression = new PropertyState<ContentFilter>(this);
                            }
                            else
                            {
                                Expression = (PropertyState<ContentFilter>)replacement;
                            }
                        }
                    }

                    instance = Expression;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<ContentFilter> m_expression;
        #endregion
    }
    #endif
    #endregion

    #region ElseGuardVariableState Class
    #if (!OPCUA_EXCLUDE_ElseGuardVariableState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ElseGuardVariableState : GuardVariableState
    {
        #region Constructors
        /// <remarks />
        public ElseGuardVariableState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ElseGuardVariableType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.LocalizedText, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAHQAAAEVsc2VHdWFyZFZhcmlhYmxlVHlwZUluc3RhbmNlAQDVOwEA1TvV" +
           "OwAAABX/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        #endregion

        #region Overridden Methods
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region RationalNumberState Class
    #if (!OPCUA_EXCLUDE_RationalNumberState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class RationalNumberState : BaseDataVariableState<RationalNumber>
    {
        #region Constructors
        /// <remarks />
        public RationalNumberState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.RationalNumberType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.RationalNumber, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGgAAAFJhdGlvbmFsTnVtYmVyVHlwZUluc3RhbmNlAQAtRQEALUUtRQAA" +
           "AQB2Sf////8BAf////8CAAAAFWCJCgIAAAAAAAkAAABOdW1lcmF0b3IBADBFAC8APzBFAAAABv////8B" +
           "Af////8AAAAAFWCJCgIAAAAAAAsAAABEZW5vbWluYXRvcgEAMUUALwA/MUUAAAAH/////wEB/////wAA" +
           "AAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<int> Numerator
        {
            get
            {
                return m_numerator;
            }

            set
            {
                if (!Object.ReferenceEquals(m_numerator, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_numerator = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<uint> Denominator
        {
            get
            {
                return m_denominator;
            }

            set
            {
                if (!Object.ReferenceEquals(m_denominator, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_denominator = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_numerator != null)
            {
                children.Add(m_numerator);
            }

            if (m_denominator != null)
            {
                children.Add(m_denominator);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.Numerator:
                {
                    if (createOrReplace)
                    {
                        if (Numerator == null)
                        {
                            if (replacement == null)
                            {
                                Numerator = new BaseDataVariableState<int>(this);
                            }
                            else
                            {
                                Numerator = (BaseDataVariableState<int>)replacement;
                            }
                        }
                    }

                    instance = Numerator;
                    break;
                }

                case Opc.Ua.BrowseNames.Denominator:
                {
                    if (createOrReplace)
                    {
                        if (Denominator == null)
                        {
                            if (replacement == null)
                            {
                                Denominator = new BaseDataVariableState<uint>(this);
                            }
                            else
                            {
                                Denominator = (BaseDataVariableState<uint>)replacement;
                            }
                        }
                    }

                    instance = Denominator;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<int> m_numerator;
        private BaseDataVariableState<uint> m_denominator;
        #endregion
    }

    #region RationalNumberValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class RationalNumberValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public RationalNumberValue(RationalNumberState variable, RationalNumber value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new RationalNumber();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public RationalNumberState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public RationalNumber Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(RationalNumberState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.Numerator;
                instance.OnReadValue = OnRead_Numerator;
                instance.OnSimpleWriteValue = OnWrite_Numerator;
                updateList.Add(instance);
                instance = m_variable.Denominator;
                instance.OnReadValue = OnRead_Denominator;
                instance.OnSimpleWriteValue = OnWrite_Denominator;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (RationalNumber)Write(value);
            }

            return ServiceResult.Good;
        }

        #region Numerator Access Methods
        /// <remarks />
        private ServiceResult OnRead_Numerator(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Numerator;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Numerator(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Numerator = (int)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Denominator Access Methods
        /// <remarks />
        private ServiceResult OnRead_Denominator(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Denominator;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Denominator(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Denominator = (uint)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private RationalNumber m_value;
        private RationalNumberState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region VectorState Class
    #if (!OPCUA_EXCLUDE_VectorState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class VectorState : BaseDataVariableState<Vector>
    {
        #region Constructors
        /// <remarks />
        public VectorState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.VectorType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.Vector, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (VectorUnit != null)
            {
                VectorUnit.Initialize(context, VectorUnit_InitializationString);
            }
        }

        #region Initialization String
        private const string VectorUnit_InitializationString =
           "//////////8VYIkKAgAAAAAACgAAAFZlY3RvclVuaXQBADNFAC4ARDNFAAABAHcD/////wEB/////wAA" +
           "AAA=";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAEgAAAFZlY3RvclR5cGVJbnN0YW5jZQEAMkUBADJFMkUAAAEAd0n/////" +
           "AQH/////AQAAABVgiQoCAAAAAAAKAAAAVmVjdG9yVW5pdAEAM0UALgBEM0UAAAEAdwP/////AQH/////" +
           "AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<EUInformation> VectorUnit
        {
            get
            {
                return m_vectorUnit;
            }

            set
            {
                if (!Object.ReferenceEquals(m_vectorUnit, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_vectorUnit = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_vectorUnit != null)
            {
                children.Add(m_vectorUnit);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.VectorUnit:
                {
                    if (createOrReplace)
                    {
                        if (VectorUnit == null)
                        {
                            if (replacement == null)
                            {
                                VectorUnit = new PropertyState<EUInformation>(this);
                            }
                            else
                            {
                                VectorUnit = (PropertyState<EUInformation>)replacement;
                            }
                        }
                    }

                    instance = VectorUnit;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<EUInformation> m_vectorUnit;
        #endregion
    }
    #endif
    #endregion

    #region ThreeDVectorState Class
    #if (!OPCUA_EXCLUDE_ThreeDVectorState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ThreeDVectorState : VectorState
    {
        #region Constructors
        /// <remarks />
        public ThreeDVectorState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ThreeDVectorType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.ThreeDVector, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAGAAAAFRocmVlRFZlY3RvclR5cGVJbnN0YW5jZQEANEUBADRFNEUAAAEA" +
           "eEn/////AQH/////AwAAABVgiQoCAAAAAAABAAAAWAEAUUkALwA/UUkAAAAL/////wEB/////wAAAAAV" +
           "YIkKAgAAAAAAAQAAAFkBAFJJAC8AP1JJAAAAC/////8BAf////8AAAAAFWCJCgIAAAAAAAEAAABaAQBT" +
           "SQAvAD9TSQAAAAv/////AQH/////AAAAAA==";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<double> X
        {
            get
            {
                return m_x;
            }

            set
            {
                if (!Object.ReferenceEquals(m_x, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_x = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> Y
        {
            get
            {
                return m_y;
            }

            set
            {
                if (!Object.ReferenceEquals(m_y, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_y = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> Z
        {
            get
            {
                return m_z;
            }

            set
            {
                if (!Object.ReferenceEquals(m_z, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_z = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_x != null)
            {
                children.Add(m_x);
            }

            if (m_y != null)
            {
                children.Add(m_y);
            }

            if (m_z != null)
            {
                children.Add(m_z);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.X:
                {
                    if (createOrReplace)
                    {
                        if (X == null)
                        {
                            if (replacement == null)
                            {
                                X = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                X = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = X;
                    break;
                }

                case Opc.Ua.BrowseNames.Y:
                {
                    if (createOrReplace)
                    {
                        if (Y == null)
                        {
                            if (replacement == null)
                            {
                                Y = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                Y = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = Y;
                    break;
                }

                case Opc.Ua.BrowseNames.Z:
                {
                    if (createOrReplace)
                    {
                        if (Z == null)
                        {
                            if (replacement == null)
                            {
                                Z = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                Z = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = Z;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<double> m_x;
        private BaseDataVariableState<double> m_y;
        private BaseDataVariableState<double> m_z;
        #endregion
    }

    #region ThreeDVectorValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ThreeDVectorValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public ThreeDVectorValue(ThreeDVectorState variable, ThreeDVector value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new ThreeDVector();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public ThreeDVectorState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public ThreeDVector Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(ThreeDVectorState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.X;
                instance.OnReadValue = OnRead_X;
                instance.OnSimpleWriteValue = OnWrite_X;
                updateList.Add(instance);
                instance = m_variable.Y;
                instance.OnReadValue = OnRead_Y;
                instance.OnSimpleWriteValue = OnWrite_Y;
                updateList.Add(instance);
                instance = m_variable.Z;
                instance.OnReadValue = OnRead_Z;
                instance.OnSimpleWriteValue = OnWrite_Z;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (ThreeDVector)Write(value);
            }

            return ServiceResult.Good;
        }

        #region X Access Methods
        /// <remarks />
        private ServiceResult OnRead_X(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.X;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_X(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.X = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Y Access Methods
        /// <remarks />
        private ServiceResult OnRead_Y(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Y;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Y(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Y = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Z Access Methods
        /// <remarks />
        private ServiceResult OnRead_Z(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Z;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Z(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Z = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private ThreeDVector m_value;
        private ThreeDVectorState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region CartesianCoordinatesState Class
    #if (!OPCUA_EXCLUDE_CartesianCoordinatesState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class CartesianCoordinatesState : BaseDataVariableState<CartesianCoordinates>
    {
        #region Constructors
        /// <remarks />
        public CartesianCoordinatesState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.CartesianCoordinatesType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.CartesianCoordinates, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (LengthUnit != null)
            {
                LengthUnit.Initialize(context, LengthUnit_InitializationString);
            }
        }

        #region Initialization String
        private const string LengthUnit_InitializationString =
           "//////////8VYIkKAgAAAAAACgAAAExlbmd0aFVuaXQBAFVJAC4ARFVJAAABAHcD/////wEB/////wAA" +
           "AAA=";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAIAAAAENhcnRlc2lhbkNvb3JkaW5hdGVzVHlwZUluc3RhbmNlAQBUSQEA" +
           "VElUSQAAAQB5Sf////8BAf////8BAAAAFWCJCgIAAAAAAAoAAABMZW5ndGhVbml0AQBVSQAuAERVSQAA" +
           "AQB3A/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<EUInformation> LengthUnit
        {
            get
            {
                return m_lengthUnit;
            }

            set
            {
                if (!Object.ReferenceEquals(m_lengthUnit, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_lengthUnit = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_lengthUnit != null)
            {
                children.Add(m_lengthUnit);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.LengthUnit:
                {
                    if (createOrReplace)
                    {
                        if (LengthUnit == null)
                        {
                            if (replacement == null)
                            {
                                LengthUnit = new PropertyState<EUInformation>(this);
                            }
                            else
                            {
                                LengthUnit = (PropertyState<EUInformation>)replacement;
                            }
                        }
                    }

                    instance = LengthUnit;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<EUInformation> m_lengthUnit;
        #endregion
    }
    #endif
    #endregion

    #region ThreeDCartesianCoordinatesState Class
    #if (!OPCUA_EXCLUDE_ThreeDCartesianCoordinatesState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ThreeDCartesianCoordinatesState : CartesianCoordinatesState
    {
        #region Constructors
        /// <remarks />
        public ThreeDCartesianCoordinatesState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ThreeDCartesianCoordinatesType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.ThreeDCartesianCoordinates, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAJgAAAFRocmVlRENhcnRlc2lhbkNvb3JkaW5hdGVzVHlwZUluc3RhbmNl" +
           "AQBWSQEAVklWSQAAAQB6Sf////8BAf////8DAAAAFWCJCgIAAAAAAAEAAABYAQBYSQAvAD9YSQAAAAv/" +
           "////AQH/////AAAAABVgiQoCAAAAAAABAAAAWQEAWUkALwA/WUkAAAAL/////wEB/////wAAAAAVYIkK" +
           "AgAAAAAAAQAAAFoBAFpJAC8AP1pJAAAAC/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<double> X
        {
            get
            {
                return m_x;
            }

            set
            {
                if (!Object.ReferenceEquals(m_x, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_x = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> Y
        {
            get
            {
                return m_y;
            }

            set
            {
                if (!Object.ReferenceEquals(m_y, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_y = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> Z
        {
            get
            {
                return m_z;
            }

            set
            {
                if (!Object.ReferenceEquals(m_z, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_z = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_x != null)
            {
                children.Add(m_x);
            }

            if (m_y != null)
            {
                children.Add(m_y);
            }

            if (m_z != null)
            {
                children.Add(m_z);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.X:
                {
                    if (createOrReplace)
                    {
                        if (X == null)
                        {
                            if (replacement == null)
                            {
                                X = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                X = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = X;
                    break;
                }

                case Opc.Ua.BrowseNames.Y:
                {
                    if (createOrReplace)
                    {
                        if (Y == null)
                        {
                            if (replacement == null)
                            {
                                Y = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                Y = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = Y;
                    break;
                }

                case Opc.Ua.BrowseNames.Z:
                {
                    if (createOrReplace)
                    {
                        if (Z == null)
                        {
                            if (replacement == null)
                            {
                                Z = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                Z = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = Z;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<double> m_x;
        private BaseDataVariableState<double> m_y;
        private BaseDataVariableState<double> m_z;
        #endregion
    }

    #region ThreeDCartesianCoordinatesValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ThreeDCartesianCoordinatesValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public ThreeDCartesianCoordinatesValue(ThreeDCartesianCoordinatesState variable, ThreeDCartesianCoordinates value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new ThreeDCartesianCoordinates();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public ThreeDCartesianCoordinatesState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public ThreeDCartesianCoordinates Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(ThreeDCartesianCoordinatesState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.X;
                instance.OnReadValue = OnRead_X;
                instance.OnSimpleWriteValue = OnWrite_X;
                updateList.Add(instance);
                instance = m_variable.Y;
                instance.OnReadValue = OnRead_Y;
                instance.OnSimpleWriteValue = OnWrite_Y;
                updateList.Add(instance);
                instance = m_variable.Z;
                instance.OnReadValue = OnRead_Z;
                instance.OnSimpleWriteValue = OnWrite_Z;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (ThreeDCartesianCoordinates)Write(value);
            }

            return ServiceResult.Good;
        }

        #region X Access Methods
        /// <remarks />
        private ServiceResult OnRead_X(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.X;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_X(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.X = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Y Access Methods
        /// <remarks />
        private ServiceResult OnRead_Y(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Y;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Y(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Y = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region Z Access Methods
        /// <remarks />
        private ServiceResult OnRead_Z(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.Z;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_Z(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.Z = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private ThreeDCartesianCoordinates m_value;
        private ThreeDCartesianCoordinatesState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region OrientationState Class
    #if (!OPCUA_EXCLUDE_OrientationState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class OrientationState : BaseDataVariableState<Orientation>
    {
        #region Constructors
        /// <remarks />
        public OrientationState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.OrientationType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.Orientation, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (AngleUnit != null)
            {
                AngleUnit.Initialize(context, AngleUnit_InitializationString);
            }
        }

        #region Initialization String
        private const string AngleUnit_InitializationString =
           "//////////8VYIkKAgAAAAAACQAAAEFuZ2xlVW5pdAEAXEkALgBEXEkAAAEAdwP/////AQH/////AAAA" +
           "AA==";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAFwAAAE9yaWVudGF0aW9uVHlwZUluc3RhbmNlAQBbSQEAW0lbSQAAAQB7" +
           "Sf////8BAf////8BAAAAFWCJCgIAAAAAAAkAAABBbmdsZVVuaXQBAFxJAC4ARFxJAAABAHcD/////wEB" +
           "/////wAAAAA=";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public PropertyState<EUInformation> AngleUnit
        {
            get
            {
                return m_angleUnit;
            }

            set
            {
                if (!Object.ReferenceEquals(m_angleUnit, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_angleUnit = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_angleUnit != null)
            {
                children.Add(m_angleUnit);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.AngleUnit:
                {
                    if (createOrReplace)
                    {
                        if (AngleUnit == null)
                        {
                            if (replacement == null)
                            {
                                AngleUnit = new PropertyState<EUInformation>(this);
                            }
                            else
                            {
                                AngleUnit = (PropertyState<EUInformation>)replacement;
                            }
                        }
                    }

                    instance = AngleUnit;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private PropertyState<EUInformation> m_angleUnit;
        #endregion
    }
    #endif
    #endregion

    #region ThreeDOrientationState Class
    #if (!OPCUA_EXCLUDE_ThreeDOrientationState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class ThreeDOrientationState : OrientationState
    {
        #region Constructors
        /// <remarks />
        public ThreeDOrientationState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.ThreeDOrientationType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.ThreeDOrientation, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);
        }

        #region Initialization String
        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAHQAAAFRocmVlRE9yaWVudGF0aW9uVHlwZUluc3RhbmNlAQBdSQEAXUld" +
           "SQAAAQB8Sf////8BAf////8DAAAAFWCJCgIAAAAAAAEAAABBAQBfSQAvAD9fSQAAAAv/////AQH/////" +
           "AAAAABVgiQoCAAAAAAABAAAAQgEAYEkALwA/YEkAAAAL/////wEB/////wAAAAAVYIkKAgAAAAAAAQAA" +
           "AEMBAGFJAC8AP2FJAAAAC/////8BAf////8AAAAA";
        #endregion
        #endif
        #endregion

        #region Public Properties
        /// <remarks />
        public BaseDataVariableState<double> A
        {
            get
            {
                return m_a;
            }

            set
            {
                if (!Object.ReferenceEquals(m_a, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_a = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> B
        {
            get
            {
                return m_b;
            }

            set
            {
                if (!Object.ReferenceEquals(m_b, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_b = value;
            }
        }

        /// <remarks />
        public BaseDataVariableState<double> C
        {
            get
            {
                return m_c;
            }

            set
            {
                if (!Object.ReferenceEquals(m_c, value))
                {
                    ChangeMasks |= NodeStateChangeMasks.Children;
                }

                m_c = value;
            }
        }
        #endregion

        #region Overridden Methods
        /// <remarks />
        public override void GetChildren(
            ISystemContext context,
            IList<BaseInstanceState> children)
        {
            if (m_a != null)
            {
                children.Add(m_a);
            }

            if (m_b != null)
            {
                children.Add(m_b);
            }

            if (m_c != null)
            {
                children.Add(m_c);
            }

            base.GetChildren(context, children);
        }
            
        /// <remarks />
        protected override BaseInstanceState FindChild(
            ISystemContext context,
            QualifiedName browseName,
            bool createOrReplace,
            BaseInstanceState replacement)
        {
            if (QualifiedName.IsNull(browseName))
            {
                return null;
            }

            BaseInstanceState instance = null;

            switch (browseName.Name)
            {
                case Opc.Ua.BrowseNames.A:
                {
                    if (createOrReplace)
                    {
                        if (A == null)
                        {
                            if (replacement == null)
                            {
                                A = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                A = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = A;
                    break;
                }

                case Opc.Ua.BrowseNames.B:
                {
                    if (createOrReplace)
                    {
                        if (B == null)
                        {
                            if (replacement == null)
                            {
                                B = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                B = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = B;
                    break;
                }

                case Opc.Ua.BrowseNames.C:
                {
                    if (createOrReplace)
                    {
                        if (C == null)
                        {
                            if (replacement == null)
                            {
                                C = new BaseDataVariableState<double>(this);
                            }
                            else
                            {
                                C = (BaseDataVariableState<double>)replacement;
                            }
                        }
                    }

                    instance = C;
                    break;
                }
            }

            if (instance != null)
            {
                return instance;
            }

            return base.FindChild(context, browseName, createOrReplace, replacement);
        }
        #endregion

        #region Private Fields
        private BaseDataVariableState<double> m_a;
        private BaseDataVariableState<double> m_b;
        private BaseDataVariableState<double> m_c;
        #endregion
    }

    #region ThreeDOrientationValue Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public class ThreeDOrientationValue : BaseVariableValue
    {
        #region Constructors
        /// <remarks />
        public ThreeDOrientationValue(ThreeDOrientationState variable, ThreeDOrientation value, object dataLock) : base(dataLock)
        {
            m_value = value;

            if (m_value == null)
            {
                m_value = new ThreeDOrientation();
            }

            Initialize(variable);
        }
        #endregion

        #region Public Members
        /// <remarks />
        public ThreeDOrientationState Variable
        {
            get { return m_variable; }
        }

        /// <remarks />
        public ThreeDOrientation Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region Private Methods
        private void Initialize(ThreeDOrientationState variable)
        {
            lock (Lock)
            {
                m_variable = variable;

                variable.Value = m_value;

                variable.OnReadValue = OnReadValue;
                variable.OnSimpleWriteValue = OnWriteValue;

                BaseVariableState instance = null;
                List<BaseInstanceState> updateList = new List<BaseInstanceState>();
                updateList.Add(variable);

                instance = m_variable.A;
                instance.OnReadValue = OnRead_A;
                instance.OnSimpleWriteValue = OnWrite_A;
                updateList.Add(instance);
                instance = m_variable.B;
                instance.OnReadValue = OnRead_B;
                instance.OnSimpleWriteValue = OnWrite_B;
                updateList.Add(instance);
                instance = m_variable.C;
                instance.OnReadValue = OnRead_C;
                instance.OnSimpleWriteValue = OnWrite_C;
                updateList.Add(instance);

                SetUpdateList(updateList);
            }
        }

        /// <remarks />
        protected ServiceResult OnReadValue(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        private ServiceResult OnWriteValue(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value = (ThreeDOrientation)Write(value);
            }

            return ServiceResult.Good;
        }

        #region A Access Methods
        /// <remarks />
        private ServiceResult OnRead_A(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.A;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_A(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.A = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region B Access Methods
        /// <remarks />
        private ServiceResult OnRead_B(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.B;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_B(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.B = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion

        #region C Access Methods
        /// <remarks />
        private ServiceResult OnRead_C(
            ISystemContext context,
            NodeState node,
            NumericRange indexRange,
            QualifiedName dataEncoding,
            ref object value,
            ref StatusCode statusCode,
            ref DateTime timestamp)
        {
            lock (Lock)
            {
                DoBeforeReadProcessing(context, node);

                if (m_value != null)
                {
                    value = m_value.C;
                }

                return Read(context, node, indexRange, dataEncoding, ref value, ref statusCode, ref timestamp);
            }
        }

        /// <remarks />
        private ServiceResult OnWrite_C(ISystemContext context, NodeState node, ref object value)
        {
            lock (Lock)
            {
                m_value.C = (double)Write(value);
            }

            return ServiceResult.Good;
        }
        #endregion
        #endregion

        #region Private Fields
        private ThreeDOrientation m_value;
        private ThreeDOrientationState m_variable;
        #endregion
    }
    #endregion
    #endif
    #endregion

    #region FrameState Class
    #if (!OPCUA_EXCLUDE_FrameState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    public partial class FrameState : BaseDataVariableState<Frame>
    {
        #region Constructors
        /// <remarks />
        public FrameState(NodeState parent) : base(parent)
        {
        }

        /// <remarks />
        protected override NodeId GetDefaultTypeDefinitionId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.VariableTypes.FrameType, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override NodeId GetDefaultDataTypeId(NamespaceTable namespaceUris)
        {
            return Opc.Ua.NodeId.Create(Opc.Ua.DataTypes.Frame, Opc.Ua.Namespaces.OpcUa, namespaceUris);
        }

        /// <remarks />
        protected override int GetDefaultValueRank()
        {
            return ValueRanks.Scalar;
        }

        #if (!OPCUA_EXCLUDE_InitializationStrings)
        /// <remarks />
        protected override void Initialize(ISystemContext context)
        {
            base.Initialize(context);
            Initialize(context, InitializationString);
            InitializeOptionalChildren(context);
        }

        /// <remarks />
        protected override void Initialize(ISystemContext context, NodeState source)
        {
            InitializeOptionalChildren(context);
            base.Initialize(context, source);
        }

        /// <remarks />
        protected override void InitializeOptionalChildren(ISystemContext context)
        {
            base.InitializeOptionalChildren(context);

            if (Constant != null)
            {
                Constant.Initialize(context, Constant_InitializationString);
            }

            if (BaseFrame != null)
            {
                BaseFrame.Initialize(context, BaseFrame_InitializationString);
            }

            if (FixedBase != null)
            {
                FixedBase.Initialize(context, FixedBase_InitializationString);
            }
        }

        #region Initialization String
        private const string Constant_InitializationString =
           "//////////8VYIkKAgAAAAAACAAAAENvbnN0YW50AQBkSQAuAERkSQAAAAH/////AQH/////AAAAAA==";

        private const string BaseFrame_InitializationString =
           "//////////8VYIkKAgAAAAAACQAAAEJhc2VGcmFtZQEAZUkALwA/ZUkAAAAR/////wEB/////wAAAAA=";

        private const string FixedBase_InitializationString =
           "//////////8VYIkKAgAAAAAACQAAAEZpeGVkQmFzZQEAZkkALgBEZkkAAAAB/////wEB/////wAAAAA=";

        private const string InitializationString =
           "//////////8VYIkCAgAAAAAAEQAAAEZyYW1lVHlwZUluc3RhbmNlAQBiSQEAYkliSQAAAQB9Sf////8B" +
           "Af////8FAAAAFWCJCgIAAAAAABQAAABDYXJ0ZXNpYW5Db29yZGluYXRlcw